(ns atlas.ontology.yorba-cache
  "Yorba cache ontology module - cache configuration and datalog support.

   This module provides cache configuration and validation for the
   `:atlas/yorba-cache` ontology.

   A yorba-cache entity represents a cache configuration:
   - References a yorba-serialisation for data serialisation format
   - Can be used by endpoints and functions for caching responses

   Example:
     (registry/register!
      :cache/user-profile
      :atlas/yorba-cache
      #{:domain/users :tier/foundation}
      {:cache/serialisation :serialisation/user-json})

   **Dependency:** This module depends on `atlas.ontology.yorba-serialisation`.
   Loading this module will automatically load its dependencies.

   Usage:
     (require '[atlas.ontology.yorba-cache :as yc])
     (yc/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [atlas.ontology.yorba-serialisation :as yorba-ser]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-cache"
  {:ontology/for :atlas/yorba-cache
   :ontology/keys [:cache/serialisation]})  ; Reference to serialisation dev-id

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-caches
  "Return all registered yorba-cache dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-cache)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn serialisation-for
  "Get the serialisation dev-id for a cache.
   Uses the ontology key: :cache/serialisation"
  [cache-dev-id]
  (:cache/serialisation (entity/props-for cache-dev-id)))

(defn caches-by-serialisation
  "Get all caches that use a specific serialisation dev-id."
  [serialisation-dev-id]
  (->> (all-caches)
       (filter #(= serialisation-dev-id (serialisation-for %)))
       set))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-cache properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :cache/serialisation → :cache/serialisation (dev-id reference)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-cache)
    (let [dev-id (:atlas/dev-id props)
          ser-ref (:cache/serialisation props)]
      (cond-> []
        ;; Serialisation reference
        ser-ref
        (conj [:db/add dev-id :cache/serialisation ser-ref])))))

(def datalog-schema
  "Datascript schema for yorba-cache properties."
  {:cache/serialisation {:db/cardinality :db.cardinality/one}})

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

(defn find-caches-using-serialisation
  "Find all caches that use a specific serialisation (datalog query).

   Example: (find-caches-using-serialisation :serialisation/user-json)
   Returns: sequence of cache dev-ids"
  [serialisation-dev-id]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?cache-id
                :in $ ?ser-id
                :where
                [?e :atlas/dev-id ?cache-id]
                [?e :cache/serialisation ?ser-id]]
              db
              serialisation-dev-id)
         (map first)
         vec)))

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value existence checks (e.g., "must have serialisation") should be
;; handled by clojure.spec at registration time.
;;
;; Registry invariants focus on structural/cross-ontology validation.

(defn invariant-cache-has-valid-serialisation
  "Every yorba-cache that references a serialisation must point to an existing entity.

   **Structural Check:** Validates cross-ontology references.

   This ensures that caches don't point to non-existent serialisation
   configurations, which would cause runtime errors when the cache tries
   to serialise/deserialise data.

   Checks: :cache/serialisation property references a valid serialisation dev-id."
  []
  (let [caches (all-caches)
        valid-serialisations (yorba-ser/all-serialisations)
        violations (for [cache-id caches
                         :let [ser-ref (serialisation-for cache-id)]
                         :when (and ser-ref
                                    (not (valid-serialisations ser-ref)))]
                     {:cache cache-id
                      :missing-serialisation ser-ref})]
    (when (seq violations)
      {:invariant :cache-has-valid-serialisation
       :violation :missing-serialisation-reference
       :details violations
       :severity :error
       :message (str "Caches reference non-existent serialisations: "
                     (mapv :missing-serialisation violations))})))

(def invariants
  "All structural invariants specific to yorba-cache ontology.

   These focus on cross-registry validation, not value existence checks.
   Value checks (e.g., serialisation presence) should be handled by clojure.spec."
  [invariant-cache-has-valid-serialisation])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-cache ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-cache
   :atlas/ontology
   #{:atlas/yorba-cache}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-cache invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-cache properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-cache ontology, invariants, and datalog support.

   This will also load the yorba-serialisation module as a dependency.

   This must be called before using yorba-cache features:
   - Registering cache entities with :cache/serialisation
   - Using helper functions like serialisation-for, caches-by-serialisation
   - Running invariants

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    ;; Load dependencies first
    (yorba-ser/load!)
    ;; Then register this module
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-cache module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-cache module (primarily for testing).

   WARNING: This does not remove already-registered caches from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset.

   Note: This does NOT unload the yorba-serialisation dependency.
   Use yorba-ser/unload! separately if needed."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
