(ns atlas.ontology.yorba-serialisation
  "Yorba serialisation ontology module - invariants and datalog support.

   This module provides invariants and datalog integration for the
   `:atlas/yorba-serialisation` ontology.

   This module adds:
   - Ontology definition registration
   - Invariant validation
   - Datalog fact extraction for queries
   - Helper functions for querying serialisations

   Usage:
     (require '[atlas.ontology.yorba-serialisation :as ys])
     (ys/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-serialisation"
  {:ontology/for :atlas/yorba-serialisation
   :ontology/keys [:co.yorba.services.serialisation/url
                   :serial/args]})  ; Fields this serialisation can handle (both input and output)

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-serialisations
  "Return all registered yorba-serialisation dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-serialisation)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn url-for
  "Get the URL for a serialisation dev-id.
   Returns nil if not found or if the serialisation has no URL."
  [serialisation-dev-id]
  (:co.yorba.services.serialisation/url (entity/props-for serialisation-dev-id)))

(defn args-for
  "Get the args (fields) that a serialisation handles.
   Uses the ontology key: :serial/args"
  [serialisation-dev-id]
  (:serial/args (entity/props-for serialisation-dev-id)))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-serialisation properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :co.yorba.services.serialisation/url → :co.yorba.services.serialisation/url
   - :serial/args → :serialisation/handles (many, fields this serialisation can handle)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-serialisation)
    (let [dev-id (:atlas/dev-id props)
          url (:co.yorba.services.serialisation/url props)
          args (:serial/args props)]
      (cond-> []
        ;; URL attribute
        url
        (conj [:db/add dev-id :co.yorba.services.serialisation/url url])

        ;; Fields this serialisation handles (many-cardinality)
        args
        (concat (map (fn [arg-kw]
                       [:db/add dev-id :serialisation/handles arg-kw])
                     args))))))

(def datalog-schema
  "Datascript schema for yorba-serialisation properties."
  {:co.yorba.services.serialisation/url {:db/cardinality :db.cardinality/one}
   :serialisation/handles {:db/cardinality :db.cardinality/many}})

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

(defn find-serialisations-handling
  "Find all serialisations that handle a specific data keyword (datalog query).

   Example: (find-serialisations-handling :user/email)
   Returns: sequence of serialisation dev-ids"
  [data-kw]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?ser-id
                :in $ ?data-kw
                :where
                [?e :atlas/dev-id ?ser-id]
                [?e :serialisation/handles ?data-kw]]
              db
              data-kw)
         (map first)
         vec)))

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value existence checks (e.g., "must have URL") are handled by
;; clojure.spec at registration time, not by registry invariants.
;;
;; Registry invariants focus on structural/cross-ontology validation:
;; - Verifying entity references exist in the registry
;; - Checking relationships between entities
;; - Validating architectural constraints across ontologies

;; Currently no structural invariants for yorba-serialisation.
;; The serialisation entity is self-contained and doesn't reference other entities.

(def invariants
  "All invariants specific to yorba-serialisation ontology.

   Currently empty - serialisation entities don't have cross-registry dependencies.
   Value checks (e.g., URL presence) should be handled by clojure.spec."
  [])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-serialisation ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-serialisation
   :atlas/ontology
   #{:atlas/yorba-serialisation}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-serialisation invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-serialisation properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-serialisation ontology, invariants, and datalog support.

   This must be called before using yorba-serialisation features:
   - Registering entities with :co.yorba.services.serialisation/url
   - Using helper functions like url-for
   - Running invariants

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-serialisation module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-serialisation module (primarily for testing).

   WARNING: This does not remove already-registered serialisations from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
