(ns atlas.ontology.yorba-endpoint
  "Yorba endpoint ontology module - invariants and datalog support.

   This module provides invariants and datalog integration for the
   `:atlas/yorba-endpoint` ontology.

   This module adds:
   - Ontology definition registration
   - Invariant validation (including cross-ontology with yorba-serialisation)
   - Datalog fact extraction for queries
   - Helper functions and cross-ontology query helpers

   **Dependency:** This module depends on `atlas.ontology.yorba-serialisation`.
   Loading this module will automatically load its dependencies.

   Usage:
     (require '[atlas.ontology.yorba-endpoint :as ye])
     (ye/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [atlas.ontology.yorba-serialisation :as yorba-ser]
            [clojure.set :as set]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-endpoint"
  {:ontology/for :atlas/yorba-endpoint
   :ontology/keys [:atlas/yorba-serialisation
                   :yorba/handler
                   :co.yorba.endpoint/input   ; Collection of input keyword specs
                   :co.yorba.endpoint/output] ; Collection of output keyword specs
   ;; Dataflow integration - map to standard entity attributes
   :dataflow/context-key :co.yorba.endpoint/input
   :dataflow/response-key :co.yorba.endpoint/output})

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-endpoints
  "Return all registered yorba-endpoint dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-endpoint)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn all-serialisation-ids
  "Return all registered yorba-serialisation dev-ids (excluding ontology definitions)."
  []
  (yorba-ser/all-serialisations))

(defn serialisation-for
  "Get the serialisation dev-id for an endpoint.
   Uses the ontology key: :atlas/yorba-serialisation"
  [endpoint-dev-id]
  (:atlas/yorba-serialisation (entity/props-for endpoint-dev-id)))

(defn handler-for
  "Get the handler for an endpoint.
   Uses the ontology key: :yorba/handler"
  [endpoint-dev-id]
  (:yorba/handler (entity/props-for endpoint-dev-id)))

(defn input-for
  "Get the input keywords for an endpoint.
   Uses the ontology key: :co.yorba.endpoint/input"
  [endpoint-dev-id]
  (:co.yorba.endpoint/input (entity/props-for endpoint-dev-id)))

(defn output-for
  "Get the output keywords for an endpoint.
   Uses the ontology key: :co.yorba.endpoint/output"
  [endpoint-dev-id]
  (:co.yorba.endpoint/output (entity/props-for endpoint-dev-id)))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-endpoint properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :atlas/yorba-serialisation → :atlas/yorba-serialisation (dev-id reference)
   - :yorba/handler → :yorba/handler (handler function symbol)
   - :co.yorba.endpoint/input → :entity/consumes (many, for cross-ontology queries)
   - :co.yorba.endpoint/output → :entity/produces (many, for cross-ontology queries)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-endpoint)
    (let [dev-id (:atlas/dev-id props)
          ser-ref (:atlas/yorba-serialisation props)
          handler (:yorba/handler props)
          inputs (:co.yorba.endpoint/input props)
          outputs (:co.yorba.endpoint/output props)]
      (cond-> []
        ;; Serialisation reference
        ser-ref
        (conj [:db/add dev-id :atlas/yorba-serialisation ser-ref])

        ;; Handler function
        handler
        (conj [:db/add dev-id :yorba/handler (str handler)])

        ;; Input keywords → entity/consumes (many-cardinality)
        inputs
        (concat (map (fn [input-kw]
                       [:db/add dev-id :entity/consumes input-kw])
                     inputs))

        ;; Output keywords → entity/produces (many-cardinality)
        outputs
        (concat (map (fn [output-kw]
                       [:db/add dev-id :entity/produces output-kw])
                     outputs))))))

(def datalog-schema
  "Datascript schema for yorba-endpoint properties.

   Note: :entity/consumes and :entity/produces are shared with execution-function
   ontology for cross-ontology data flow queries."
  {:atlas/yorba-serialisation {:db/cardinality :db.cardinality/one}
   :yorba/handler {:db/cardinality :db.cardinality/one}
   :entity/consumes {:db/cardinality :db.cardinality/many}
   :entity/produces {:db/cardinality :db.cardinality/many}})

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value existence checks (e.g., "must have handler", "must have serialisation")
;; are handled by clojure.spec at registration time, not by registry invariants.
;;
;; Registry invariants focus on structural/cross-ontology validation:
;; - Verifying entity references exist in the registry
;; - Checking relationships between entities
;; - Validating architectural constraints across ontologies

(defn invariant-endpoint-has-valid-serialisation
  "Every yorba-endpoint that references a serialisation must point to an existing entity.

   **Structural Check:** Validates cross-ontology references.

   This ensures that endpoints don't point to non-existent serialisation
   configurations, which would cause runtime errors when the endpoint tries
   to use the serialisation.

   Checks: :atlas/yorba-serialisation property references a valid serialisation dev-id
   registered in the system."
  []
  (let [;; Filter out ontology definitions
        endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))
        valid-serialisations (all-serialisation-ids)
        violations (for [ep-id endpoints
                         :let [props (entity/props-for ep-id)
                               ser-ref (:atlas/yorba-serialisation props)]
                         :when (and ser-ref
                                    (not (valid-serialisations ser-ref)))]
                     {:endpoint ep-id
                      :missing-serialisation ser-ref})]
    (when (seq violations)
      {:invariant :endpoint-has-valid-serialisation
       :violation :missing-serialisation-reference
       :details violations
       :severity :error
       :message (str "Endpoints reference non-existent serialisations: "
                     (mapv :missing-serialisation violations))})))

(defn invariant-endpoint-inherits-serialisation-aspects
  "Endpoints must inherit all aspects from their referenced serialisation.

   **Structural Check:** Validates aspect inheritance across ontologies.

   An endpoint uses a serialisation's configuration, so it should share the
   serialisation's semantic characteristics (domain, compliance tags, etc.).
   This ensures semantic consistency across related entities.

   Exception: :atlas/yorba-serialisation aspect is not required on endpoints
   (it's the serialisation's entity type, not a semantic tag).

   Example:
     Serialisation: #{:atlas/yorba-serialisation :domain/users :compliance/pii}
     Endpoint must have: #{:domain/users :compliance/pii} (at minimum)

   Checks: endpoint compound identity contains all serialisation aspects
   except :atlas/yorba-serialisation itself."
  []
  (let [;; Filter out ontology definitions
        endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))
        violations (for [ep-id endpoints
                         :let [ep-identity (entity/identity-for ep-id)
                               ser-ref (serialisation-for ep-id)
                               ser-identity (when ser-ref (entity/identity-for ser-ref))
                               ;; Aspects that should be inherited (exclude entity type)
                               required-aspects (when ser-identity
                                                  (disj ser-identity :atlas/yorba-serialisation))
                               missing-aspects (when required-aspects
                                                 (set/difference required-aspects ep-identity))]
                         :when (and ser-identity (seq missing-aspects))]
                     {:endpoint ep-id
                      :serialisation ser-ref
                      :missing-aspects (vec missing-aspects)
                      :endpoint-identity ep-identity
                      :serialisation-identity ser-identity})]
    (when (seq violations)
      {:invariant :endpoint-inherits-serialisation-aspects
       :violation :missing-inherited-aspects
       :details violations
       :severity :error
       :message (str "Endpoints must inherit all aspects from their serialisation. "
                     "Violations: " (count violations))})))

(defn invariant-endpoint-authz-for-sensitive-data
  "Endpoints outputting user/system data MUST have :wrap/authz aspect.

   **Structural Check:** Validates security aspect based on data classification.

   If an endpoint's output contains a keyword that is registered as a yorba-data
   entity with type :yorba-data/user or :yorba-data/system, the endpoint MUST
   have the :wrap/authz aspect in its compound identity.

   This ensures that endpoints exposing sensitive data are properly protected
   with authorization checks.

   Requires: atlas.ontology.yorba-data to be loaded
   Checks: cross-ontology reference from endpoint output to yorba-data classification"
  []
  (let [;; Filter out ontology definitions
        endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))

        ;; Get all user and system data field dev-ids
        sensitive-data-ids (->> (entity/all-with-aspect :atlas/yorba-data)
                                (remove #(entity/has-aspect? % :atlas/ontology))
                                (filter (fn [data-id]
                                          (let [data-type (:yorba-data/type (entity/props-for data-id))]
                                            (or (= data-type :yorba-data/user)
                                                (= data-type :yorba-data/system)))))
                                set)

        violations (for [ep-id endpoints
                         :let [ep-identity (entity/identity-for ep-id)
                               ep-output (output-for ep-id)
                               ;; Check if any output field is sensitive data
                               sensitive-outputs (when ep-output
                                                   (filter sensitive-data-ids ep-output))]
                         :when (and (seq sensitive-outputs)
                                    (not (contains? ep-identity :wrap/authz)))]
                     {:endpoint ep-id
                      :sensitive-outputs (vec sensitive-outputs)
                      :missing-aspect :wrap/authz})]
    (when (seq violations)
      {:invariant :endpoint-authz-for-sensitive-data
       :violation :missing-authz-aspect
       :details violations
       :severity :error
       :message (str "Endpoints outputting user/system data must have :wrap/authz aspect. "
                     "Violations: " (count violations))})))

(def invariants
  "All structural invariants specific to yorba-endpoint ontology.

   These focus on cross-registry validation, not value existence checks.
   Value checks (e.g., handler presence) should be handled by clojure.spec."
  [invariant-endpoint-has-valid-serialisation
   invariant-endpoint-inherits-serialisation-aspects
   invariant-endpoint-authz-for-sensitive-data])

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

(defn find-endpoints-consuming
  "Find all endpoints that consume a specific data keyword (datalog query).

   Example: (find-endpoints-consuming :user/email)
   Returns: sequence of endpoint dev-ids"
  [data-kw]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?ep-id
                :in $ ?data-kw
                :where
                [?e :atlas/dev-id ?ep-id]
                [?e :entity/consumes ?data-kw]]
              db
              data-kw)
         (map first)
         vec)))

(defn find-endpoints-producing
  "Find all endpoints that produce a specific data keyword (datalog query).

   Example: (find-endpoints-producing :user/id)
   Returns: sequence of endpoint dev-ids"
  [data-kw]
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?ep-id
                :in $ ?data-kw
                :where
                [?e :atlas/dev-id ?ep-id]
                [?e :entity/produces ?data-kw]]
              db
              data-kw)
         (map first)
         vec)))

(defn find-data-flow-pairs
  "Find producer-consumer pairs where one endpoint's output matches another's input.

   Returns: sequence of maps with {:producer, :consumer, :data-keyword}"
  []
  (let [db (datalog/create-db)]
    (->> (d/q '[:find ?producer-id ?consumer-id ?data-kw
                :where
                [?producer :entity/produces ?data-kw]
                [?consumer :entity/consumes ?data-kw]
                [?producer :atlas/dev-id ?producer-id]
                [?consumer :atlas/dev-id ?consumer-id]
                ;; Avoid self-references
                [(not= ?producer ?consumer)]]
              db)
         (map (fn [[producer consumer data-kw]]
                {:producer producer
                 :consumer consumer
                 :data-keyword data-kw}))
         vec)))

(defn find-aspect-inheritance-violations-datalog
  "Find endpoints that don't inherit all serialisation aspects (using datalog).

   This is an alternative implementation of invariant-endpoint-inherits-serialisation-aspects
   using datalog queries. Useful for complex cross-entity queries.

   Returns: sequence of maps with {:endpoint, :serialisation, :missing-aspects}"
  []
  (let [db (datalog/create-db)
        ;; Query all endpoints with their serialisation references
        endpoint-ser-pairs (d/q '[:find ?ep-id ?ser-id
                                  :where
                                  [?ep :atlas/dev-id ?ep-id]
                                  [?ep :atlas/yorba-serialisation ?ser-id]]
                                db)
        violations (for [[ep-id ser-id] endpoint-ser-pairs
                         :let [ep-identity (entity/identity-for ep-id)
                               ser-identity (entity/identity-for ser-id)
                               required-aspects (disj ser-identity :atlas/yorba-serialisation)
                               missing-aspects (set/difference required-aspects ep-identity)]
                         :when (seq missing-aspects)]
                     {:endpoint ep-id
                      :serialisation ser-id
                      :missing-aspects (vec missing-aspects)})]
    violations))

;; =============================================================================
;; CROSS-ONTOLOGY QUERIES
;; =============================================================================

(defn endpoints-using-serialisation
  "Find all endpoints that use a specific serialisation."
  [serialisation-dev-id]
  (let [endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))]
    (filter (fn [ep-id]
              (= serialisation-dev-id (serialisation-for ep-id)))
            endpoints)))

(defn endpoint-with-serialisation-details
  "Get endpoint details including resolved serialisation info."
  [endpoint-dev-id]
  (let [ep-props (entity/props-for endpoint-dev-id)
        ser-ref (:atlas/yorba-serialisation ep-props)
        ser-props (when ser-ref (entity/props-for ser-ref))]
    {:endpoint endpoint-dev-id
     :endpoint-props ep-props
     :serialisation ser-ref
     :serialisation-props ser-props}))

(defn orphan-serialisations
  "Find serialisations that are not used by any endpoint.

   Useful for cleanup and identifying dead configuration."
  []
  (let [all-sers (->> (all-serialisation-ids)
                      (remove #(entity/has-aspect? % :atlas/ontology)))
        used-sers (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology))
                       (map serialisation-for)
                       (remove nil?)
                       set)]
    (set/difference (set all-sers) used-sers)))

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-endpoint ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-endpoint
   :atlas/ontology
   #{:atlas/yorba-endpoint}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-endpoint invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-endpoint properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-endpoint ontology, invariants, and datalog support.

   This will also load the yorba-serialisation module as a dependency.

   This must be called before using yorba-endpoint features:
   - Registering entities with :atlas/yorba-serialisation, :yorba/handler
   - Using helper functions like serialisation-for, handler-for
   - Running invariants

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    ;; Load dependencies first
    (yorba-ser/load!)
    ;; Then register this module
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-endpoint module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-endpoint module (primarily for testing).

   WARNING: This does not remove already-registered endpoints from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset.

   Note: This does NOT unload the yorba-serialisation dependency.
   Use yorba-ser/unload! separately if needed."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
