(ns atlas.ontology.yorba-cache-index
  "todo add docs"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [datascript.core :as d]))

;; =============================================================================
;; ONTOLOGY DEFINITION
;; =============================================================================

(def ontology-definition
  "The ontology definition for :atlas/yorba-data"
  {:ontology/for :atlas/yorba-cache-index
   :ontology/keys [:yorba-data/serialisation
                   :serial/args]}) 

;; =============================================================================
;; HELPER FUNCTIONS
;; =============================================================================

(defn all-data-fields
  "Return all registered yorba-data dev-ids (excluding ontology definitions)."
  []
  (->> (entity/all-with-aspect :atlas/yorba-cache-index)
       (remove #(entity/has-aspect? % :atlas/ontology))
       set))

(defn serialisation-for
  ""
  [data-dev-id]
  (:yorba-data/serialisation (entity/props-for data-dev-id)))

(defn args-for
  ""
  [serialisation-dev-id]
  (:serial/args (entity/props-for serialisation-dev-id)))


(defn data-fields-by-serialisation
  ""
  [data-serialisation-dev-id]
  (->> (all-data-fields)
       (filter #(= data-serialisation-dev-id (serialisation-for %)))
       set))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-data properties.
   Called by atlas.datalog when building database.

   Property mappings:
   - :yorba-data/type → :yorba-data/type (classification)"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-cache)
    (let [dev-id (:atlas/dev-id props)
          ser-ref (:atlas/yorba-serialisation props)
          args (:serial/args props)]
      (cond-> []
        args
        (concat (map (fn [arg-kw]
                       [:db/add dev-id :serialisation/handles arg-kw])
                     args))         
        ser-ref
        (conj [:db/add dev-id :atlas/yorba-serialisation ser-ref])))))

(def datalog-schema
  "Datascript schema for yorba-data properties."
  {:atlas/yorba-serialisation {:db/cardinality :db.cardinality/one}
   :serialisation/handles {:db/cardinality :db.cardinality/many}})

;; =============================================================================
;; DATALOG-BASED QUERIES
;; =============================================================================

;; =============================================================================
;; INVARIANTS
;; =============================================================================

;; NOTE: Value validation (type must be one of allowed values) should be
;; handled by clojure.spec at registration time.
;;
;; Registry invariants focus on structural/cross-ontology validation.

(def invariants
  "All structural invariants specific to yorba-data ontology.

   Currently empty - cross-ontology invariants with endpoints are defined
   in the yorba-endpoint module."
  [])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-ontology!
  "Register the yorba-data ontology in the registry."
  []
  (registry/register!
   :atlas/yorba-cache-index
   :atlas/ontology
   #{:atlas/yorba-cache-index}
   ontology-definition))

(defn- register-invariants!
  "Register yorba-data invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-data properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-data ontology, invariants, and datalog support.

   This must be called before using yorba-data features:
   - Registering data field entities with :yorba-data/type
   - Using helper functions like type-for, data-fields-by-type
   - Running cross-ontology invariants with endpoints

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    (register-ontology!)
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-data module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-data module (primarily for testing).

   WARNING: This does not remove already-registered data fields from the
   registry or clean up datalog extensions. Use reset! on registry/registry
   and datalog/reset-extensions! for a full reset."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing).

   Use this before calling load! in test fixtures when you've also reset
   the registry. This ensures load! will re-register the ontology."
  []
  (reset! loaded? false))
