(ns atlas.ontology.yorba-serialisation
  "Yorba serialisation ontology module - invariants and datalog support.

   This module provides invariants and datalog integration for the
   `:atlas/yorba-serialisation` ontology.

   The ontology itself is defined in your client project. This module adds:
   - Invariant validation
   - Datalog fact extraction for queries

   Usage:
     (require '[atlas.ontology.yorba-serialisation :as ys])
     (ys/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-serialisation properties.
   Called by atlas.datalog when building database.

   Uses the ontology key: :co.yorba.services.serialisation/url"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-serialisation)
    (let [dev-id (:atlas/dev-id props)
          url (:co.yorba.services.serialisation/url props)]
      (cond-> []
        url
        (conj [:db/add dev-id :serialisation/url url])))))

(def datalog-schema
  "Datascript schema for yorba-serialisation properties."
  {:serialisation/url {:db/cardinality :db.cardinality/one}})

;; =============================================================================
;; INVARIANTS
;; =============================================================================

(defn invariant-serialisation-has-url
  "Every yorba-serialisation must have a URL defined.

   A serialisation without a URL is incomplete and cannot be used
   for actual serialisation operations.

   Checks for: :co.yorba.services.serialisation/url"
  []
  (let [;; Filter out ontology definitions (they have :atlas/ontology aspect)
        serialisations (->> (entity/all-with-aspect :atlas/yorba-serialisation)
                            (remove #(entity/has-aspect? % :atlas/ontology)))
        violations (for [ser-id serialisations
                         :let [props (entity/props-for ser-id)
                               url (:co.yorba.services.serialisation/url props)]
                         :when (nil? url)]
                     {:serialisation ser-id})]
    (when (seq violations)
      {:invariant :serialisation-has-url
       :violation :missing-url
       :details violations
       :severity :error
       :message (str "Serialisations must have :co.yorba.services.serialisation/url: "
                     (mapv :serialisation violations))})))

(defn invariant-serialisation-url-unique
  "Serialisation URLs should be unique across the registry.

   Two serialisations pointing to the same URL is likely a configuration
   error or duplication that should be resolved."
  []
  (let [;; Filter out ontology definitions
        serialisations (->> (entity/all-with-aspect :atlas/yorba-serialisation)
                            (remove #(entity/has-aspect? % :atlas/ontology)))
        url-groups (->> serialisations
                        (map (fn [ser-id]
                               {:id ser-id
                                :url (:co.yorba.services.serialisation/url (entity/props-for ser-id))}))
                        (filter :url)
                        (group-by :url))
        duplicates (filter (fn [[_ sers]] (> (count sers) 1)) url-groups)]
    (when (seq duplicates)
      {:invariant :serialisation-url-unique
       :violation :duplicate-urls
       :details (into {} (map (fn [[url sers]]
                                [url (mapv :id sers)])
                              duplicates))
       :severity :warning
       :message (str "Duplicate serialisation URLs found: "
                     (keys (into {} duplicates)))})))

(def invariants
  "All invariants specific to yorba-serialisation ontology"
  [invariant-serialisation-has-url
   invariant-serialisation-url-unique])

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-invariants!
  "Register yorba-serialisation invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-serialisation properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-serialisation invariants and datalog support.

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-serialisation module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-serialisation module (primarily for testing)."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing)."
  []
  (reset! loaded? false))
