(ns atlas.ontology.yorba-endpoint
  "Yorba endpoint ontology module - invariants and datalog support.

   This module provides invariants and datalog integration for the
   `:atlas/yorba-endpoint` ontology.

   The ontology itself is defined in your client project. This module adds:
   - Invariant validation (including cross-ontology with yorba-serialisation)
   - Datalog fact extraction for queries
   - Cross-ontology query helpers

   **Dependency:** This module depends on `atlas.ontology.yorba-serialisation`.
   Loading this module will automatically load its dependencies.

   Usage:
     (require '[atlas.ontology.yorba-endpoint :as ye])
     (ye/load!)"
  (:require [atlas.registry :as registry]
            [atlas.registry.lookup :as entity]
            [atlas.invariant :as invariant]
            [atlas.datalog :as datalog]
            [atlas.ontology.yorba-serialisation :as yorba-ser]
            [clojure.set :as set]))

;; =============================================================================
;; HELPERS
;; =============================================================================

(defn all-serialisation-ids
  "Return all registered yorba-serialisation dev-ids."
  []
  (set (entity/all-with-aspect :atlas/yorba-serialisation)))

(defn serialisation-for
  "Get the serialisation dev-id for an endpoint.
   Uses the ontology key: :atlas/yorba-serialisation"
  [endpoint-dev-id]
  (:atlas/yorba-serialisation (entity/props-for endpoint-dev-id)))

(defn handler-for
  "Get the handler for an endpoint.
   Uses the ontology key: :yorba/handler"
  [endpoint-dev-id]
  (:yorba/handler (entity/props-for endpoint-dev-id)))

;; =============================================================================
;; DATALOG INTEGRATION
;; =============================================================================

(defn extract-facts
  "Extract Datascript facts from yorba-endpoint properties.
   Called by atlas.datalog when building database.

   Uses ontology keys:
   - :atlas/yorba-serialisation (dev-id reference)
   - :yorba/handler"
  [compound-id props]
  (when (contains? compound-id :atlas/yorba-endpoint)
    (let [dev-id (:atlas/dev-id props)
          ser-ref (:atlas/yorba-serialisation props)
          handler (:yorba/handler props)]
      (cond-> []
        ser-ref
        (conj [:db/add dev-id :endpoint/uses-serialisation ser-ref])

        handler
        (conj [:db/add dev-id :endpoint/handler (str handler)])))))

(def datalog-schema
  "Datascript schema for yorba-endpoint properties."
  {:endpoint/uses-serialisation {:db/cardinality :db.cardinality/one}
   :endpoint/handler {:db/cardinality :db.cardinality/one}})

;; =============================================================================
;; INVARIANTS
;; =============================================================================

(defn invariant-endpoint-has-valid-serialisation
  "Every yorba-endpoint must reference an existing yorba-serialisation.

   This ensures that endpoints don't point to non-existent serialisation
   configurations, which would cause runtime errors.

   Checks: :atlas/yorba-serialisation property references a valid serialisation dev-id"
  []
  (let [;; Filter out ontology definitions
        endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))
        valid-serialisations (all-serialisation-ids)
        violations (for [ep-id endpoints
                         :let [props (entity/props-for ep-id)
                               ser-ref (:atlas/yorba-serialisation props)]
                         :when (and ser-ref
                                    (not (valid-serialisations ser-ref)))]
                     {:endpoint ep-id
                      :missing-serialisation ser-ref})]
    (when (seq violations)
      {:invariant :endpoint-has-valid-serialisation
       :violation :missing-serialisation-reference
       :details violations
       :severity :error
       :message (str "Endpoints reference non-existent serialisations: "
                     (mapv :missing-serialisation violations))})))

(defn invariant-endpoint-has-serialisation
  "Every yorba-endpoint must have a serialisation reference.

   Endpoints without serialisation configuration are incomplete.

   Checks for: :atlas/yorba-serialisation property"
  []
  (let [;; Filter out ontology definitions
        endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))
        violations (for [ep-id endpoints
                         :let [props (entity/props-for ep-id)
                               ser-ref (:atlas/yorba-serialisation props)]
                         :when (nil? ser-ref)]
                     {:endpoint ep-id})]
    (when (seq violations)
      {:invariant :endpoint-has-serialisation
       :violation :missing-serialisation
       :details violations
       :severity :error
       :message (str "Endpoints must have :atlas/yorba-serialisation reference: "
                     (mapv :endpoint violations))})))

(defn invariant-endpoint-has-handler
  "Every yorba-endpoint must have a handler defined.

   An endpoint without a handler cannot process requests.

   Checks for: :yorba/handler property"
  []
  (let [;; Filter out ontology definitions
        endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))
        violations (for [ep-id endpoints
                         :let [props (entity/props-for ep-id)
                               handler (:yorba/handler props)]
                         :when (nil? handler)]
                     {:endpoint ep-id})]
    (when (seq violations)
      {:invariant :endpoint-has-handler
       :violation :missing-handler
       :details violations
       :severity :error
       :message (str "Endpoints must have :yorba/handler: "
                     (mapv :endpoint violations))})))

(def invariants
  "All invariants specific to yorba-endpoint ontology"
  [invariant-endpoint-has-serialisation
   invariant-endpoint-has-valid-serialisation
   invariant-endpoint-has-handler])

;; =============================================================================
;; CROSS-ONTOLOGY QUERIES
;; =============================================================================

(defn endpoints-using-serialisation
  "Find all endpoints that use a specific serialisation."
  [serialisation-dev-id]
  (let [endpoints (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology)))]
    (filter (fn [ep-id]
              (= serialisation-dev-id (serialisation-for ep-id)))
            endpoints)))

(defn endpoint-with-serialisation-details
  "Get endpoint details including resolved serialisation info."
  [endpoint-dev-id]
  (let [ep-props (entity/props-for endpoint-dev-id)
        ser-ref (:atlas/yorba-serialisation ep-props)
        ser-props (when ser-ref (entity/props-for ser-ref))]
    {:endpoint endpoint-dev-id
     :endpoint-props ep-props
     :serialisation ser-ref
     :serialisation-props ser-props}))

(defn orphan-serialisations
  "Find serialisations that are not used by any endpoint.

   Useful for cleanup and identifying dead configuration."
  []
  (let [all-sers (->> (all-serialisation-ids)
                      (remove #(entity/has-aspect? % :atlas/ontology)))
        used-sers (->> (entity/all-with-aspect :atlas/yorba-endpoint)
                       (remove #(entity/has-aspect? % :atlas/ontology))
                       (map serialisation-for)
                       (remove nil?)
                       set)]
    (set/difference (set all-sers) used-sers)))

;; =============================================================================
;; LOADING
;; =============================================================================

(defn- register-invariants!
  "Register yorba-endpoint invariants with the invariant module."
  []
  (doseq [inv invariants]
    (invariant/register-ontology-invariant! inv)))

(defn- register-datalog!
  "Register datalog extensions for yorba-endpoint properties."
  []
  (datalog/register-fact-extractor! extract-facts)
  (datalog/register-schema! datalog-schema))

(defonce ^:private loaded? (atom false))

(defn load!
  "Load yorba-endpoint invariants and datalog support.

   This will also load the yorba-serialisation module as a dependency.

   Safe to call multiple times - subsequent calls are no-ops."
  []
  (when-not @loaded?
    ;; Load dependencies first
    (yorba-ser/load!)
    ;; Then register this module
    (register-invariants!)
    (register-datalog!)
    (reset! loaded? true))
  :loaded)

(defn loaded?*
  "Check if the yorba-endpoint module has been loaded."
  []
  @loaded?)

(defn unload!
  "Unload the yorba-endpoint module (primarily for testing).

   Note: This does NOT unload the yorba-serialisation dependency.
   Use yorba-ser/unload! separately if needed."
  []
  (when @loaded?
    (doseq [inv invariants]
      (invariant/unregister-ontology-invariant! inv))
    (reset! loaded? false))
  :unloaded)

(defn reset-loaded-state!
  "Reset the loaded state to false (for testing)."
  []
  (reset! loaded? false))
