goog.provide('atlas.query');
/**
 * Find all entries containing aspect(s).
 * Returns map of identities→values, sorted by identity size.
 * 
 * UNIFIES:
 * - atlas.registry/find-with
 * - atlas.semantic-queries/find-by
 */
atlas.query.find_by_aspect = (function atlas$query$find_by_aspect(registry,aspect){
var aspect_set = ((cljs.core.set_QMARK_(aspect))?aspect:cljs.core.PersistentHashSet.createAsIfByAssoc([aspect]));
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,cljs.core.sort_by.cljs$core$IFn$_invoke$arity$2(cljs.core.comp.cljs$core$IFn$_invoke$arity$2(cljs.core.count,cljs.core.key),cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8493){
var vec__8494 = p__8493;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8494,(0),null);
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8494,(1),null);
return clojure.set.superset_QMARK_(id,aspect_set);
}),registry)));
});
/**
 * Find the [identity value] pair for a given dev-id.
 * Returns nil if not found.
 * 
 * UNIFIES:
 * - atlas.registry/find-by-dev-id
 * - atlas.registry.graph/fetch-by-dev-id
 */
atlas.query.find_by_dev_id = (function atlas$query$find_by_dev_id(registry,dev_id){
if(cljs.core.truth_(dev_id)){
return cljs.core.first(cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8497){
var vec__8498 = p__8497;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8498,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8498,(1),null);
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(new cljs.core.Keyword("atlas","dev-id","atlas/dev-id",2002261553).cljs$core$IFn$_invoke$arity$1(v),dev_id);
}),registry));
} else {
return null;
}
});
/**
 * Find all dev-ids with given aspect(s). Returns set of dev-ids.
 * 
 * Derived from find-by-aspect for convenience.
 * Replaces atlas.registry.graph/all-with-aspect.
 */
atlas.query.find_dev_ids_with_aspect = (function atlas$query$find_dev_ids_with_aspect(registry,aspect){
return cljs.core.set(cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p__8504){
var vec__8505 = p__8504;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8505,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8505,(1),null);
return new cljs.core.Keyword("atlas","dev-id","atlas/dev-id",2002261553).cljs$core$IFn$_invoke$arity$1(v);
}),atlas.query.find_by_aspect(registry,aspect)));
});
/**
 * Find exact identity match. Returns value or nil.
 * 
 * From atlas.registry/find-exact
 */
atlas.query.find_exact = (function atlas$query$find_exact(registry,identity_set){
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(registry,identity_set);
});
/**
 * Filter registry entries by predicate (pred identity value).
 * Returns filtered map.
 * 
 * From atlas.registry/where
 */
atlas.query.where = (function atlas$query$where(registry,pred){
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8514){
var vec__8515 = p__8514;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8515,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8515,(1),null);
return (pred.cljs$core$IFn$_invoke$arity$2 ? pred.cljs$core$IFn$_invoke$arity$2(id,v) : pred.call(null,id,v));
}),registry));
});
/**
 * Return all compound identities in registry as vector.
 * 
 * From atlas.registry/all-identities
 */
atlas.query.all_identities = (function atlas$query$all_identities(registry){
return cljs.core.vec(cljs.core.keys(registry));
});
/**
 * Return how many of `selected-aspects` are present in `identity`.
 */
atlas.query.match_count = (function atlas$query$match_count(identity,selected_aspects){
var identity_set = ((cljs.core.set_QMARK_(identity))?identity:cljs.core.set(identity));
var selected_set = ((cljs.core.set_QMARK_(selected_aspects))?selected_aspects:cljs.core.set(selected_aspects));
return cljs.core.count(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,selected_set));
});
/**
 * Return match score (0.0 to 1.0) for `identity` against `selected-aspects`.
 * 
 *   query-mode:
 *   - :and   => 1.0 iff identity contains ALL selected aspects, else 0.0
 *   - :or    => 1.0 iff identity contains ANY selected aspect, else 0.0
 *   - :count => (match-count / total-selected)
 * 
 *   Returns 0.0 when `selected-aspects` is empty.
 */
atlas.query.match_score = (function atlas$query$match_score(identity,selected_aspects,query_mode){
var identity_set = ((cljs.core.set_QMARK_(identity))?identity:cljs.core.set(identity));
var selected_set = ((cljs.core.set_QMARK_(selected_aspects))?selected_aspects:cljs.core.set(selected_aspects));
var total_selected = cljs.core.count(selected_set);
if((total_selected === (0))){
return 0.0;
} else {
var matched_count = cljs.core.count(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,selected_set));
var G__8536 = query_mode;
var G__8536__$1 = (((G__8536 instanceof cljs.core.Keyword))?G__8536.fqn:null);
switch (G__8536__$1) {
case "and":
if(cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(matched_count,total_selected)){
return 1.0;
} else {
return 0.0;
}

break;
case "or":
if((matched_count > (0))){
return 1.0;
} else {
return 0.0;
}

break;
case "count":
return (matched_count / total_selected);

break;
default:
return 0.0;

}
}
});
/**
 * Return true if `identity` matches `selected-aspects` under `query-mode`.
 * 
 *   query-mode:
 *   - :and => ALL selected aspects must be present
 *   - :or  => ANY selected aspect must be present
 *   - :count => ANY selected aspect must be present
 * 
 *   Returns false when `selected-aspects` is empty.
 */
atlas.query.matches_QMARK_ = (function atlas$query$matches_QMARK_(identity,selected_aspects,query_mode){
var identity_set = ((cljs.core.set_QMARK_(identity))?identity:cljs.core.set(identity));
var selected_set = ((cljs.core.set_QMARK_(selected_aspects))?selected_aspects:cljs.core.set(selected_aspects));
var total_selected = cljs.core.count(selected_set);
if((total_selected > (0))){
var matched_count = cljs.core.count(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,selected_set));
var G__8548 = query_mode;
var G__8548__$1 = (((G__8548 instanceof cljs.core.Keyword))?G__8548.fqn:null);
switch (G__8548__$1) {
case "and":
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(matched_count,total_selected);

break;
case "or":
return (matched_count > (0));

break;
case "count":
return (matched_count > (0));

break;
default:
return false;

}
} else {
return false;
}
});
atlas.query.__GT_aspect_set = (function atlas$query$__GT_aspect_set(aspects){
if((aspects == null)){
return cljs.core.PersistentHashSet.EMPTY;
} else {
if(cljs.core.set_QMARK_(aspects)){
return aspects;
} else {
return cljs.core.set(aspects);

}
}
});
/**
 * Get a query value from either a namespaced key (preferred) or a legacy
 *   unqualified key.
 */
atlas.query.qget = (function atlas$query$qget(query,namespaced_key,legacy_key){
if(cljs.core.contains_QMARK_(query,namespaced_key)){
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(query,namespaced_key);
} else {
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(query,legacy_key);
}
});
/**
 * Return true if identity contains any of the `negated-aspects`.
 */
atlas.query.excluded_by_negation_QMARK_ = (function atlas$query$excluded_by_negation_QMARK_(identity,negated_aspects){
var identity_set = ((cljs.core.set_QMARK_(identity))?identity:cljs.core.set(identity));
var negated_set = atlas.query.__GT_aspect_set(negated_aspects);
return cljs.core.boolean$(cljs.core.seq(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,negated_set)));
});
/**
 * Return match score (0.0-1.0) for `identity` against a query map.
 * 
 *   Query map:
 *   - ::selected  collection of aspects (may be empty)
 *   - ::negated   collection of aspects (may be empty)
 *   - ::mode      :and | :or | :count (defaults to :count)
 * 
 *   If `identity` contains any `:negated` aspect, the score is 0.0.
 */
atlas.query.query_score = (function atlas$query$query_score(identity,query){
var identity_set = ((cljs.core.set_QMARK_(identity))?identity:cljs.core.set(identity));
var selected_set = atlas.query.__GT_aspect_set(atlas.query.qget(query,new cljs.core.Keyword("atlas.query","selected","atlas.query/selected",1069414447),new cljs.core.Keyword(null,"selected","selected",574897764)));
var negated_set = atlas.query.__GT_aspect_set(atlas.query.qget(query,new cljs.core.Keyword("atlas.query","negated","atlas.query/negated",1430445364),new cljs.core.Keyword(null,"negated","negated",-273117033)));
var mode = (function (){var or__5002__auto__ = atlas.query.qget(query,new cljs.core.Keyword("atlas.query","mode","atlas.query/mode",1284108834),new cljs.core.Keyword(null,"mode","mode",654403691));
if(cljs.core.truth_(or__5002__auto__)){
return or__5002__auto__;
} else {
return new cljs.core.Keyword(null,"count","count",2139924085);
}
})();
if(cljs.core.seq(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,negated_set))){
return 0.0;
} else {
return atlas.query.match_score(identity_set,selected_set,mode);
}
});
/**
 * Return true if `identity` matches a query map.
 * 
 *   Query map:
 *   - ::selected  collection of aspects (may be empty)
 *   - ::negated   collection of aspects (may be empty)
 *   - ::mode      :and | :or | :count (defaults to :count)
 *   - ::min-score minimum score threshold (defaults to 0.0)
 * 
 *   Notes:
 *   - If `:selected` is empty, the query matches as long as negations don't exclude.
 *   - `:min-score` is only applied when `:selected` is non-empty.
 */
atlas.query.query_matches_QMARK_ = (function atlas$query$query_matches_QMARK_(identity,query){
var identity_set = ((cljs.core.set_QMARK_(identity))?identity:cljs.core.set(identity));
var selected_set = atlas.query.__GT_aspect_set(atlas.query.qget(query,new cljs.core.Keyword("atlas.query","selected","atlas.query/selected",1069414447),new cljs.core.Keyword(null,"selected","selected",574897764)));
var negated_set = atlas.query.__GT_aspect_set(atlas.query.qget(query,new cljs.core.Keyword("atlas.query","negated","atlas.query/negated",1430445364),new cljs.core.Keyword(null,"negated","negated",-273117033)));
var mode = (function (){var or__5002__auto__ = atlas.query.qget(query,new cljs.core.Keyword("atlas.query","mode","atlas.query/mode",1284108834),new cljs.core.Keyword(null,"mode","mode",654403691));
if(cljs.core.truth_(or__5002__auto__)){
return or__5002__auto__;
} else {
return new cljs.core.Keyword(null,"count","count",2139924085);
}
})();
var min_score = (function (){var or__5002__auto__ = atlas.query.qget(query,new cljs.core.Keyword("atlas.query","min-score","atlas.query/min-score",-135983776),new cljs.core.Keyword(null,"min-score","min-score",-762388731));
if(cljs.core.truth_(or__5002__auto__)){
return or__5002__auto__;
} else {
return 0.0;
}
})();
var excluded_QMARK_ = cljs.core.seq(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,negated_set));
var selected_empty_QMARK_ = cljs.core.empty_QMARK_(selected_set);
var matched_count = cljs.core.count(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(identity_set,selected_set));
var positive_match_QMARK_ = (function (){var G__8568 = mode;
var G__8568__$1 = (((G__8568 instanceof cljs.core.Keyword))?G__8568.fqn:null);
switch (G__8568__$1) {
case "and":
return clojure.set.superset_QMARK_(identity_set,selected_set);

break;
case "or":
return ((selected_empty_QMARK_) || ((matched_count > (0))));

break;
case "count":
return ((selected_empty_QMARK_) || ((matched_count > (0))));

break;
default:
return ((selected_empty_QMARK_) || ((matched_count > (0))));

}
})();
var and__5000__auto__ = cljs.core.not(excluded_QMARK_);
if(and__5000__auto__){
var and__5000__auto____$1 = positive_match_QMARK_;
if(cljs.core.truth_(and__5000__auto____$1)){
return ((selected_empty_QMARK_) || ((atlas.query.query_score(identity_set,new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"selected","selected",574897764),selected_set,new cljs.core.Keyword(null,"negated","negated",-273117033),negated_set,new cljs.core.Keyword(null,"mode","mode",654403691),mode], null)) >= min_score)));
} else {
return and__5000__auto____$1;
}
} else {
return and__5000__auto__;
}
});
/**
 * Find identities similar to target identity using Jaccard similarity.
 * Returns sorted list of {:identity :shared :similarity} maps.
 * min-similarity is threshold (0.0-1.0), defaults to 0.0.
 * 
 * UNIFIES:
 * - atlas.registry/semantic-neighbors
 * - atlas.semantic-queries/semantic-similarity
 */
atlas.query.semantic_similarity = (function atlas$query$semantic_similarity(var_args){
var G__8582 = arguments.length;
switch (G__8582) {
case 2:
return atlas.query.semantic_similarity.cljs$core$IFn$_invoke$arity$2((arguments[(0)]),(arguments[(1)]));

break;
case 3:
return atlas.query.semantic_similarity.cljs$core$IFn$_invoke$arity$3((arguments[(0)]),(arguments[(1)]),(arguments[(2)]));

break;
default:
throw (new Error(["Invalid arity: ",cljs.core.str.cljs$core$IFn$_invoke$arity$1(arguments.length)].join('')));

}
});

(atlas.query.semantic_similarity.cljs$core$IFn$_invoke$arity$2 = (function (registry,target_identity){
return atlas.query.semantic_similarity.cljs$core$IFn$_invoke$arity$3(registry,target_identity,0.0);
}));

(atlas.query.semantic_similarity.cljs$core$IFn$_invoke$arity$3 = (function (registry,target_identity,min_similarity){
return cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3(new cljs.core.Keyword(null,"similarity","similarity",1871163855),cljs.core._GT_,cljs.core.keep.cljs$core$IFn$_invoke$arity$2((function (other){
if(cljs.core.not_EQ_.cljs$core$IFn$_invoke$arity$2(target_identity,other)){
var shared = clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(target_identity,other);
var union = clojure.set.union.cljs$core$IFn$_invoke$arity$2(target_identity,other);
var sim = ((cljs.core.seq(union))?(cljs.core.count(shared) / cljs.core.count(union)):(0));
if((sim >= min_similarity)){
return new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"identity","identity",1647396035),other,new cljs.core.Keyword(null,"shared","shared",-384145993),shared,new cljs.core.Keyword(null,"similarity","similarity",1871163855),sim], null);
} else {
return null;
}
} else {
return null;
}
}),cljs.core.keys(registry)));
}));

(atlas.query.semantic_similarity.cljs$lang$maxFixedArity = 3);

/**
 * Return frequency map of co-occurring aspects for given aspect.
 * Sorted by frequency descending.
 * 
 * From atlas.registry/related-to
 */
atlas.query.related_aspects = (function atlas$query$related_aspects(registry,aspect){
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3(cljs.core.val,cljs.core._GT_,cljs.core.frequencies(cljs.core.remove.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentHashSet.createAsIfByAssoc([aspect]),cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(cljs.core.identity,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8589_SHARP_){
return cljs.core.contains_QMARK_(p1__8589_SHARP_,aspect);
}),atlas.query.all_identities(registry))], 0))))));
});
/**
 * Return frequency map of aspects across all identities, sorted descending.
 * 
 * From atlas.registry/aspect-frequency
 */
atlas.query.aspect_frequency = (function atlas$query$aspect_frequency(registry){
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3(cljs.core.val,cljs.core._GT_,cljs.core.frequencies(cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(cljs.core.identity,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.keys(registry)], 0)))));
});
/**
 * Return list of {:identity :size :value} sorted by descending size.
 * 
 * From atlas.registry/identity-stats
 */
atlas.query.identity_stats = (function atlas$query$identity_stats(registry){
return cljs.core.sort_by.cljs$core$IFn$_invoke$arity$3(new cljs.core.Keyword(null,"size","size",1098693007),cljs.core._GT_,cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (id){
return new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"identity","identity",1647396035),id,new cljs.core.Keyword(null,"size","size",1098693007),cljs.core.count(id),new cljs.core.Keyword(null,"value","value",305978217),cljs.core.get.cljs$core$IFn$_invoke$arity$2(registry,id)], null);
}),atlas.query.all_identities(registry)));
});
/**
 * Find entities producing a data key.
 * property-key is the key to check (e.g., :context/response or :semantic-namespace/response).
 * Returns map of identities→values.
 * 
 * UNIFIES:
 * - atlas.semantic-queries/produces
 * - atlas.registry.graph/trace-data-flow (producer side)
 */
atlas.query.find_producers = (function atlas$query$find_producers(registry,data_key,property_key){
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8596){
var vec__8597 = p__8596;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8597,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8597,(1),null);
return cljs.core.some(cljs.core.PersistentHashSet.createAsIfByAssoc([data_key]),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,property_key));
}),registry));
});
/**
 * Find entities consuming a data key.
 * property-key is the key to check (e.g., :context/input or :semantic-namespace/context).
 * Returns map of identities→values.
 * 
 * UNIFIES:
 * - atlas.semantic-queries/consumes
 * - atlas.registry.graph/trace-data-flow (consumer side)
 */
atlas.query.find_consumers = (function atlas$query$find_consumers(registry,data_key,property_key){
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8601){
var vec__8602 = p__8601;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8602,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8602,(1),null);
return cljs.core.some(cljs.core.PersistentHashSet.createAsIfByAssoc([data_key]),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,property_key));
}),registry));
});
/**
 * Trace data flow for a key.
 * producer-key and consumer-key specify which properties to check.
 * Returns {:data-key :producers :consumers :connected?}.
 * 
 * UNIFIES:
 * - atlas.semantic-queries/trace-data-flow
 * - atlas.registry.graph/trace-data-flow
 */
atlas.query.trace_data_flow = (function atlas$query$trace_data_flow(registry,data_key,producer_key,consumer_key){
var producers = atlas.query.find_producers(registry,data_key,producer_key);
var consumers = atlas.query.find_consumers(registry,data_key,consumer_key);
return new cljs.core.PersistentArrayMap(null, 4, [new cljs.core.Keyword(null,"data-key","data-key",1775480631),data_key,new cljs.core.Keyword(null,"producers","producers",-658364712),producers,new cljs.core.Keyword(null,"consumers","consumers",270616753),consumers,new cljs.core.Keyword(null,"connected?","connected?",-1197551387),((cljs.core.seq(producers)) && (cljs.core.seq(consumers)))], null);
});
/**
 * Perform set-algebraic queries over compound identities.
 * Supported ops:
 *   {:intersection [aspect-sets]}  - Find identities containing ALL aspect sets
 *   {:union [aspect-sets]}         - Find identities containing ANY aspect set
 *   {:difference [include exclude]} - Find identities with include but not exclude
 * 
 * Returns vector of sorted identities.
 * 
 * From atlas.registry/query-algebra
 */
atlas.query.query_algebra = (function atlas$query$query_algebra(registry,ops){
var all_ids = cljs.core.set(atlas.query.all_identities(registry));
var results = (cljs.core.truth_(new cljs.core.Keyword(null,"intersection","intersection",2003891010).cljs$core$IFn$_invoke$arity$1(ops))?cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8611_SHARP_){
return cljs.core.every_QMARK_((function (x){
return clojure.set.superset_QMARK_(p1__8611_SHARP_,x);
}),new cljs.core.Keyword(null,"intersection","intersection",2003891010).cljs$core$IFn$_invoke$arity$1(ops));
}),all_ids):(cljs.core.truth_(new cljs.core.Keyword(null,"union","union",2142937499).cljs$core$IFn$_invoke$arity$1(ops))?cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8612_SHARP_){
return cljs.core.some((function (x){
return clojure.set.superset_QMARK_(p1__8612_SHARP_,x);
}),new cljs.core.Keyword(null,"union","union",2142937499).cljs$core$IFn$_invoke$arity$1(ops));
}),all_ids):(cljs.core.truth_(new cljs.core.Keyword(null,"difference","difference",1916101396).cljs$core$IFn$_invoke$arity$1(ops))?(function (){var vec__8617 = new cljs.core.Keyword(null,"difference","difference",1916101396).cljs$core$IFn$_invoke$arity$1(ops);
var include = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8617,(0),null);
var exclude = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8617,(1),null);
return cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8613_SHARP_){
return ((clojure.set.superset_QMARK_(p1__8613_SHARP_,include)) && (cljs.core.empty_QMARK_(clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(p1__8613_SHARP_,exclude))));
}),all_ids);
})():(function(){throw cljs.core.ex_info.cljs$core$IFn$_invoke$arity$2("Unknown operation",new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null,"ops","ops",1237330063),ops], null))})()
)));
return cljs.core.vec(cljs.core.sort.cljs$core$IFn$_invoke$arity$1(cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p1__8614_SHARP_){
return cljs.core.vec(cljs.core.sort.cljs$core$IFn$_invoke$arity$1(p1__8614_SHARP_));
}),results)));
});
/**
 * Return all compound identities that are supersets of find*.
 * Returns vector of sorted identities.
 * 
 * From atlas.registry/query
 */
atlas.query.query_superset = (function atlas$query$query_superset(registry,find_STAR_){
var find_STAR___$1 = cljs.core.set(find_STAR_);
return cljs.core.vec(cljs.core.sort.cljs$core$IFn$_invoke$arity$1(cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p1__8637_SHARP_){
return cljs.core.vec(cljs.core.sort.cljs$core$IFn$_invoke$arity$1(p1__8637_SHARP_));
}),cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8636_SHARP_){
return clojure.set.superset_QMARK_(p1__8636_SHARP_,find_STAR___$1);
}),cljs.core.keys(registry)))));
});
/**
 * Build dependency graph from entities with :deps key.
 * Returns seq of {:id :deps :identity} maps.
 * 
 * From atlas.semantic-queries/dependency-graph
 */
atlas.query.dependency_graph = (function atlas$query$dependency_graph(registry,id_key,deps_key){
return cljs.core.remove.cljs$core$IFn$_invoke$arity$2((function (p1__8641_SHARP_){
return cljs.core.empty_QMARK_(new cljs.core.Keyword(null,"deps","deps",1883360319).cljs$core$IFn$_invoke$arity$1(p1__8641_SHARP_));
}),cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p__8644){
var vec__8645 = p__8644;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8645,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8645,(1),null);
return new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"id","id",-1388402092),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"deps","deps",1883360319),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,deps_key),new cljs.core.Keyword(null,"identity","identity",1647396035),id], null);
}),cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8650){
var vec__8651 = p__8650;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8651,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8651,(1),null);
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,deps_key);
}),registry)));
});
/**
 * Group entities by architectural tier.
 * Returns map of tier→[dev-ids].
 * 
 * From atlas.semantic-queries/by-tier
 */
atlas.query.by_tier = (function atlas$query$by_tier(registry,id_key){
var tiers = new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Keyword("tier","foundation","tier/foundation",-1943771629),new cljs.core.Keyword("tier","service","tier/service",-1966897213),new cljs.core.Keyword("tier","api","tier/api",-897042558)], null);
return cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,(function (){var iter__5480__auto__ = (function atlas$query$by_tier_$_iter__8664(s__8665){
return (new cljs.core.LazySeq(null,(function (){
var s__8665__$1 = s__8665;
while(true){
var temp__5804__auto__ = cljs.core.seq(s__8665__$1);
if(temp__5804__auto__){
var s__8665__$2 = temp__5804__auto__;
if(cljs.core.chunked_seq_QMARK_(s__8665__$2)){
var c__5478__auto__ = cljs.core.chunk_first(s__8665__$2);
var size__5479__auto__ = cljs.core.count(c__5478__auto__);
var b__8667 = cljs.core.chunk_buffer(size__5479__auto__);
if((function (){var i__8666 = (0);
while(true){
if((i__8666 < size__5479__auto__)){
var tier = cljs.core._nth(c__5478__auto__,i__8666);
cljs.core.chunk_append(b__8667,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [tier,cljs.core.vec(cljs.core.remove.cljs$core$IFn$_invoke$arity$2(cljs.core.nil_QMARK_,cljs.core.map.cljs$core$IFn$_invoke$arity$2(((function (i__8666,tier,c__5478__auto__,size__5479__auto__,b__8667,s__8665__$2,temp__5804__auto__,tiers){
return (function (p1__8655_SHARP_){
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(p1__8655_SHARP_,id_key);
});})(i__8666,tier,c__5478__auto__,size__5479__auto__,b__8667,s__8665__$2,temp__5804__auto__,tiers))
,cljs.core.vals(atlas.query.find_by_aspect(registry,tier)))))], null));

var G__8991 = (i__8666 + (1));
i__8666 = G__8991;
continue;
} else {
return true;
}
break;
}
})()){
return cljs.core.chunk_cons(cljs.core.chunk(b__8667),atlas$query$by_tier_$_iter__8664(cljs.core.chunk_rest(s__8665__$2)));
} else {
return cljs.core.chunk_cons(cljs.core.chunk(b__8667),null);
}
} else {
var tier = cljs.core.first(s__8665__$2);
return cljs.core.cons(new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [tier,cljs.core.vec(cljs.core.remove.cljs$core$IFn$_invoke$arity$2(cljs.core.nil_QMARK_,cljs.core.map.cljs$core$IFn$_invoke$arity$2(((function (tier,s__8665__$2,temp__5804__auto__,tiers){
return (function (p1__8655_SHARP_){
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(p1__8655_SHARP_,id_key);
});})(tier,s__8665__$2,temp__5804__auto__,tiers))
,cljs.core.vals(atlas.query.find_by_aspect(registry,tier)))))], null),atlas$query$by_tier_$_iter__8664(cljs.core.rest(s__8665__$2)));
}
} else {
return null;
}
break;
}
}),null,null));
});
return iter__5480__auto__(tiers);
})());
});
/**
 * Analyze inter-domain dependencies.
 * Returns seq of {:domain :depends-on :entity-count} maps.
 * 
 * From atlas.semantic-queries/domain-coupling
 */
atlas.query.domain_coupling = (function atlas$query$domain_coupling(registry,id_key,deps_key){
var domains = cljs.core.set(cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8676_SHARP_){
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2("domain",cljs.core.namespace(p1__8676_SHARP_));
}),cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(cljs.core.identity,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.keys(registry)], 0))));
var iter__5480__auto__ = (function atlas$query$domain_coupling_$_iter__8687(s__8688){
return (new cljs.core.LazySeq(null,(function (){
var s__8688__$1 = s__8688;
while(true){
var temp__5804__auto__ = cljs.core.seq(s__8688__$1);
if(temp__5804__auto__){
var s__8688__$2 = temp__5804__auto__;
if(cljs.core.chunked_seq_QMARK_(s__8688__$2)){
var c__5478__auto__ = cljs.core.chunk_first(s__8688__$2);
var size__5479__auto__ = cljs.core.count(c__5478__auto__);
var b__8690 = cljs.core.chunk_buffer(size__5479__auto__);
if((function (){var i__8689 = (0);
while(true){
if((i__8689 < size__5479__auto__)){
var domain = cljs.core._nth(c__5478__auto__,i__8689);
var domain_entities = atlas.query.find_by_aspect(registry,domain);
var deps = cljs.core.set(cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(((function (i__8689,domain_entities,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains){
return (function (p1__8677_SHARP_){
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(p1__8677_SHARP_,deps_key);
});})(i__8689,domain_entities,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains))
,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.vals(domain_entities)], 0)));
var dep_domains = cljs.core.set(cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(((function (i__8689,domain_entities,deps,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains){
return (function (p__8699){
var vec__8700 = p__8699;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8700,(0),null);
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8700,(1),null);
return cljs.core.filter.cljs$core$IFn$_invoke$arity$2(((function (i__8689,vec__8700,id,_,domain_entities,deps,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains){
return (function (p1__8679_SHARP_){
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2("domain",cljs.core.namespace(p1__8679_SHARP_));
});})(i__8689,vec__8700,id,_,domain_entities,deps,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains))
,id);
});})(i__8689,domain_entities,deps,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains))
,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.filter.cljs$core$IFn$_invoke$arity$2(((function (i__8689,domain_entities,deps,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains){
return (function (p__8707){
var vec__8711 = p__8707;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8711,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8711,(1),null);
return cljs.core.some(deps,cljs.core.PersistentHashSet.createAsIfByAssoc([cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key)]));
});})(i__8689,domain_entities,deps,domain,c__5478__auto__,size__5479__auto__,b__8690,s__8688__$2,temp__5804__auto__,domains))
,registry)], 0)));
cljs.core.chunk_append(b__8690,new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"domain","domain",1847214937),domain,new cljs.core.Keyword(null,"depends-on","depends-on",-1448442022),cljs.core.disj.cljs$core$IFn$_invoke$arity$2(dep_domains,domain),new cljs.core.Keyword(null,"entity-count","entity-count",870517368),cljs.core.count(domain_entities)], null));

var G__8998 = (i__8689 + (1));
i__8689 = G__8998;
continue;
} else {
return true;
}
break;
}
})()){
return cljs.core.chunk_cons(cljs.core.chunk(b__8690),atlas$query$domain_coupling_$_iter__8687(cljs.core.chunk_rest(s__8688__$2)));
} else {
return cljs.core.chunk_cons(cljs.core.chunk(b__8690),null);
}
} else {
var domain = cljs.core.first(s__8688__$2);
var domain_entities = atlas.query.find_by_aspect(registry,domain);
var deps = cljs.core.set(cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(((function (domain_entities,domain,s__8688__$2,temp__5804__auto__,domains){
return (function (p1__8677_SHARP_){
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(p1__8677_SHARP_,deps_key);
});})(domain_entities,domain,s__8688__$2,temp__5804__auto__,domains))
,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.vals(domain_entities)], 0)));
var dep_domains = cljs.core.set(cljs.core.mapcat.cljs$core$IFn$_invoke$arity$variadic(((function (domain_entities,deps,domain,s__8688__$2,temp__5804__auto__,domains){
return (function (p__8726){
var vec__8727 = p__8726;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8727,(0),null);
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8727,(1),null);
return cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8679_SHARP_){
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2("domain",cljs.core.namespace(p1__8679_SHARP_));
}),id);
});})(domain_entities,deps,domain,s__8688__$2,temp__5804__auto__,domains))
,cljs.core.prim_seq.cljs$core$IFn$_invoke$arity$2([cljs.core.filter.cljs$core$IFn$_invoke$arity$2(((function (domain_entities,deps,domain,s__8688__$2,temp__5804__auto__,domains){
return (function (p__8731){
var vec__8732 = p__8731;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8732,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8732,(1),null);
return cljs.core.some(deps,cljs.core.PersistentHashSet.createAsIfByAssoc([cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key)]));
});})(domain_entities,deps,domain,s__8688__$2,temp__5804__auto__,domains))
,registry)], 0)));
return cljs.core.cons(new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"domain","domain",1847214937),domain,new cljs.core.Keyword(null,"depends-on","depends-on",-1448442022),cljs.core.disj.cljs$core$IFn$_invoke$arity$2(dep_domains,domain),new cljs.core.Keyword(null,"entity-count","entity-count",870517368),cljs.core.count(domain_entities)], null),atlas$query$domain_coupling_$_iter__8687(cljs.core.rest(s__8688__$2)));
}
} else {
return null;
}
break;
}
}),null,null));
});
return iter__5480__auto__(domains);
});
/**
 * Analyze what would be affected if entity changes.
 * Returns {:entity :entity/produces :direct-dependents}.
 * 
 * From atlas.semantic-queries/impact-of-change
 */
atlas.query.impact_of_change = (function atlas$query$impact_of_change(registry,entity_id,id_key,deps_key,response_key){
var entity_entry = cljs.core.first(cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8741){
var vec__8742 = p__8741;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8742,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8742,(1),null);
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2(entity_id,cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key));
}),registry));
var entity_response = cljs.core.get.cljs$core$IFn$_invoke$arity$2(cljs.core.second(entity_entry),response_key);
var direct_deps = cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p__8750){
var vec__8751 = p__8750;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8751,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8751,(1),null);
return cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key);
}),cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p__8756){
var vec__8757 = p__8756;
var _ = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8757,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8757,(1),null);
return cljs.core.some(cljs.core.PersistentHashSet.createAsIfByAssoc([entity_id]),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,deps_key));
}),registry));
return new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"entity","entity",-450970276),entity_id,new cljs.core.Keyword("entity","produces","entity/produces",1636269614),entity_response,new cljs.core.Keyword(null,"direct-dependents","direct-dependents",960910571),cljs.core.vec(direct_deps)], null);
});
/**
 * Find all entities handling PII with audit status.
 * Returns seq of {:id :audited? :context :response} maps.
 * 
 * From atlas.semantic-queries/pii-surface
 */
atlas.query.pii_surface = (function atlas$query$pii_surface(registry,id_key,context_key,response_key){
return cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p__8771){
var vec__8772 = p__8771;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8772,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8772,(1),null);
return new cljs.core.PersistentArrayMap(null, 4, [new cljs.core.Keyword(null,"id","id",-1388402092),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"audited?","audited?",678011184),cljs.core.contains_QMARK_(id,new cljs.core.Keyword("compliance","audited","compliance/audited",-598614598)),new cljs.core.Keyword(null,"context","context",-830191113),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,context_key),new cljs.core.Keyword(null,"response","response",-1068424192),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,response_key)], null);
}),atlas.query.find_by_aspect(registry,new cljs.core.Keyword("compliance","pii","compliance/pii",-811893690)));
});
/**
 * Show which entities have specific cross-cutting concerns.
 * Returns seq of {:id :has} where :has is map of aspect→boolean.
 * 
 * From atlas.semantic-queries/aspect-coverage
 */
atlas.query.aspect_coverage = (function atlas$query$aspect_coverage(registry,entity_type,id_key,aspects){
var iter__5480__auto__ = (function atlas$query$aspect_coverage_$_iter__8818(s__8819){
return (new cljs.core.LazySeq(null,(function (){
var s__8819__$1 = s__8819;
while(true){
var temp__5804__auto__ = cljs.core.seq(s__8819__$1);
if(temp__5804__auto__){
var s__8819__$2 = temp__5804__auto__;
if(cljs.core.chunked_seq_QMARK_(s__8819__$2)){
var c__5478__auto__ = cljs.core.chunk_first(s__8819__$2);
var size__5479__auto__ = cljs.core.count(c__5478__auto__);
var b__8821 = cljs.core.chunk_buffer(size__5479__auto__);
if((function (){var i__8820 = (0);
while(true){
if((i__8820 < size__5479__auto__)){
var vec__8825 = cljs.core._nth(c__5478__auto__,i__8820);
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8825,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8825,(1),null);
cljs.core.chunk_append(b__8821,new cljs.core.PersistentArrayMap(null, 2, [new cljs.core.Keyword(null,"id","id",-1388402092),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"has","has",1466247447),cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,(function (){var iter__5480__auto__ = ((function (i__8820,vec__8825,id,v,c__5478__auto__,size__5479__auto__,b__8821,s__8819__$2,temp__5804__auto__){
return (function atlas$query$aspect_coverage_$_iter__8818_$_iter__8855(s__8856){
return (new cljs.core.LazySeq(null,((function (i__8820,vec__8825,id,v,c__5478__auto__,size__5479__auto__,b__8821,s__8819__$2,temp__5804__auto__){
return (function (){
var s__8856__$1 = s__8856;
while(true){
var temp__5804__auto____$1 = cljs.core.seq(s__8856__$1);
if(temp__5804__auto____$1){
var s__8856__$2 = temp__5804__auto____$1;
if(cljs.core.chunked_seq_QMARK_(s__8856__$2)){
var c__5478__auto____$1 = cljs.core.chunk_first(s__8856__$2);
var size__5479__auto____$1 = cljs.core.count(c__5478__auto____$1);
var b__8858 = cljs.core.chunk_buffer(size__5479__auto____$1);
if((function (){var i__8857 = (0);
while(true){
if((i__8857 < size__5479__auto____$1)){
var a = cljs.core._nth(c__5478__auto____$1,i__8857);
cljs.core.chunk_append(b__8858,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [a,cljs.core.contains_QMARK_(id,a)], null));

var G__9012 = (i__8857 + (1));
i__8857 = G__9012;
continue;
} else {
return true;
}
break;
}
})()){
return cljs.core.chunk_cons(cljs.core.chunk(b__8858),atlas$query$aspect_coverage_$_iter__8818_$_iter__8855(cljs.core.chunk_rest(s__8856__$2)));
} else {
return cljs.core.chunk_cons(cljs.core.chunk(b__8858),null);
}
} else {
var a = cljs.core.first(s__8856__$2);
return cljs.core.cons(new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [a,cljs.core.contains_QMARK_(id,a)], null),atlas$query$aspect_coverage_$_iter__8818_$_iter__8855(cljs.core.rest(s__8856__$2)));
}
} else {
return null;
}
break;
}
});})(i__8820,vec__8825,id,v,c__5478__auto__,size__5479__auto__,b__8821,s__8819__$2,temp__5804__auto__))
,null,null));
});})(i__8820,vec__8825,id,v,c__5478__auto__,size__5479__auto__,b__8821,s__8819__$2,temp__5804__auto__))
;
return iter__5480__auto__(aspects);
})())], null));

var G__9013 = (i__8820 + (1));
i__8820 = G__9013;
continue;
} else {
return true;
}
break;
}
})()){
return cljs.core.chunk_cons(cljs.core.chunk(b__8821),atlas$query$aspect_coverage_$_iter__8818(cljs.core.chunk_rest(s__8819__$2)));
} else {
return cljs.core.chunk_cons(cljs.core.chunk(b__8821),null);
}
} else {
var vec__8884 = cljs.core.first(s__8819__$2);
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8884,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8884,(1),null);
return cljs.core.cons(new cljs.core.PersistentArrayMap(null, 2, [new cljs.core.Keyword(null,"id","id",-1388402092),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"has","has",1466247447),cljs.core.into.cljs$core$IFn$_invoke$arity$2(cljs.core.PersistentArrayMap.EMPTY,(function (){var iter__5480__auto__ = ((function (vec__8884,id,v,s__8819__$2,temp__5804__auto__){
return (function atlas$query$aspect_coverage_$_iter__8818_$_iter__8889(s__8890){
return (new cljs.core.LazySeq(null,(function (){
var s__8890__$1 = s__8890;
while(true){
var temp__5804__auto____$1 = cljs.core.seq(s__8890__$1);
if(temp__5804__auto____$1){
var s__8890__$2 = temp__5804__auto____$1;
if(cljs.core.chunked_seq_QMARK_(s__8890__$2)){
var c__5478__auto__ = cljs.core.chunk_first(s__8890__$2);
var size__5479__auto__ = cljs.core.count(c__5478__auto__);
var b__8892 = cljs.core.chunk_buffer(size__5479__auto__);
if((function (){var i__8891 = (0);
while(true){
if((i__8891 < size__5479__auto__)){
var a = cljs.core._nth(c__5478__auto__,i__8891);
cljs.core.chunk_append(b__8892,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [a,cljs.core.contains_QMARK_(id,a)], null));

var G__9014 = (i__8891 + (1));
i__8891 = G__9014;
continue;
} else {
return true;
}
break;
}
})()){
return cljs.core.chunk_cons(cljs.core.chunk(b__8892),atlas$query$aspect_coverage_$_iter__8818_$_iter__8889(cljs.core.chunk_rest(s__8890__$2)));
} else {
return cljs.core.chunk_cons(cljs.core.chunk(b__8892),null);
}
} else {
var a = cljs.core.first(s__8890__$2);
return cljs.core.cons(new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [a,cljs.core.contains_QMARK_(id,a)], null),atlas$query$aspect_coverage_$_iter__8818_$_iter__8889(cljs.core.rest(s__8890__$2)));
}
} else {
return null;
}
break;
}
}),null,null));
});})(vec__8884,id,v,s__8819__$2,temp__5804__auto__))
;
return iter__5480__auto__(aspects);
})())], null),atlas$query$aspect_coverage_$_iter__8818(cljs.core.rest(s__8819__$2)));
}
} else {
return null;
}
break;
}
}),null,null));
});
return iter__5480__auto__(atlas.query.find_by_aspect(registry,entity_type));
});
/**
 * Check if error handlers exist for marked concerns.
 * Returns {:handlers :coverage}.
 * 
 * From atlas.semantic-queries/error-handler-coverage
 */
atlas.query.error_handler_coverage = (function atlas$query$error_handler_coverage(registry,id_key){
var handlers = atlas.query.find_by_aspect(registry,new cljs.core.Keyword("semantic-namespace","error-handler","semantic-namespace/error-handler",848160348));
var needs_handling = new cljs.core.PersistentHashSet(null, new cljs.core.PersistentArrayMap(null, 4, [new cljs.core.Keyword("authorization","required","authorization/required",-1235852971),null,new cljs.core.Keyword("temporal","timeout","temporal/timeout",-1212243268),null,new cljs.core.Keyword("capacity","rate-limited","capacity/rate-limited",1009987166),null,new cljs.core.Keyword("protocol","http","protocol/http",1404791679),null], null), null);
return new cljs.core.PersistentArrayMap(null, 2, [new cljs.core.Keyword(null,"handlers","handlers",79528781),cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p__8909){
var vec__8910 = p__8909;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8910,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8910,(1),null);
return new cljs.core.PersistentArrayMap(null, 2, [new cljs.core.Keyword(null,"id","id",-1388402092),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"handles","handles",-1061347879),clojure.set.intersection.cljs$core$IFn$_invoke$arity$2(id,needs_handling)], null);
}),handlers),new cljs.core.Keyword(null,"coverage","coverage",1606292651),(function (){var iter__5480__auto__ = (function atlas$query$error_handler_coverage_$_iter__8914(s__8915){
return (new cljs.core.LazySeq(null,(function (){
var s__8915__$1 = s__8915;
while(true){
var temp__5804__auto__ = cljs.core.seq(s__8915__$1);
if(temp__5804__auto__){
var xs__6360__auto__ = temp__5804__auto__;
var concern = cljs.core.first(xs__6360__auto__);
var iterys__5476__auto__ = ((function (s__8915__$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling){
return (function atlas$query$error_handler_coverage_$_iter__8914_$_iter__8916(s__8917){
return (new cljs.core.LazySeq(null,((function (s__8915__$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling){
return (function (){
var s__8917__$1 = s__8917;
while(true){
var temp__5804__auto____$1 = cljs.core.seq(s__8917__$1);
if(temp__5804__auto____$1){
var s__8917__$2 = temp__5804__auto____$1;
if(cljs.core.chunked_seq_QMARK_(s__8917__$2)){
var c__5478__auto__ = cljs.core.chunk_first(s__8917__$2);
var size__5479__auto__ = cljs.core.count(c__5478__auto__);
var b__8919 = cljs.core.chunk_buffer(size__5479__auto__);
if((function (){var i__8918 = (0);
while(true){
if((i__8918 < size__5479__auto__)){
var vec__8927 = cljs.core._nth(c__5478__auto__,i__8918);
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8927,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8927,(1),null);
cljs.core.chunk_append(b__8919,new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"entity","entity",-450970276),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"concern","concern",-531166280),concern,new cljs.core.Keyword(null,"has-handler?","has-handler?",1969404999),cljs.core.boolean$(cljs.core.some(((function (i__8918,s__8915__$1,vec__8927,id,v,c__5478__auto__,size__5479__auto__,b__8919,s__8917__$2,temp__5804__auto____$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling){
return (function (p1__8902_SHARP_){
return cljs.core.contains_QMARK_(p1__8902_SHARP_,concern);
});})(i__8918,s__8915__$1,vec__8927,id,v,c__5478__auto__,size__5479__auto__,b__8919,s__8917__$2,temp__5804__auto____$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling))
,cljs.core.keys(handlers)))], null));

var G__9024 = (i__8918 + (1));
i__8918 = G__9024;
continue;
} else {
return true;
}
break;
}
})()){
return cljs.core.chunk_cons(cljs.core.chunk(b__8919),atlas$query$error_handler_coverage_$_iter__8914_$_iter__8916(cljs.core.chunk_rest(s__8917__$2)));
} else {
return cljs.core.chunk_cons(cljs.core.chunk(b__8919),null);
}
} else {
var vec__8933 = cljs.core.first(s__8917__$2);
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8933,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8933,(1),null);
return cljs.core.cons(new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null,"entity","entity",-450970276),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"concern","concern",-531166280),concern,new cljs.core.Keyword(null,"has-handler?","has-handler?",1969404999),cljs.core.boolean$(cljs.core.some(((function (s__8915__$1,vec__8933,id,v,s__8917__$2,temp__5804__auto____$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling){
return (function (p1__8902_SHARP_){
return cljs.core.contains_QMARK_(p1__8902_SHARP_,concern);
});})(s__8915__$1,vec__8933,id,v,s__8917__$2,temp__5804__auto____$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling))
,cljs.core.keys(handlers)))], null),atlas$query$error_handler_coverage_$_iter__8914_$_iter__8916(cljs.core.rest(s__8917__$2)));
}
} else {
return null;
}
break;
}
});})(s__8915__$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling))
,null,null));
});})(s__8915__$1,concern,xs__6360__auto__,temp__5804__auto__,handlers,needs_handling))
;
var fs__5477__auto__ = cljs.core.seq(iterys__5476__auto__(atlas.query.find_by_aspect(registry,concern)));
if(fs__5477__auto__){
return cljs.core.concat.cljs$core$IFn$_invoke$arity$2(fs__5477__auto__,atlas$query$error_handler_coverage_$_iter__8914(cljs.core.rest(s__8915__$1)));
} else {
var G__9028 = cljs.core.rest(s__8915__$1);
s__8915__$1 = G__9028;
continue;
}
} else {
return null;
}
break;
}
}),null,null));
});
return iter__5480__auto__(needs_handling);
})()], null);
});
/**
 * Group architectural decisions by category.
 * Returns map of category→[decision-maps].
 * 
 * From atlas.semantic-queries/decisions-by-category
 */
atlas.query.decisions_by_category = (function atlas$query$decisions_by_category(registry,id_key){
return cljs.core.group_by(new cljs.core.Keyword(null,"category","category",-593092832),cljs.core.map.cljs$core$IFn$_invoke$arity$2((function (p__8961){
var vec__8962 = p__8961;
var id = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8962,(0),null);
var v = cljs.core.nth.cljs$core$IFn$_invoke$arity$3(vec__8962,(1),null);
return new cljs.core.PersistentArrayMap(null, 5, [new cljs.core.Keyword(null,"id","id",-1388402092),cljs.core.get.cljs$core$IFn$_invoke$arity$2(v,id_key),new cljs.core.Keyword(null,"category","category",-593092832),cljs.core.first(cljs.core.filter.cljs$core$IFn$_invoke$arity$2((function (p1__8952_SHARP_){
return cljs.core._EQ_.cljs$core$IFn$_invoke$arity$2("decision-category",cljs.core.namespace(p1__8952_SHARP_));
}),id)),new cljs.core.Keyword(null,"question","question",-1411720117),new cljs.core.Keyword("decision","question","decision/question",237153079).cljs$core$IFn$_invoke$arity$1(v),new cljs.core.Keyword(null,"chosen","chosen",-1422409985),new cljs.core.Keyword("decision","chosen","decision/chosen",-1919971621).cljs$core$IFn$_invoke$arity$1(v),new cljs.core.Keyword(null,"priority","priority",1431093715),new cljs.core.Keyword("decision","priority","decision/priority",-209382857).cljs$core$IFn$_invoke$arity$1(v)], null);
}),atlas.query.find_by_aspect(registry,new cljs.core.Keyword("semantic-namespace","architectural-decision","semantic-namespace/architectural-decision",-1719334219))));
});

//# sourceMappingURL=atlas.query.js.map
