(ns atlas-ui-v3.core
  "Atlas UI v3 - Treemap Aspect Explorer

   Progressive filtering treemap:
   - Show top 10 aspects by entity count
   - Rectangles sized by percentage
   - Color by namespace
   - Click to filter, show breadcrumb path"
  (:require [reagent.core :as r]
            [reagent.dom :as rdom]
            [atlas-ui-v2.api :as api]))

;; =============================================================================
;; State
;; =============================================================================

(defonce app-state
  (r/atom {:registry nil
           :loading? true
           :error nil
           :filter-path []  ; Vector of selected aspects for breadcrumb
           :auth-required? false  ; Set to true on 401
           :username ""
           :password ""}))

;; =============================================================================
;; Namespace Colors
;; =============================================================================

(def namespace-colors
  {"domain" "#4a9eff"
   "tier" "#9d4aef"
   "operation" "#ef4a9d"
   "effect" "#ef9d4a"
   "compliance" "#4aef7a"
   "security" "#ef4a4a"
   "observability" "#7aef4a"
   "scope" "#4a7aef"
   "entity" "#ef7a4a"
   "wrap" "#7a4aef"
   "atlas" "#aaaaaa"
   "default" "#6a6a8a"})

(defn namespace-color [aspect-kw]
  (let [ns-name (namespace aspect-kw)]
    (get namespace-colors ns-name (:default namespace-colors))))

;; =============================================================================
;; Data Processing
;; =============================================================================

(defn count-entities-by-aspect
  "Count how many entities have each aspect.
   Returns map of {aspect-kw count}"
  [registry]
  (reduce (fn [acc [identity _props]]
            (reduce (fn [acc2 aspect]
                      (update acc2 aspect (fnil inc 0)))
                    acc
                    identity))
          {}
          registry))

(defn filter-entities-by-aspects
  "Filter registry to entities containing ALL aspects in aspect-set"
  [registry aspect-set]
  (if (empty? aspect-set)
    registry
    (filter (fn [[identity _props]]
              (every? #(contains? identity %) aspect-set))
            registry)))

(defn top-n-aspects
  "Get top N aspects by entity count from filtered registry"
  [registry aspect-filter n]
  (let [filtered (filter-entities-by-aspects registry aspect-filter)
        counts (count-entities-by-aspect filtered)
        ;; Exclude aspects already in filter
        available-counts (apply dissoc counts aspect-filter)]
    (->> available-counts
         (sort-by val >)
         (take n))))

(defn calculate-treemap-rects
  "Calculate rectangle positions for treemap layout.
   Simple row-based layout: fill rows left-to-right, top-to-bottom"
  [aspects width height]
  (let [total (reduce + (map second aspects))
        aspect-data (map (fn [[aspect count]]
                           {:aspect aspect
                            :count count
                            :percentage (/ count total)})
                         aspects)
        ;; Calculate sizes
        rects (map (fn [data]
                     (assoc data
                            :width (* width (:percentage data))
                            :height (/ height 2))) ; 2 rows for now
                   aspect-data)
        ;; Position them
        positioned (loop [remaining rects
                          positioned []
                          x 0
                          y 0
                          row-width 0
                          row-height (/ height 2)]
                     (if (empty? remaining)
                       positioned
                       (let [rect (first remaining)
                             new-x (if (> (+ row-width (:width rect)) width)
                                     0  ; New row
                                     x)
                             new-y (if (> (+ row-width (:width rect)) width)
                                     (+ y row-height)
                                     y)
                             new-row-width (if (> (+ row-width (:width rect)) width)
                                             (:width rect)
                                             (+ row-width (:width rect)))]
                         (recur (rest remaining)
                                (conj positioned (assoc rect :x new-x :y new-y))
                                (+ new-x (:width rect))
                                new-y
                                new-row-width
                                row-height))))]
    positioned))

;; =============================================================================
;; Components
;; =============================================================================

(defn breadcrumb []
  (let [filter-path (:filter-path @app-state)]
    [:div {:style {:padding "1rem"
                   :background "#1a1a2e"
                   :color "#eee"
                   :border-bottom "1px solid #333"
                   :display "flex"
                   :gap "0.5rem"
                   :align-items "center"
                   :flex-wrap "wrap"}}
     [:span {:style {:font-weight "bold"
                     :color "#8a8aaa"}}
      "Filter path:"]
     (if (empty? filter-path)
       [:span {:style {:color "#6a6a8a"
                       :font-style "italic"}}
        "(none - showing all)"]
       (for [[idx aspect] (map-indexed vector filter-path)]
         ^{:key idx}
         [:div {:style {:display "flex"
                        :gap "0.5rem"
                        :align-items "center"}}
          (when (> idx 0)
            [:span {:style {:color "#555"}} "→"])
          [:span {:on-click (fn []
                              (swap! app-state assoc :filter-path
                                     (vec (take (inc idx) filter-path))))
                  :style {:padding "0.3rem 0.6rem"
                          :background (namespace-color aspect)
                          :border-radius "4px"
                          :cursor "pointer"
                          :font-family "monospace"
                          :font-size "0.9rem"
                          :transition "opacity 0.2s"
                          :opacity 1}
                  :on-mouse-over #(set! (-> % .-target .-style .-opacity) "0.7")
                  :on-mouse-out #(set! (-> % .-target .-style .-opacity) "1")}
           (str aspect)]]))
     (when (seq filter-path)
       [:button {:on-click #(swap! app-state assoc :filter-path [])
                 :style {:padding "0.3rem 0.6rem"
                         :background "#4a4a6a"
                         :color "#eee"
                         :border "none"
                         :border-radius "4px"
                         :cursor "pointer"
                         :margin-left "1rem"}}
        "Clear"])]))

(defn treemap-rect [{:keys [aspect count percentage x y width height]} on-click]
  (let [color (namespace-color aspect)
        aspect-str (str aspect)
        ns-name (namespace aspect)
        aspect-name (name aspect)]
    [:g {:on-click #(on-click aspect)
         :style {:cursor "pointer"}}
     [:rect {:x x
             :y y
             :width width
             :height height
             :fill color
             :stroke "#0f0f1a"
             :stroke-width 2
             :rx 4
             :opacity 0.8
             :on-mouse-over #(set! (-> % .-target .-style .-opacity) "1")
             :on-mouse-out #(set! (-> % .-target .-style .-opacity) "0.8")}]
     ;; Namespace label (small, top)
     [:text {:x (+ x 8)
             :y (+ y 16)
             :fill "rgba(255,255,255,0.6)"
             :font-size "11px"
             :font-family "monospace"
             :pointer-events "none"}
      ns-name]
     ;; Aspect name label (larger, center)
     [:text {:x (+ x (/ width 2))
             :y (+ y (/ height 2))
             :fill "#fff"
             :font-size "14px"
             :font-weight "bold"
             :font-family "monospace"
             :text-anchor "middle"
             :dominant-baseline "middle"
             :pointer-events "none"}
      aspect-name]
     ;; Count + percentage label (bottom)
     [:text {:x (+ x (/ width 2))
             :y (+ y height -12)
             :fill "rgba(255,255,255,0.8)"
             :font-size "12px"
             :font-family "monospace"
             :text-anchor "middle"
             :pointer-events "none"}
      (str count " entities (" (.toFixed (* percentage 100) 1) "%)")]]))

(defn treemap-view []
  (let [{:keys [registry filter-path]} @app-state
        top-aspects (top-n-aspects registry (set filter-path) 10)
        width 1200
        height 600
        rects (calculate-treemap-rects top-aspects width height)]
    [:div {:style {:padding "2rem"
                   :background "#0f0f1a"
                   :display "flex"
                   :justify-content "center"
                   :align-items "center"
                   :min-height "calc(100vh - 120px)"}}
     (if (empty? rects)
       [:div {:style {:color "#888"
                      :font-size "1.2rem"
                      :text-align "center"}}
        [:p "No more aspects to filter by."]
        [:p {:style {:font-size "0.9rem"
                     :margin-top "1rem"}}
         "Click 'Clear' above to start over."]]
       [:svg {:width width
              :height height
              :style {:background "#1a1a2e"
                      :border-radius "8px"
                      :box-shadow "0 4px 12px rgba(0,0,0,0.3)"}}
        [:g
         (for [rect rects]
           ^{:key (:aspect rect)}
           [treemap-rect rect
            (fn [aspect]
              (swap! app-state update :filter-path conj aspect))])]])]))

(defn header []
  [:div {:style {:padding "1.5rem"
                 :background "#1a1a2e"
                 :color "#eee"
                 :border-bottom "2px solid #333"}}
   [:h1 {:style {:margin 0
                 :font-size "2rem"
                 :font-weight 300}}
    "Atlas Explorer v3"
    [:span {:style {:color "#6a6a8a"
                    :font-size "1rem"
                    :margin-left "1rem"
                    :font-weight "normal"}}
     "Treemap Aspect Explorer"]]])

(defn legend []
  [:div {:style {:padding "1rem"
                 :background "#1a1a2e"
                 :border-top "1px solid #333"
                 :display "flex"
                 :gap "1.5rem"
                 :flex-wrap "wrap"
                 :justify-content "center"}}
   (for [[ns-name color] (sort-by first namespace-colors)]
     (when (not= ns-name "default")
       ^{:key ns-name}
       [:div {:style {:display "flex"
                      :align-items "center"
                      :gap "0.5rem"}}
        [:div {:style {:width "16px"
                       :height "16px"
                       :background color
                       :border-radius "3px"}}]
        [:span {:style {:color "#aaa"
                        :font-size "0.85rem"}}
         (str ns-name "/")]]))])

(defn stats []
  (let [{:keys [registry filter-path]} @app-state
        filtered (filter-entities-by-aspects registry (set filter-path))
        total-entities (count registry)
        filtered-entities (count filtered)]
    [:div {:style {:padding "1rem"
                   :background "#2a2a4e"
                   :color "#aaa"
                   :text-align "center"
                   :font-size "0.9rem"}}
     (if (empty? filter-path)
       [:span "Showing " [:strong {:style {:color "#eee"}} total-entities] " total entities"]
       [:span "Filtered to "
        [:strong {:style {:color "#4aef7a"}} filtered-entities]
        " of "
        [:strong {:style {:color "#eee"}} total-entities]
        " entities ("
        [:strong (.toFixed (* (/ filtered-entities total-entities) 100) 1) "%"]
        ")"])]))

(defn loading-view []
  [:div {:style {:display "flex"
                 :justify-content "center"
                 :align-items "center"
                 :height "100vh"
                 :background "#0f0f1a"
                 :color "#eee"}}
   [:div "Loading registry..."]])

(defn login-view []
  (let [{:keys [username password]} @app-state]
    [:div {:style {:display "flex"
                   :justify-content "center"
                   :align-items "center"
                   :height "100vh"
                   :background "#0f0f1a"
                   :color "#eee"}}
     [:div {:style {:background "#1a1a2e"
                    :padding "2rem"
                    :border-radius "8px"
                    :box-shadow "0 4px 12px rgba(0,0,0,0.3)"
                    :max-width "400px"}}
      [:h2 {:style {:margin "0 0 1rem 0"
                    :color "#eee"}}
       "Authentication Required"]
      [:p {:style {:margin "0 0 1.5rem 0"
                   :color "#aaa"
                   :font-size "0.9rem"}}
       "The registry server requires authentication."]
      [:div {:style {:margin-bottom "1rem"}}
       [:label {:style {:display "block"
                        :margin-bottom "0.5rem"
                        :color "#aaa"
                        :font-size "0.9rem"}}
        "Username"]
       [:input {:type "text"
                :value username
                :on-change #(swap! app-state assoc :username (-> % .-target .-value))
                :placeholder "atlas"
                :style {:width "100%"
                        :padding "0.5rem"
                        :background "#2a2a4e"
                        :color "#eee"
                        :border "1px solid #4a4a6a"
                        :border-radius "4px"
                        :font-size "1rem"}}]]
      [:div {:style {:margin-bottom "1.5rem"}}
       [:label {:style {:display "block"
                        :margin-bottom "0.5rem"
                        :color "#aaa"
                        :font-size "0.9rem"}}
        "Password"]
       [:input {:type "password"
                :value password
                :on-change #(swap! app-state assoc :password (-> % .-target .-value))
                :on-key-press #(when (= (.-key %) "Enter")
                                 (let [user (:username @app-state)
                                       pass (:password @app-state)
                                       port (-> (js/URLSearchParams. (.-search js/location))
                                               (.get "port"))
                                       new-url (str (.-protocol js/location) "//"
                                                   (.-hostname js/location) ":"
                                                   (.-port js/location)
                                                   "/?port=" port
                                                   "&user=" (js/encodeURIComponent user)
                                                   "&password=" (js/encodeURIComponent pass))]
                                   (set! (.-location js/window) new-url)))
                :placeholder "atlasp"
                :style {:width "100%"
                        :padding "0.5rem"
                        :background "#2a2a4e"
                        :color "#eee"
                        :border "1px solid #4a4a6a"
                        :border-radius "4px"
                        :font-size "1rem"}}]]
      [:button {:on-click (fn []
                            (let [user (:username @app-state)
                                  pass (:password @app-state)
                                  port (-> (js/URLSearchParams. (.-search js/location))
                                          (.get "port"))
                                  new-url (str (.-protocol js/location) "//"
                                              (.-hostname js/location) ":"
                                              (.-port js/location)
                                              "/?port=" port
                                              "&user=" (js/encodeURIComponent user)
                                              "&password=" (js/encodeURIComponent pass))]
                              (set! (.-location js/window) new-url)))
                :style {:width "100%"
                        :padding "0.75rem"
                        :background "#4a9eff"
                        :color "#fff"
                        :border "none"
                        :border-radius "4px"
                        :font-size "1rem"
                        :font-weight "bold"
                        :cursor "pointer"
                        :transition "background 0.2s"}
                :on-mouse-over #(set! (-> % .-target .-style .-background) "#3a8eef")
                :on-mouse-out #(set! (-> % .-target .-style .-background) "#4a9eff")}
       "Login"]
      [:p {:style {:margin "1rem 0 0 0"
                   :color "#6a6a8a"
                   :font-size "0.8rem"
                   :text-align "center"}}
       "Or add ?user=USERNAME&password=PASSWORD to URL"]]]))

(defn error-view [error]
  [:div {:style {:display "flex"
                 :justify-content "center"
                 :align-items "center"
                 :height "100vh"
                 :background "#0f0f1a"
                 :color "#ff6b6b"}}
   [:div {:style {:background "#1a1a2e"
                  :padding "2rem"
                  :border-radius "8px"
                  :max-width "500px"}}
    [:h2 "Error loading registry"]
    [:pre {:style {:color "#aaa"
                   :background "#0f0f1a"
                   :padding "1rem"
                   :border-radius "4px"
                   :overflow "auto"}}
     (pr-str error)]
    [:button {:on-click #(do
                           (swap! app-state assoc :loading? true :error nil)
                           (api/fetch-registry! on-registry-loaded on-registry-error))
              :style {:margin-top "1rem"
                      :padding "0.5rem 1rem"
                      :background "#4a9eff"
                      :color "#fff"
                      :border "none"
                      :border-radius "4px"
                      :cursor "pointer"}}
     "Retry"]]])

(defn app []
  (let [{:keys [loading? error registry auth-required?]} @app-state]
    (cond
      auth-required? [login-view]
      loading? [loading-view]
      error [error-view error]
      (nil? registry) [loading-view]
      :else
      [:div {:style {:display "flex"
                     :flex-direction "column"
                     :min-height "100vh"
                     :background "#0f0f1a"}}
       [header]
       [breadcrumb]
       [stats]
       [treemap-view]
       [legend]])))

;; =============================================================================
;; Init
;; =============================================================================

(defn on-registry-loaded [data]
  (let [registry (or (:atlas-ui.api.response/registry data) data)]
    (js/console.log "Registry loaded - entities:" (count registry))
    (swap! app-state assoc
           :registry registry
           :loading? false
           :error nil)))

(defn on-registry-error [error]
  (js/console.error "Registry error:" error)
  ;; Check if it's a 401 auth error
  (let [status (-> error :error :status)
        is-401? (= status 401)]
    (swap! app-state assoc
           :loading? false
           :error (when-not is-401? error)
           :auth-required? is-401?)))

(defn mount-root []
  (when-let [el (js/document.getElementById "app")]
    (rdom/render [app] el)))

(defn init []
  (js/console.log "Atlas UI v3 initializing...")
  (api/fetch-registry! on-registry-loaded on-registry-error)
  (mount-root))
