;; Copyright © 2022 Manetu, Inc.  All rights reserved
;;
;; SPDX-License-Identifier: Apache-2.0

(ns protojure.pedestal.interceptors.authz
  "A [Pedestal](http://pedestal.io/) [interceptor](http://pedestal.io/reference/interceptors) for authorizing Protojure GRPC endpoints"
  (:require [io.pedestal.interceptor :as pedestal]
            [protojure.pedestal.interceptors.grpc :as grpc]
            [protojure.grpc.status :as grpc.status]))

(defn- authz-enter
  [m pred {{:keys [path-info] :as request} :request :as context}]
  (let [e (get m path-info)]
    (if-not (pred (cond-> request (some? e) (assoc :grpc-requestinfo e)))
      (grpc.status/error :permission-denied)
      context)))

(defn- nested?
  "Returns true if the metadata is nested in a vector"
  [m]
  (vector? (first m)))

(defn- merge-metadata
  "Concatenates multiple metadata instances together into one list"
  [m]
  (apply concat m))

(defn ->metadata [m]
  (->> (cond-> m (nested? m) merge-metadata)
       (map #(select-keys % [:pkg :service :method]))
       (reduce
        (fn [acc entry]
          (assoc acc (str "/" (grpc/method-desc entry)) entry))
        {})))

(defn interceptor
  "
  Installs a predicate function to authorize requests.

  Arguments:

  - `m`:    rpc-metadata as generated by protojure compiler.  Can either be directly submitted as generated, or as a vector
            of 1 or more instances to support multiple interfaces.
  - 'pred': An arity-1 predicate function that accepts a pedestal request map as input.  Evaluating to true signals that the
            call is authorized, and should continue.  Evaluating to false stops further execution and triggers a permission
            denied response.  The request-map is augmented with :grpc-requestinfo containing the :pkg, :service, and :method of
            the call.

  "
  [m pred]
  (pedestal/interceptor {:name ::interceptor
                         :enter (partial authz-enter (->metadata m) pred)}))
