(ns bling.fontlib
  (:require [clojure.data]
            [clojure.pprint :refer [pprint]]
            [clojure.repl]
            [clojure.set]
            [clojure.string :as string]
            [bling.defs :as defs]
            [bling.macros :refer [? keyed]]
            [bling.util :refer [sjr]]))

(def ascii-chars
  ;; 32-126
  [
   " " 
   "!" 
   "\""
   "#" 
   "$" 
   "%" 
   "&" 
   "'" 
   "(" 
   ")" 
   "*" 
   "+" 
   "," 
   "-" 
   "." 
   "/" 
   "0" 
   "1" 
   "2" 
   "3" 
   "4" 
   "5" 
   "6" 
   "7" 
   "8" 
   "9" 
   ":" 
   ";" 
   "<" 
   "=" 
   ">" 
   "?" 
   "@"             
   "A"             
   "B"             
   "C"             
   "D"             
   "E"             
   "F"             
   "G"             
   "H"             
   "I"             
   "J"             
   "K"             
   "L"             
   "M"             
   "N"             
   "O"             
   "P"             
   "Q"             
   "R"             
   "S"             
   "T"             
   "U"             
   "V"             
   "W"             
   "X"             
   "Y"             
   "Z"             
   "["             
   "\\"             
   "]"             
   "^"             
   "_"
   "`"
   "a"
   "b"
   "c"
   "d"
   "e"
   "f"
   "g"
   "h"
   "i"
   "j"
   "k"
   "l"
   "m"
   "n"
   "o"
   "p"
   "q"
   "r"
   "s"
   "t"
   "u"
   "v"
   "w"
   "x"
   "y"
   "z"
   "{"
   "|"
   "}"
   "~"
   "Ä"
   "Ö"
   "Ü"
   "ä"
   "ö"
   "ü"
   "ß"
   ])

(def ascii-chars-partitioned-6-rows-vec
  (->> bling.fontlib/ascii-chars
       (partition 16) 
       (mapv string/join)))

(def ascii-chars-by-index-map
  (reduce 
   (fn [acc i]
     (assoc acc i (nth ascii-chars i :not-found)))
   {}
   (-> ascii-chars count range)))

(def ascii-indices-by-chars
  (clojure.set/map-invert ascii-chars-by-index-map))


(def isometric-1
 {:font-name      
  "Isometric 1"
  :example        
[
 "                 ___           ___           ___           ___           ___           ___                       ___"
 "     ___        /\\  \\         /\\  \\         /\\__\\         /\\  \\         /\\  \\         /\\  \\          ___        /\\  \\"
 "    /\\  \\      /::\\  \\       /::\\  \\       /::|  |       /::\\  \\        \\:\\  \\       /::\\  \\        /\\  \\      /::\\  \\"
 "    \\:\\  \\    /:/\\ \\  \\     /:/\\:\\  \\     /:|:|  |      /:/\\:\\  \\        \\:\\  \\     /:/\\:\\  \\       \\:\\  \\    /:/\\:\\  \\"
 "    /::\\__\\  _\\:\\~\\ \\  \\   /:/  \\:\\  \\   /:/|:|__|__   /::\\~\\:\\  \\       /::\\  \\   /::\\~\\:\\  \\      /::\\__\\  /:/  \\:\\  \\"
 " __/:/\\/__/ /\\ \\:\\ \\ \\__\\ /:/__/ \\:\\__\\ /:/ |::::\\__\\ /:/\\:\\ \\:\\__\\     /:/\\:\\__\\ /:/\\:\\ \\:\\__\\  __/:/\\/__/ /:/__/ \\:\\__\\"
 "/\\/:/  /    \\:\\ \\:\\ \\/__/ \\:\\  \\ /:/  / \\/__/~~/:/  / \\:\\~\\:\\ \\/__/    /:/  \\/__/ \\/_|::\\/:/  / /\\/:/  /    \\:\\  \\  \\/__/"
 "\\::/__/      \\:\\ \\:\\__\\    \\:\\  /:/  /        /:/  /   \\:\\ \\:\\__\\     /:/  /         |:|::/  /  \\::/__/      \\:\\  \\"
 " \\:\\__\\       \\:\\/:/  /     \\:\\/:/  /        /:/  /     \\:\\ \\/__/     \\/__/          |:|\\/__/    \\:\\__\\       \\:\\  \\"
 "  \\/__/        \\::/  /       \\::/  /        /:/  /       \\:\\__\\                      |:|  |       \\/__/        \\:\\__\\"
 "                \\/__/         \\/__/         \\/__/         \\/__/                       \\|__|                     \\/__/"
 ]
 :author         
 "Lennert Stock, Kent Nassen"
 :desc
"Figlet conversion by Kent Nassen (kentn@cyberspace.org), 8-10-94, based
on the fonts posted by Lennert Stock:

From: stock@fwi.uva.nl (Lennert Stock)
Date: 15 Jul 1994 00:04:25 GMT

Here are some fonts. Non-figlet I'm afraid, if you wanna convert them, be
my guest. I posted the isometric fonts before.

------------------------------------------------------------------------------

     .x%%%%%%x.                                             .x%%%%%%x.
    ,%%%%%%%%%%.                                           .%%%%%%%%%%.
   ,%%%'  )'  \\)                                           :(  `(  `%%%.
  ,%x%)________) --------- L e n n e r t   S t o c k       ( _   __ (%x%.
  (%%%~^88P~88P|                                           |~=> .=-~ %%%)
  (%%::. .:,\\ .'                                           `. /,:. .::%%)
  `;%:`\\. `-' |                                             | `-' ./':%:'
   ``x`. -===.'                   stock@fwi.uva.nl -------- `.===- .'x''
    / `:`.__.;                                               :.__.':' \\
 .d8b.     ..`.                                             .'..     .d8b."
   :figlet-metrics "flf2a$ 11 11 18 -1 23"
   :figlet-string
"$       $@
$       $@
$       $@
$       $@
$       $@
$       $@
$       $@
$       $@
$       $@
$       $@
$       $@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/__\\:\\/:/  /@
      \\::/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\__\\ @
 /:/\\:\\ \\:|__|@
 \\:\\~\\:\\/:/  /@
  \\:\\ \\::/  / @
   \\:\\/:/  /  @
    \\::/__/   @
     ‾‾       @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\  \\ @
 /:/__/ \\:\\__\\@
 \\:\\  \\  \\/__/@
  \\:\\  \\      @
   \\:\\  \\     @
    \\:\\__\\    @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\__\\ @
 /:/__/ \\:|__|@
 \\:\\  \\ /:/  /@
  \\:\\  /:/  / @
   \\:\\/:/  /  @
    \\::/__/   @
     ‾‾       @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\:\\~\\:\\ \\/__/@
  \\:\\ \\:\\__\\  @
   \\:\\ \\/__/  @
    \\:\\__\\    @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/__\\:\\ \\/__/@
      \\:\\__\\  @
       \\/__/  @
              @
              @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\  \\ @
 /:/__/_\\:\\__\\@
 \\:\\  /\\ \\/__/@
  \\:\\ \\:\\__\\  @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/__/     @
  /::\\  \\ ___ @
 /:/\\:\\  /\\__\\@
 \\/__\\:\\/:/  /@
      \\::/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
            @
      ___   @
     /\\  \\  @
     \\:\\  \\ @
     /::\\__\\@
  __/:/\\/__/@
 /\\/:/  /   @
 \\::/__/    @
  \\:\\__\\    @
   \\/__/    @
            @@
       ___   @
      /\\  \\  @
      \\:\\  \\ @
  ___ /::\\__\\@
 /\\  /:/\\/__/@
 \\:\\/:/  /   @
  \\::/  /    @
   \\/__/     @
             @
             @
             @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/__/     @
  /::\\__\\____ @
 /:/\\:::::\\__\\@
 \\/_|:|~~|~   @
    |:|  |    @
    |:|  |    @
    |:|  |    @
     \\|__|    @@
      ___ @
     /\\__\\@
    /:/  /@
   /:/  / @
  /:/  /  @
 /:/__/   @
 \\:\\  \\   @
  \\:\\  \\  @
   \\:\\  \\ @
    \\:\\__\\@
     \\/__/@@
      ___     @
     /\\__\\    @
    /::|  |   @
   /:|:|  |   @
  /:/|:|__|__ @
 /:/ |::::\\__\\@
 \\/__/~~/:/  /@
       /:/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\__\\    @
    /::|  |   @
   /:|:|  |   @
  /:/|:|  |__ @
 /:/ |:| /\\__\\@
 \\/__|:|/:/  /@
     |:/:/  / @
     |::/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\  \\ @
 /:/__/ \\:\\__\\@
 \\:\\  \\ /:/  /@
  \\:\\  /:/  / @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/__\\:\\/:/  /@
      \\::/  / @
       \\/__/  @
              @
              @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
   \\:\\~\\:\\  \\ @
    \\:\\ \\:\\__\\@
     \\:\\/:/  /@
      \\::/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/_|::\\/:/  /@
    |:|::/  / @
    |:|\\/__/  @
    |:|  |    @
     \\|__|    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\ \\  \\  @
  _\\:\\~\\ \\  \\ @
 /\\ \\:\\ \\ \\__\\@
 \\:\\ \\:\\ \\/__/@
  \\:\\ \\:\\__\\  @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
     \\:\\  \\   @
      \\:\\  \\  @
      /::\\  \\ @
     /:/\\:\\__\\@
    /:/  \\/__/@
   /:/  /     @
   \\/__/      @
              @
              @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/  /     @
  /:/  /  ___ @
 /:/__/  /\\__\\@
 \\:\\  \\ /:/  /@
  \\:\\  /:/  / @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/  /     @
  /:/__/  ___ @
  |:|  | /\\__\\@
  |:|  |/:/  /@
  |:|__/:/  / @
   \\::::/__/  @
    ~~~~      @
              @@
      ___     @
     /\\__\\    @
    /:/ _/_   @
   /:/ /\\__\\  @
  /:/ /:/ _/_ @
 /:/_/:/ /\\__\\@
 \\:\\/:/ /:/  /@
  \\::/_/:/  / @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     |\\__\\    @
     |:|  |   @
     |:|  |   @
     |:|__|__ @
 ____/::::\\__\\@
 \\::::/~~/~   @
  ~~|:|~~|    @
    |:|  |    @
    |:|  |    @
     \\|__|    @@
      ___     @
     |\\__\\    @
     |:|  |   @
     |:|  |   @
     |:|__|__ @
     /::::\\__\\@
    /:/~~/~   @
   /:/  /     @
   \\/__/      @
              @
              @@
      ___     @
     /\\  \\    @
     \\:\\  \\   @
      \\:\\  \\  @
       \\:\\  \\ @
 _______\\:\\__\\@
 \\::::::::/__/@
  \\:\\~~\\~~    @
   \\:\\  \\     @
    \\:\\__\\    @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /::::\\  \\  @
  /::::::\\  \\ @
 /:::LS:::\\__\\@
 \\::1994::/  /@
  \\::::::/  / @
   \\::::/  /  @
    \\::/  /   @
     \\/__/    @@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/__\\:\\/:/  /@
      \\::/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\__\\ @
 /:/\\:\\ \\:|__|@
 \\:\\~\\:\\/:/  /@
  \\:\\ \\::/  / @
   \\:\\/:/  /  @
    \\::/__/   @
     ‾‾       @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\  \\ @
 /:/__/ \\:\\__\\@
 \\:\\  \\  \\/__/@
  \\:\\  \\      @
   \\:\\  \\     @
    \\:\\__\\    @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\__\\ @
 /:/__/ \\:|__|@
 \\:\\  \\ /:/  /@
  \\:\\  /:/  / @
   \\:\\/:/  /  @
    \\::/__/   @
     ‾‾       @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\:\\~\\:\\ \\/__/@
  \\:\\ \\:\\__\\  @
   \\:\\ \\/__/  @
    \\:\\__\\    @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/__\\:\\ \\/__/@
      \\:\\__\\  @
       \\/__/  @
              @
              @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\  \\ @
 /:/__/_\\:\\__\\@
 \\:\\  /\\ \\/__/@
  \\:\\ \\:\\__\\  @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/__/     @
  /::\\  \\ ___ @
 /:/\\:\\  /\\__\\@
 \\/__\\:\\/:/  /@
      \\::/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
            @
      ___   @
     /\\  \\  @
     \\:\\  \\ @
     /::\\__\\@
  __/:/\\/__/@
 /\\/:/  /   @
 \\::/__/    @
  \\:\\__\\    @
   \\/__/    @
            @@
       ___   @
      /\\  \\  @
      \\:\\  \\ @
  ___ /::\\__\\@
 /\\  /:/\\/__/@
 \\:\\/:/  /   @
  \\::/  /    @
   \\/__/     @
             @
             @
             @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/__/     @
  /::\\__\\____ @
 /:/\\:::::\\__\\@
 \\/_|:|~~|~   @
    |:|  |    @
    |:|  |    @
    |:|  |    @
     \\|__|    @@
      ___ @
     /\\__\\@
    /:/  /@
   /:/  / @
  /:/  /  @
 /:/__/   @
 \\:\\  \\   @
  \\:\\  \\  @
   \\:\\  \\ @
    \\:\\__\\@
     \\/__/@@
      ___     @
     /\\__\\    @
    /::|  |   @
   /:|:|  |   @
  /:/|:|__|__ @
 /:/ |::::\\__\\@
 \\/__/~~/:/  /@
       /:/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\__\\    @
    /::|  |   @
   /:|:|  |   @
  /:/|:|  |__ @
 /:/ |:| /\\__\\@
 \\/__|:|/:/  /@
     |:/:/  / @
     |::/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /:/  \\:\\  \\ @
 /:/__/ \\:\\__\\@
 \\:\\  \\ /:/  /@
  \\:\\  /:/  / @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/__\\:\\/:/  /@
      \\::/  / @
       \\/__/  @
              @
              @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
   \\:\\~\\:\\  \\ @
    \\:\\ \\:\\__\\@
     \\:\\/:/  /@
      \\::/  / @
      /:/  /  @
     /:/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\:\\  \\  @
  /::\\~\\:\\  \\ @
 /:/\\:\\ \\:\\__\\@
 \\/_|::\\/:/  /@
    |:|::/  / @
    |:|\\/__/  @
    |:|  |    @
     \\|__|    @@
      ___     @
     /\\  \\    @
    /::\\  \\   @
   /:/\\ \\  \\  @
  _\\:\\~\\ \\  \\ @
 /\\ \\:\\ \\ \\__\\@
 \\:\\ \\:\\ \\/__/@
  \\:\\ \\:\\__\\  @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\  \\    @
     \\:\\  \\   @
      \\:\\  \\  @
      /::\\  \\ @
     /:/\\:\\__\\@
    /:/  \\/__/@
   /:/  /     @
   \\/__/      @
              @
              @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/  /     @
  /:/  /  ___ @
 /:/__/  /\\__\\@
 \\:\\  \\ /:/  /@
  \\:\\  /:/  / @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     /\\__\\    @
    /:/  /    @
   /:/  /     @
  /:/__/  ___ @
  |:|  | /\\__\\@
  |:|  |/:/  /@
  |:|__/:/  / @
   \\::::/__/  @
    ‾‾‾‾      @
              @@
      ___     @
     /\\__\\    @
    /:/ _/_   @
   /:/ /\\__\\  @
  /:/ /:/ _/_ @
 /:/_/:/ /\\__\\@
 \\:\\/:/ /:/  /@
  \\::/_/:/  / @
   \\:\\/:/  /  @
    \\::/  /   @
     \\/__/    @@
      ___     @
     |\\__\\    @
     |:|  |   @
     |:|  |   @
     |:|__|__ @
 ____/::::\\__\\@
 \\::::/~~/~   @
  ~~|:|~~|    @
    |:|  |    @
    |:|  |    @
     \\|__|    @@
      ___     @
     |\\__\\    @
     |:|  |   @
     |:|  |   @
     |:|__|__ @
     /::::\\__\\@
    /:/~~/~   @
   /:/  /     @
   \\/__/      @
              @
              @@
      ___     @
     /\\  \\    @
     \\:\\  \\   @
      \\:\\  \\  @
       \\:\\  \\ @
 _______\\:\\__\\@
 \\::::::::/__/@
  \\:\\~~\\~~    @
   \\:\\  \\     @
    \\:\\__\\    @
     \\/__/    @@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@
@@"})



(def ansi-shadow
{
 :font-name "ANSI Shadow"
 :author "?"
 :desc ""
 :more-info
 "https://web.archive.org/web/20120819044459/http://www.roysac.com/thedrawfonts-tdf.asp"
 :url ""
 :example
[
" █████╗ ███╗   ██╗███████╗██╗    ███████╗██╗  ██╗ █████╗ ██████╗  ██████╗ ██╗    ██╗"
"██╔══██╗████╗  ██║██╔════╝██║    ██╔════╝██║  ██║██╔══██╗██╔══██╗██╔═══██╗██║    ██║"
"███████║██╔██╗ ██║███████╗██║    ███████╗███████║███████║██║  ██║██║   ██║██║ █╗ ██║"
"██╔══██║██║╚██╗██║╚════██║██║    ╚════██║██╔══██║██╔══██║██║  ██║██║   ██║██║███╗██║"
"██║  ██║██║ ╚████║███████║██║    ███████║██║  ██║██║  ██║██████╔╝╚██████╔╝╚███╔███╔╝"
"╚═╝  ╚═╝╚═╝  ╚═══╝╚══════╝╚═╝    ╚══════╝╚═╝  ╚═╝╚═╝  ╚═╝╚═════╝  ╚═════╝  ╚══╝╚══╝ "
]
 :figlet-metrics "flf2a$ 7 7 13 0 7 0 64 0"
 :figlet-string
 "@@
██╗@
██║@
██║@
╚═╝@
██╗@
╚═╝@
   @@
██╗██╗@
╚═╝╚═╝@
      @
      @
      @
      @
      @@
 ██╗ ██╗ @
████████╗@
╚██╔═██╔╝@
████████╗@
╚██╔═██╔╝@
 ╚═╝ ╚═╝ @
         @@
▄▄███▄▄·@
██╔════╝@
███████╗@
╚════██║@
███████║@
╚═▀▀▀══╝@
        @@
██╗ ██╗@
╚═╝██╔╝@
  ██╔╝ @
 ██╔╝  @
██╔╝██╗@
╚═╝ ╚═╝@
       @@
   ██╗   @
   ██║   @
████████╗@
██╔═██╔═╝@
██████║  @
╚═════╝  @
         @@
██╗@
╚═╝@
   @
   @
   @
   @
   @@
 ██╗@
██╔╝@
██║ @
██║ @
╚██╗@
 ╚═╝@
    @@
██╗ @
╚██╗@
 ██║@
 ██║@
██╔╝@
╚═╝ @
    @@
      @
▄ ██╗▄@
 ████╗@
▀╚██╔▀@
  ╚═╝ @
      @
       @@
       @
  ██╗  @
██████╗@
╚═██╔═╝@
  ╚═╝  @
       @
       @@
   @
   @
   @
   @
▄█╗@
╚═╝@
   @@
      @
      @
█████╗@
╚════╝@
      @
      @
      @@
   @
   @
   @
   @
██╗@
╚═╝@
   @@
    ██╗@
   ██╔╝@
  ██╔╝ @
 ██╔╝  @
██╔╝   @
╚═╝    @
       @@
 ██████╗ @
██╔═████╗@
██║██╔██║@
████╔╝██║@
╚██████╔╝@
 ╚═════╝ @
         @@
 ██╗@
███║@
╚██║@
 ██║@
 ██║@
 ╚═╝@
    @@
██████╗ @
╚════██╗@
 █████╔╝@
██╔═══╝ @
███████╗@
╚══════╝@
        @@
██████╗ @
╚════██╗@
 █████╔╝@
 ╚═══██╗@
██████╔╝@
╚═════╝ @
        @@
██╗  ██╗@
██║  ██║@
███████║@
╚════██║@
     ██║@
     ╚═╝@
        @@
███████╗@
██╔════╝@
███████╗@
╚════██║@
███████║@
╚══════╝@
        @@
 ██████╗ @
██╔════╝ @
███████╗ @
██╔═══██╗@
╚██████╔╝@
 ╚═════╝ @
         @@
███████╗@
╚════██║@
    ██╔╝@
   ██╔╝ @
   ██║  @
   ╚═╝  @
        @@
 █████╗ @
██╔══██╗@
╚█████╔╝@
██╔══██╗@
╚█████╔╝@
 ╚════╝ @
        @@
 █████╗ @
██╔══██╗@
╚██████║@
 ╚═══██║@
 █████╔╝@
 ╚════╝ @
        @@
   @
██╗@
╚═╝@
██╗@
╚═╝@
   @
   @@
   @
██╗@
╚═╝@
▄█╗@
▀═╝@
   @
   @@
  ██╗@
 ██╔╝@
██╔╝ @
╚██╗ @
 ╚██╗@
  ╚═╝@
     @@
       @
██████╗@
╚═════╝@
██████╗@
╚═════╝@
       @
       @@
██╗  @
╚██╗ @
 ╚██╗@
 ██╔╝@
██╔╝ @
╚═╝  @
     @@
██████╗ @
╚════██╗@
  ▄███╔╝@
  ▀▀══╝ @
  ██╗   @
  ╚═╝   @
        @@
 ██████╗ @
██╔═══██╗@
██║██╗██║@
██║██║██║@
╚█║████╔╝@
 ╚╝╚═══╝ @
         @@
 █████╗ @
██╔══██╗@
███████║@
██╔══██║@
██║  ██║@
╚═╝  ╚═╝@
        @@
██████╗ @
██╔══██╗@
██████╔╝@
██╔══██╗@
██████╔╝@
╚═════╝ @
        @@
 ██████╗@
██╔════╝@
██║     @
██║     @
╚██████╗@
 ╚═════╝@
        @@
██████╗ @
██╔══██╗@
██║  ██║@
██║  ██║@
██████╔╝@
╚═════╝ @
        @@
███████╗@
██╔════╝@
█████╗  @
██╔══╝  @
███████╗@
╚══════╝@
        @@
███████╗@
██╔════╝@
█████╗  @
██╔══╝  @
██║     @
╚═╝     @
        @@
 ██████╗ @
██╔════╝ @
██║  ███╗@
██║   ██║@
╚██████╔╝@
 ╚═════╝ @
         @@
██╗  ██╗@
██║  ██║@
███████║@
██╔══██║@
██║  ██║@
╚═╝  ╚═╝@
        @@
██╗@
██║@
██║@
██║@
██║@
╚═╝@
   @@
     ██╗@
     ██║@
     ██║@
██   ██║@
╚█████╔╝@
 ╚════╝ @
        @@
██╗  ██╗@
██║ ██╔╝@
█████╔╝ @
██╔═██╗ @
██║  ██╗@
╚═╝  ╚═╝@
        @@
██╗     @
██║     @
██║     @
██║     @
███████╗@
╚══════╝@
        @@
███╗   ███╗@
████╗ ████║@
██╔████╔██║@
██║╚██╔╝██║@
██║ ╚═╝ ██║@
╚═╝     ╚═╝@
           @@
███╗   ██╗@
████╗  ██║@
██╔██╗ ██║@
██║╚██╗██║@
██║ ╚████║@
╚═╝  ╚═══╝@
          @@
 ██████╗ @
██╔═══██╗@
██║   ██║@
██║   ██║@
╚██████╔╝@
 ╚═════╝ @
         @@
██████╗ @
██╔══██╗@
██████╔╝@
██╔═══╝ @
██║     @
╚═╝     @
        @@
 ██████╗ @
██╔═══██╗@
██║   ██║@
██║▄▄ ██║@
╚██████╔╝@
 ╚══▀▀═╝ @
         @@
██████╗ @
██╔══██╗@
██████╔╝@
██╔══██╗@
██║  ██║@
╚═╝  ╚═╝@
        @@
███████╗@
██╔════╝@
███████╗@
╚════██║@
███████║@
╚══════╝@
        @@
████████╗@
╚══██╔══╝@
   ██║   @
   ██║   @
   ██║   @
   ╚═╝   @
         @@
██╗   ██╗@
██║   ██║@
██║   ██║@
██║   ██║@
╚██████╔╝@
 ╚═════╝ @
         @@
██╗   ██╗@
██║   ██║@
██║   ██║@
╚██╗ ██╔╝@
 ╚████╔╝ @
  ╚═══╝  @
         @@
██╗    ██╗@
██║    ██║@
██║ █╗ ██║@
██║███╗██║@
╚███╔███╔╝@
 ╚══╝╚══╝ @
          @@
██╗  ██╗@
╚██╗██╔╝@
 ╚███╔╝ @
 ██╔██╗ @
██╔╝ ██╗@
╚═╝  ╚═╝@
        @@
██╗   ██╗@
╚██╗ ██╔╝@
 ╚████╔╝ @
  ╚██╔╝  @
   ██║   @
   ╚═╝   @
         @@
███████╗@
╚══███╔╝@
  ███╔╝ @
 ███╔╝  @
███████╗@
╚══════╝@
        @@
███╗@
██╔╝@
██║ @
██║ @
███╗@
╚══╝@
    @@
██╗    @
╚██╗   @
 ╚██╗  @
  ╚██╗ @
   ╚██╗@
    ╚═╝@
       @@
███╗@
╚██║@
 ██║@
 ██║@
███║@
╚══╝@
    @@
 ███╗ @
██╔██╗@
╚═╝╚═╝@
      @
      @
      @
      @@
        @
        @
        @
        @
███████╗@
╚══════╝@
        @@
██╗     @
╚██╗    @
 ╚═╝    @
        @
        @
        @
       @@
 █████╗ @
██╔══██╗@
███████║@
██╔══██║@
██║  ██║@
╚═╝  ╚═╝@
        @@
██████╗ @
██╔══██╗@
██████╔╝@
██╔══██╗@
██████╔╝@
╚═════╝ @
        @@
 ██████╗@
██╔════╝@
██║     @
██║     @
╚██████╗@
 ╚═════╝@
        @@
██████╗ @
██╔══██╗@
██║  ██║@
██║  ██║@
██████╔╝@
╚═════╝ @
        @@
███████╗@
██╔════╝@
█████╗  @
██╔══╝  @
███████╗@
╚══════╝@
        @@
███████╗@
██╔════╝@
█████╗  @
██╔══╝  @
██║     @
╚═╝     @
        @@
 ██████╗ @
██╔════╝ @
██║  ███╗@
██║   ██║@
╚██████╔╝@
 ╚═════╝ @
         @@
██╗  ██╗@
██║  ██║@
███████║@
██╔══██║@
██║  ██║@
╚═╝  ╚═╝@
        @@
██╗@
██║@
██║@
██║@
██║@
╚═╝@
   @@
     ██╗@
     ██║@
     ██║@
██   ██║@
╚█████╔╝@
 ╚════╝ @
        @@
██╗  ██╗@
██║ ██╔╝@
█████╔╝ @
██╔═██╗ @
██║  ██╗@
╚═╝  ╚═╝@
        @@
██╗     @
██║     @
██║     @
██║     @
███████╗@
╚══════╝@
        @@
███╗   ███╗@
████╗ ████║@
██╔████╔██║@
██║╚██╔╝██║@
██║ ╚═╝ ██║@
╚═╝     ╚═╝@
           @@
███╗   ██╗@
████╗  ██║@
██╔██╗ ██║@
██║╚██╗██║@
██║ ╚████║@
╚═╝  ╚═══╝@
          @@
 ██████╗ @
██╔═══██╗@
██║   ██║@
██║   ██║@
╚██████╔╝@
 ╚═════╝ @
         @@
██████╗ @
██╔══██╗@
██████╔╝@
██╔═══╝ @
██║     @
╚═╝     @
        @@
 ██████╗ @
██╔═══██╗@
██║   ██║@
██║▄▄ ██║@
╚██████╔╝@
 ╚══▀▀═╝ @
         @@
██████╗ @
██╔══██╗@
██████╔╝@
██╔══██╗@
██║  ██║@
╚═╝  ╚═╝@
        @@
███████╗@
██╔════╝@
███████╗@
╚════██║@
███████║@
╚══════╝@
        @@
████████╗@
╚══██╔══╝@
   ██║   @
   ██║   @
   ██║   @
   ╚═╝   @
         @@
██╗   ██╗@
██║   ██║@
██║   ██║@
██║   ██║@
╚██████╔╝@
 ╚═════╝ @
         @@
██╗   ██╗@
██║   ██║@
██║   ██║@
╚██╗ ██╔╝@
 ╚████╔╝ @
  ╚═══╝  @
         @@
██╗    ██╗@
██║    ██║@
██║ █╗ ██║@
██║███╗██║@
╚███╔███╔╝@
 ╚══╝╚══╝ @
          @@
██╗  ██╗@
╚██╗██╔╝@
 ╚███╔╝ @
 ██╔██╗ @
██╔╝ ██╗@
╚═╝  ╚═╝@
        @@
██╗   ██╗@
╚██╗ ██╔╝@
 ╚████╔╝ @
  ╚██╔╝  @
   ██║   @
   ╚═╝   @
         @@
███████╗@
╚══███╔╝@
  ███╔╝ @
 ███╔╝  @
███████╗@
╚══════╝@
        @@"})


(def big-money
{
 :font-name      
 "Big Money"
 :example        
 [
"$$$$$$$\\  $$\\                $$\\      $$\\"
"$$  __$$\\ \\__|               $$$\\    $$$ |"
"$$ |  $$ |$$\\  $$$$$$\\       $$$$\\  $$$$ | $$$$$$\\  $$$$$$$\\   $$$$$$\\  $$\\   $$\\"
"$$$$$$$\\ |$$ |$$  __$$\\      $$\\$$\\$$ $$ |$$  __$$\\ $$  __$$\\ $$  __$$\\ $$ |  $$ |"
"$$  __$$\\ $$ |$$ /  $$ |     $$ \\$$$  $$ |$$ /  $$ |$$ |  $$ |$$$$$$$$ |$$ |  $$ |"
"$$ |  $$ |$$ |$$ |  $$ |     $$ |\\$  /$$ |$$ |  $$ |$$ |  $$ |$$   ____|$$ |  $$ |"
"$$$$$$$  |$$ |\\$$$$$$$ |     $$ | \\_/ $$ |\\$$$$$$  |$$ |  $$ |\\$$$$$$$\\ \\$$$$$$$ |"
"\\_______/ \\__| \\____$$ |     \\__|     \\__| \\______/ \\__|  \\__| \\_______| \\____$$ |"
"              $$\\   $$ |                                                $$\\   $$ |"
"              \\$$$$$$  |                                                \\$$$$$$  |"
"               \\______/                                                  \\______/"
 ]
 :author         
 "nathan bloomfield"
 :desc
 "bigmoney-nw : by nathan bloomfield (xzovik@gmail.com)
  based on art from the legendary MAKEMONEYFAST chain letter

  History:
  5-30-2007 : first version (required characters only)"
   :figlet-metrics "flf2a& 11 8 18 0 25"
   :figlet-string
"@@
$$\\ @
$$ |@
$$ |@
$$ |@
\\__|@
    @
$$\\ @
\\__|@
    @
    @
    @@
$$\\ $$\\ @
$$ |$$ |@
$$ |$$ |@
\\__|\\__|@
        @
        @
        @
        @
        @
        @
        @@
  $$\\ $$\\   @
  $$ \\$$ \\  @
$$$$$$$$$$\\ @
\\_$$  $$   |@
$$$$$$$$$$\\ @
\\_$$  $$  _|@
  $$ |$$ |  @
  \\__|\\__|  @
            @
            @
            @@
   $$\\    @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  \\__|@
\\$$$$$$\\  @
 \\___ $$\\ @
$$\\  \\$$ |@
\\$$$$$$  |@
 \\_$$  _/ @
   \\ _/   @
          @@
$$\\   $$\\ @
\\__| $$  |@
    $$  / @
   $$  /  @
  $$  /   @
 $$  /    @
$$  / $$\\ @
\\__/  \\__|@
          @
          @
          @@
 $$$\\     @
$$ $$\\    @
\\$$$\\ |   @
$$\\$$\\$$\\ @
$$ \\$$ __|@
$$ |\\$$\\  @
 $$$$ $$\\ @
 \\____\\__|@
          @
          @
          @@
$$\\ @
$  |@
\\_/ @
    @
    @
    @
    @
    @
    @
    @
    @@
  $$$\\ @
 $$  _|@
$$  /  @
$$ |   @
$$ |   @
\\$$\\   @
 \\$$$\\ @
  \\___|@
       @
       @
       @@
$$$\\   @
 \\$$\\  @
  \\$$\\ @
   $$ |@
   $$ |@
  $$  |@
$$$  / @
\\___/  @
       @
       @
       @@
         @
 $$\\$$\\  @
 \\$$$  | @
$$$$$$$\\ @
\\_$$$ __|@
 $$ $$\\  @
 \\__\\__| @
         @
         @
         @
         @@
          @
   $$\\    @
   $$ |   @
$$$$$$$$\\ @
\\__$$  __|@
   $$ |   @
   \\__|   @
          @
          @
          @
          @@
    @
    @
    @
    @
    @
    @
$$\\ @
$  |@
\\_/ @
    @
    @@
        @
        @
        @
$$$$$$\\ @
\\______|@
        @
        @
        @
        @
        @
        @@
    @
    @
    @
    @
    @
    @
$$\\ @
\\__|@
    @
    @
    @@
      $$\\ @
     $$  |@
    $$  / @
   $$  /  @
  $$  /   @
 $$  /    @
$$  /     @
\\__/      @
          @
          @
          @@
 $$$$$$\\  @
$$$ __$$\\ @
$$$$\\ $$ |@
$$\\$$\\$$ |@
$$ \\$$$$ |@
$$ |\\$$$ |@
\\$$$$$$  /@
 \\______/ @
          @
          @
          @@
  $$\\   @
$$$$ |  @
\\_$$ |  @
  $$ |  @
  $$ |  @
  $$ |  @
$$$$$$\\ @
\\______|@
        @
        @
        @@
 $$$$$$\\  @
$$  __$$\\ @
\\__/  $$ |@
 $$$$$$  |@
$$  ____/ @
$$ |      @
$$$$$$$$\\ @
\\________|@
          @
          @
          @@
 $$$$$$\\  @
$$ ___$$\\ @
\\_/   $$ |@
  $$$$$ / @
  \\___$$\\ @
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$$$$$$$ |@
\\_____$$ |@
      $$ |@
      $$ |@
      \\__|@
          @
          @
          @@
$$$$$$$\\  @
$$  ____| @
$$ |      @
$$$$$$$\\  @
\\_____$$\\ @
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  \\__|@
$$$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
 $$$$$$  |@
 \\______/ @
          @
          @
          @@
$$$$$$$$\\ @
\\____$$  |@
    $$  / @
   $$  /  @
  $$  /   @
 $$  /    @
$$  /     @
\\__/      @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
 $$$$$$  |@
$$  __$$< @
$$ /  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
\\$$$$$$$ |@
 \\____$$ |@
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
    @
    @
$$\\ @
\\__|@
    @
$$\\ @
\\__|@
    @
    @
    @
    @@
    @
    @
$$\\ @
\\__|@
    @
$$\\ @
$  |@
\\_/ @
    @
    @
    @@
   $$\\ @
  $$  |@
 $$  / @
$$  /  @
\\$$<   @
 \\$$\\  @
  \\$$\\ @
   \\__|@
       @
       @
       @@
      @
      @
$$$$\\ @
\\____|@
$$$$\\ @
\\____|@
      @
      @
      @
      @
      @@
$$\\    @
\\$$\\   @
 \\$$\\  @
  \\$$\\ @
  $$  |@
 $$  / @
$$  /  @
\\__/   @
       @
       @
       @@
 $$$$\\  @
$$  $$\\ @
\\__/$$ |@
   $$  |@
  $$  / @
  \\__/  @
  $$\\   @
  \\__|  @
        @
        @
        @@
    $$$$$$\\     @
  $$$ ___$$$\\   @
 $$ _/   \\_$$\\  @
$$ / $$$$$\\ $$\\ @
$$ |$$  $$ |$$ |@
$$ |$$ /$$ |$$ |@
$$ |\\$$$$$$$$  |@
\\$$\\ \\________/ @
 \\$$$\\   $$$\\   @
  \\_$$$$$$  _|  @
    \\______/    @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$$$$$$$ |@
$$  __$$ |@
$$ |  $$ |@
$$ |  $$ |@
\\__|  \\__|@
          @
          @
          @@
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$$$$$$\\ |@
$$  __$$\\ @
$$ |  $$ |@
$$$$$$$  |@
\\_______/ @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  \\__|@
$$ |      @
$$ |      @
$$ |  $$\\ @
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
$$$$$$$  |@
\\_______/ @
          @
          @
          @@
$$$$$$$$\\ @
$$  _____|@
$$ |      @
$$$$$\\    @
$$  __|   @
$$ |      @
$$$$$$$$\\ @
\\________|@
          @
          @
          @@
$$$$$$$$\\ @
$$  _____|@
$$ |      @
$$$$$\\    @
$$  __|   @
$$ |      @
$$ |      @
\\__|      @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  \\__|@
$$ |$$$$\\ @
$$ |\\_$$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$$$$$$$ |@
$$  __$$ |@
$$ |  $$ |@
$$ |  $$ |@
\\__|  \\__|@
          @
          @
          @@
$$$$$$\\ @
\\_$$  _|@
  $$ |  @
  $$ |  @
  $$ |  @
  $$ |  @
$$$$$$\\ @
\\______|@
        @
        @
        @@
   $$$$$\\ @
   \\__$$ |@
      $$ |@
      $$ |@
$$\\   $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
$$\\   $$\\ @
$$ | $$  |@
$$ |$$  / @
$$$$$  /  @
$$  $$<   @
$$ |\\$$\\  @
$$ | \\$$\\ @
\\__|  \\__|@
          @
          @
          @@
$$\\       @
$$ |      @
$$ |      @
$$ |      @
$$ |      @
$$ |      @
$$$$$$$$\\ @
\\________|@
          @
          @
          @@
$$\\      $$\\ @
$$$\\    $$$ |@
$$$$\\  $$$$ |@
$$\\$$\\$$ $$ |@
$$ \\$$$  $$ |@
$$ |\\$  /$$ |@
$$ | \\_/ $$ |@
\\__|     \\__|@
             @
             @
             @@
$$\\   $$\\ @
$$$\\  $$ |@
$$$$\\ $$ |@
$$ $$\\$$ |@
$$ \\$$$$ |@
$$ |\\$$$ |@
$$ | \\$$ |@
\\__|  \\__|@
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
 $$$$$$  |@
 \\______/ @
          @
          @
          @@
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$$$$$$  |@
$$  ____/ @
$$ |      @
$$ |      @
\\__|      @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
$$ |  $$ |@
$$ $$\\$$ |@
\\$$$$$$ / @
 \\___$$$\\ @
     \\___|@
          @
          @@
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$$$$$$  |@
$$  __$$< @
$$ |  $$ |@
$$ |  $$ |@
\\__|  \\__|@
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  \\__|@
\\$$$$$$\\  @
 \\____$$\\ @
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
$$$$$$$$\\ @
\\__$$  __|@
   $$ |   @
   $$ |   @
   $$ |   @
   $$ |   @
   $$ |   @
   \\__|   @
          @
          @
          @@
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
$$\\    $$\\ @
$$ |   $$ |@
$$ |   $$ |@
\\$$\\  $$  |@
 \\$$\\$$  / @
  \\$$$  /  @
   \\$  /   @
    \\_/    @
           @
           @
           @@
$$\\      $$\\ @
$$ | $\\  $$ |@
$$ |$$$\\ $$ |@
$$ $$ $$\\$$ |@
$$$$  _$$$$ |@
$$$  / \\$$$ |@
$$  /   \\$$ |@
\\__/     \\__|@
             @
             @
             @@
$$\\   $$\\ @
$$ |  $$ |@
\\$$\\ $$  |@
 \\$$$$  / @
 $$  $$<  @
$$  /\\$$\\ @
$$ /  $$ |@
\\__|  \\__|@
          @
          @
          @@
$$\\     $$\\ @
\\$$\\   $$  |@
 \\$$\\ $$  / @
  \\$$$$  /  @
   \\$$  /   @
    $$ |    @
    $$ |    @
    \\__|    @
            @
            @
            @@
$$$$$$$$\\ @
\\____$$  |@
    $$  / @
   $$  /  @
  $$  /   @
 $$  /    @
$$$$$$$$\\ @
\\________|@
          @
          @
          @@
$$$$\\ @
$$  _|@
$$ |  @
$$ |  @
$$ |  @
$$ |  @
$$$$\\ @
\\____|@
      @
      @
      @@
$$\\       @
\\$$\\      @
 \\$$\\     @
  \\$$\\    @
   \\$$\\   @
    \\$$\\  @
     \\$$\\ @
      \\__|@
          @
          @
          @@
$$$$\\ @
\\_$$ |@
  $$ |@
  $$ |@
  $$ |@
  $$ |@
$$$$ |@
\\____|@
      @
      @
      @@
   $\\    @
  $$$\\   @
 $$ $$\\  @
$$  \\$$\\ @
\\__/ \\__|@
         @
         @
         @
         @
         @
         @@
        @
        @
        @
        @
        @
        @
        @
$$$$$$\\ @
\\______|@
        @
        @@
$$\\ @
\\$ |@
 \\_|@
    @
    @
    @
    @
    @
    @
    @
    @@
          @
          @
 $$$$$$\\  @
 \\____$$\\ @
 $$$$$$$ |@
$$  __$$ |@
\\$$$$$$$ |@
 \\_______|@
          @
          @
          @@
$$\\       @
$$ |      @
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$$$$$$  |@
\\_______/ @
          @
          @
          @@
          @
          @
 $$$$$$$\\ @
$$  _____|@
$$ /      @
$$ |      @
\\$$$$$$$\\ @
 \\_______|@
          @
          @
          @@
      $$\\ @
      $$ |@
 $$$$$$$ |@
$$  __$$ |@
$$ /  $$ |@
$$ |  $$ |@
\\$$$$$$$ |@
 \\_______|@
          @
          @
          @@
          @
          @
 $$$$$$\\  @
$$  __$$\\ @
$$$$$$$$ |@
$$   ____|@
\\$$$$$$$\\ @
 \\_______|@
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ /  \\__|@
$$$$\\     @
$$  _|    @
$$ |      @
$$ |      @
\\__|      @
          @
          @
          @@
          @
          @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
\\$$$$$$$ |@
 \\____$$ |@
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @@
$$\\       @
$$ |      @
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\__|  \\__|@
          @
          @
          @@
$$\\ @
\\__|@
$$\\ @
$$ |@
$$ |@
$$ |@
$$ |@
\\__|@
    @
    @
    @@
          @
          @
      $$\\ @
      \\__|@
      $$\\ @
      $$ |@
      $$ |@
      $$ |@
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @@
$$\\       @
$$ |      @
$$ |  $$\\ @
$$ | $$  |@
$$$$$$  / @
$$  _$$<  @
$$ | \\$$\\ @
\\__|  \\__|@
          @
          @
          @@
$$\\ @
$$ |@
$$ |@
$$ |@
$$ |@
$$ |@
$$ |@
\\__|@
    @
    @
    @@
              @
              @
$$$$$$\\$$$$\\  @
$$  _$$  _$$\\ @
$$ / $$ / $$ |@
$$ | $$ | $$ |@
$$ | $$ | $$ |@
\\__| \\__| \\__|@
              @
              @
              @@
          @
          @
$$$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\__|  \\__|@
          @
          @
          @@
          @
          @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
          @
          @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
$$$$$$$  |@
$$  ____/ @
$$ |      @
$$ |      @
\\__|      @@
          @
          @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
\\$$$$$$$ |@
 \\____$$ |@
      $$ |@
      $$ |@
      \\__|@@
          @
          @
 $$$$$$\\  @
$$  __$$\\ @
$$ |  \\__|@
$$ |      @
$$ |      @
\\__|      @
          @
          @
          @@
          @
          @
 $$$$$$$\\ @
$$  _____|@
\\$$$$$$\\  @
 \\____$$\\ @
$$$$$$$  |@
\\_______/ @
          @
          @
          @@
  $$\\     @
  $$ |    @
$$$$$$\\   @
\\_$$  _|  @
  $$ |    @
  $$ |$$\\ @
  \\$$$$  |@
   \\____/ @
          @
          @
          @@
          @
          @
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
           @
           @
$$\\    $$\\ @
\\$$\\  $$  |@
 \\$$\\$$  / @
  \\$$$  /  @
   \\$  /   @
    \\_/    @
           @
           @
           @@
              @
              @
$$\\  $$\\  $$\\ @
$$ | $$ | $$ |@
$$ | $$ | $$ |@
$$ | $$ | $$ |@
\\$$$$$\\$$$$  |@
 \\_____\\____/ @
              @
              @
              @@
          @
          @
$$\\   $$\\ @
\\$$\\ $$  |@
 \\$$$$  / @
 $$  $$<  @
$$  /\\$$\\ @
\\__/  \\__|@
          @
          @
          @@
          @
          @
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\$$$$$$$ |@
 \\____$$ |@
$$\\   $$ |@
\\$$$$$$  |@
 \\______/ @@
          @
          @
$$$$$$$$\\ @
\\____$$  |@
  $$$$ _/ @
 $$  _/   @
$$$$$$$$\\ @
\\________|@
          @
          @
          @@
  $$$\\ @
 $$  _|@
 $$ |  @
$$$ |  @
\\$$ |  @
 $$ |  @
 \\$$$\\ @
  \\___|@
       @
       @
       @@
$$\\ @
$$ |@
$$ |@
\\__|@
$$\\ @
$$ |@
$$ |@
\\__|@
    @
    @
    @@
$$$\\   @
\\_$$\\  @
  $$ | @
  $$$\\ @
  $$  |@
  $$ / @
$$$  | @
\\___/  @
       @
       @
       @@
 $$$\\ $$\\ @
$$ $$\\$$ |@
$$ \\$$$  |@
\\__|\\___/ @
          @
          @
          @
          @
          @
          @
          @@
  $\\ $\\   @
  \\_|\\_|  @
 $$$$$$\\  @
 \\____$$\\ @
 $$$$$$$ |@
$$  __$$ |@
\\$$$$$$$ |@
 \\_______|@
          @
          @
          @@
  $\\ $\\   @
  \\_|\\_|  @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
  $\\ $\\   @
  \\_|\\_|  @
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
  $\\ $\\   @
  \\_|\\_|  @
 $$$$$$\\  @
 \\____$$\\ @
 $$$$$$$ |@
$$  __$$ |@
\\$$$$$$$ |@
 \\_______|@
          @
          @
          @@
  $\\ $\\   @
  \\_|\\_|  @
 $$$$$$\\  @
$$  __$$\\ @
$$ /  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
  $\\ $\\   @
  \\_|\\_|  @
$$\\   $$\\ @
$$ |  $$ |@
$$ |  $$ |@
$$ |  $$ |@
\\$$$$$$  |@
 \\______/ @
          @
          @
          @@
 $$$$$$\\  @
$$  __$$\\ @
$$ |  $$ |@
$$ $$$$  |@
$$ \\__$$< @
$$ |  $$ |@
$$ $$$$  |@
$$ \\____/ @
$$ |      @
$$ |      @
\\__|      @@"})


(def drippy
  {:font-name      "Drippy"
   :example        [
"▓█████▄  ██▀███   ██▓ ██▓███   ██▓███  ▓██   ██▓"
"▒██▀ ██▌▓██ ▒ ██▒▓██▒▓██░  ██▒▓██░  ██▒ ▒██  ██▒"
"░██   █▌▓██ ░▄█ ▒▒██▒▓██░ ██▓▒▓██░ ██▓▒  ▒██ ██░"
"░▓█▄   ▌▒██▀▀█▄  ░██░▒██▄█▓▒ ▒▒██▄█▓▒ ▒  ░ ▐██▓░"
"░▒████▓ ░██▓ ▒██▒░██░▒██▒ ░  ░▒██▒ ░  ░  ░ ██▒▓░"
" ▒▒▓  ▒ ░ ▒▓ ░▒▓░░▓  ▒▓▒░ ░  ░▒▓▒░ ░  ░   ██▒▒▒"
" ░ ▒  ▒   ░▒ ░ ▒░ ▒ ░░▒ ░     ░▒ ░      ▓██ ░▒░"
" ░ ░  ░   ░░   ░  ▒ ░░░       ░░        ▒ ▒ ░░"
"   ░       ░      ░                     ░ ░"
" ░                                      ░ ░"
                    ]
   :author         "?"
   :desc           "Original font name: \"Bloody\""
   :figlet-metrics "flf2a$ 10 5 10 0 3 0"
   :figlet-string  "$ $@
$ $@
$ $@
$ $@
$ $@
$ $@
$ $@
$ $@
$ $@
$ $@@
▐██▌@
▐██▌@
▐██▌@
▓██▒@
▒▄▄ @
░▀▀▒@
░  ░@
   ░@
░   @
    @@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
$   $@
$   $@
$   $@
$   $@
$██▓$@
$▒▓▒$@
$░▒ $@
$░  $@
$ ░ $@
$ ░ $@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
 ▄▄▄      @
▒████▄    @
▒██  ▀█▄  @
░██▄▄▄▄██ @
 ▓█   ▓██▒@
 ▒▒   ▓▒█░@
  ▒   ▒▒ ░@
  ░   ▒   @
      ░  ░@
          @@
 ▄▄▄▄   @
▓█████▄ @
▒██▒ ▄██@
▒██░█▀  @
░▓█  ▀█▓@
░▒▓███▀▒@
▒░▒   ░ @
 ░    ░ @
 ░      @
      ░ @@
 ▄████▄  @
▒██▀ ▀█  @
▒▓█    ▄ @
▒▓▓▄ ▄██▒@
▒ ▓███▀ ░@
░ ░▒ ▒  ░@
  ░  ▒   @
░        @
░ ░      @
░        @@
▓█████▄ @
▒██▀ ██▌@
░██   █▌@
░▓█▄   ▌@
░▒████▓ @
 ▒▒▓  ▒ @
 ░ ▒  ▒ @
 ░ ░  ░ @
   ░    @
 ░      @@
▓█████ @
▓█   ▀ @
▒███   @
▒▓█  ▄ @
░▒████▒@
░░ ▒░ ░@
 ░ ░  ░@
   ░   @
   ░  ░@
       @@
  █████▒@
▓██   ▒ @
▒████ ░ @
░▓█▒  ░ @
░▒█░    @
 ▒ ░    @
 ░      @
 ░ ░    @
        @
        @@
  ▄████ @
 ██▒ ▀█▒@
▒██░▄▄▄░@
░▓█  ██▓@
░▒▓███▀▒@
 ░▒   ▒ @
  ░   ░ @
░ ░   ░ @
      ░ @
        @@
 ██░ ██ @
▓██░ ██▒@
▒██▀▀██░@
░▓█ ░██ @
░▓█▒░██▓@
 ▒ ░░▒░▒@
 ▒ ░▒░ ░@
 ░  ░░ ░@
 ░  ░  ░@
        @@
 ██▓@
▓██▒@
▒██▒@
░██░@
░██░@
░▓  @
 ▒ ░@
 ▒ ░@
 ░  @
    @@
 ▄▄▄██▀▀▀@
   ▒██   @
   ░██   @
▓██▄██▓  @
 ▓███▒   @
 ▒▓▒▒░   @
 ▒ ░▒░   @
 ░ ░ ░   @
 ░   ░   @
         @@
 ██ ▄█▀@
 ██▄█▒ @
▓███▄░ @
▓██ █▄ @
▒██▒ █▄@
▒ ▒▒ ▓▒@
░ ░▒ ▒░@
░ ░░ ░ @
░  ░   @
       @@
 ██▓    @
▓██▒    @
▒██░    @
▒██░    @
░██████▒@
░ ▒░▓  ░@
░ ░ ▒  ░@
  ░ ░   @
    ░  ░@
        @@
 ███▄ ▄███▓@
▓██▒▀█▀ ██▒@
▓██    ▓██░@
▒██    ▒██ @
▒██▒   ░██▒@
░ ▒░   ░  ░@
░  ░      ░@
░      ░   @
       ░   @
           @@
 ███▄    █ @
 ██ ▀█   █ @
▓██  ▀█ ██▒@
▓██▒  ▐▌██▒@
▒██░   ▓██░@
░ ▒░   ▒ ▒ @
░ ░░   ░ ▒░@
   ░   ░ ░ @
         ░ @
           @@
 ▒█████  @
▒██▒  ██▒@
▒██░  ██▒@
▒██   ██░@
░ ████▓▒░@
░ ▒░▒░▒░ @
  ░ ▒ ▒░ @
░ ░ ░ ▒  @
    ░ ░  @
         @@
 ██▓███  @
▓██░  ██▒@
▓██░ ██▓▒@
▒██▄█▓▒ ▒@
▒██▒ ░  ░@
▒▓▒░ ░  ░@
░▒ ░     @
░░       @
         @
         @@
  █████  @
▒██▓  ██▒@
▒██▒  ██░@
░██  █▀ ░@
░▒███▒█▄ @
░░ ▒▒░ ▒ @
 ░ ▒░  ░ @
   ░   ░ @
    ░    @
         @@
 ██▀███  @
▓██ ▒ ██▒@
▓██ ░▄█ ▒@
▒██▀▀█▄  @
░██▓ ▒██▒@
░ ▒▓ ░▒▓░@
  ░▒ ░ ▒░@
  ░░   ░ @
   ░     @
         @@
  ██████ @
▒██    ▒ @
░ ▓██▄   @
  ▒   ██▒@
▒██████▒▒@
▒ ▒▓▒ ▒ ░@
░ ░▒  ░ ░@
░  ░  ░  @
      ░  @
         @@
▄▄▄█████▓@
▓  ██▒ ▓▒@
▒ ▓██░ ▒░@
░ ▓██▓ ░ @
  ▒██▒ ░ @
  ▒ ░░   @
    ░    @
  ░      @
         @
         @@
 █    ██ @
 ██  ▓██▒@
▓██  ▒██░@
▓▓█  ░██░@
▒▒█████▓ @
░▒▓▒ ▒ ▒ @
░░▒░ ░ ░ @
 ░░░ ░ ░ @
   ░     @
         @@
 ██▒   █▓@
▓██░   █▒@
 ▓██  █▒░@
  ▒██ █░░@
   ▒▀█░  @
   ░ ▐░  @
   ░ ░░  @
     ░░  @
      ░  @
     ░   @@
 █     █░@
▓█░ █ ░█░@
▒█░ █ ░█ @
░█░ █ ░█ @
░░██▒██▓ @
░ ▓░▒ ▒  @
  ▒ ░ ░  @
  ░   ░  @
    ░    @
         @@
▒██   ██▒@
▒▒ █ █ ▒░@
░░  █   ░@
 ░ █ █ ▒ @
▒██▒ ▒██▒@
▒▒ ░ ░▓ ░@
░░   ░▒ ░@
 ░    ░  @
 ░    ░  @
         @@
▓██   ██▓@
 ▒██  ██▒@
  ▒██ ██░@
  ░ ▐██▓░@
  ░ ██▒▓░@
   ██▒▒▒ @
 ▓██ ░▒░ @
 ▒ ▒ ░░  @
 ░ ░     @
 ░ ░     @@
▒███████▒@
▒ ▒ ▒ ▄▀░@
░ ▒ ▄▀▒░ @
  ▄▀▒   ░@
▒███████▒@
░▒▒ ▓░▒░▒@
░░▒ ▒ ░ ▒@
░ ░ ░ ░ ░@
  ░ ░    @
░        @@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
 ▄▄▄      @
▒████▄    @
▒██  ▀█▄  @
░██▄▄▄▄██ @
 ▓█   ▓██▒@
 ▒▒   ▓▒█░@
  ▒   ▒▒ ░@
  ░   ▒   @
      ░  ░@
          @@
 ▄▄▄▄   @
▓█████▄ @
▒██▒ ▄██@
▒██░█▀  @
░▓█  ▀█▓@
░▒▓███▀▒@
▒░▒   ░ @
 ░    ░ @
 ░      @
      ░ @@
 ▄████▄  @
▒██▀ ▀█  @
▒▓█    ▄ @
▒▓▓▄ ▄██▒@
▒ ▓███▀ ░@
░ ░▒ ▒  ░@
  ░  ▒   @
░        @
░ ░      @
░        @@
▓█████▄ @
▒██▀ ██▌@
░██   █▌@
░▓█▄   ▌@
░▒████▓ @
 ▒▒▓  ▒ @
 ░ ▒  ▒ @
 ░ ░  ░ @
   ░    @
 ░      @@
▓█████ @
▓█   ▀ @
▒███   @
▒▓█  ▄ @
░▒████▒@
░░ ▒░ ░@
 ░ ░  ░@
   ░   @
   ░  ░@
       @@
  █████▒@
▓██   ▒ @
▒████ ░ @
░▓█▒  ░ @
░▒█░    @
 ▒ ░    @
 ░      @
 ░ ░    @
        @
        @@
  ▄████ @
 ██▒ ▀█▒@
▒██░▄▄▄░@
░▓█  ██▓@
░▒▓███▀▒@
 ░▒   ▒ @
  ░   ░ @
░ ░   ░ @
      ░ @
        @@
 ██░ ██ @
▓██░ ██▒@
▒██▀▀██░@
░▓█ ░██ @
░▓█▒░██▓@
 ▒ ░░▒░▒@
 ▒ ░▒░ ░@
 ░  ░░ ░@
 ░  ░  ░@
        @@
 ██▓@
▓██▒@
▒██▒@
░██░@
░██░@
░▓  @
 ▒ ░@
 ▒ ░@
 ░  @
    @@
 ▄▄▄██▀▀▀@
   ▒██   @
   ░██   @
▓██▄██▓  @
 ▓███▒   @
 ▒▓▒▒░   @
 ▒ ░▒░   @
 ░ ░ ░   @
 ░   ░   @
         @@
 ██ ▄█▀@
 ██▄█▒ @
▓███▄░ @
▓██ █▄ @
▒██▒ █▄@
▒ ▒▒ ▓▒@
░ ░▒ ▒░@
░ ░░ ░ @
░  ░   @
       @@
 ██▓    @
▓██▒    @
▒██░    @
▒██░    @
░██████▒@
░ ▒░▓  ░@
░ ░ ▒  ░@
  ░ ░   @
    ░  ░@
        @@
 ███▄ ▄███▓@
▓██▒▀█▀ ██▒@
▓██    ▓██░@
▒██    ▒██ @
▒██▒   ░██▒@
░ ▒░   ░  ░@
░  ░      ░@
░      ░   @
       ░   @
           @@
 ███▄    █ @
 ██ ▀█   █ @
▓██  ▀█ ██▒@
▓██▒  ▐▌██▒@
▒██░   ▓██░@
░ ▒░   ▒ ▒ @
░ ░░   ░ ▒░@
   ░   ░ ░ @
         ░ @
           @@
 ▒█████  @
▒██▒  ██▒@
▒██░  ██▒@
▒██   ██░@
░ ████▓▒░@
░ ▒░▒░▒░ @
  ░ ▒ ▒░ @
░ ░ ░ ▒  @
    ░ ░  @
         @@
 ██▓███  @
▓██░  ██▒@
▓██░ ██▓▒@
▒██▄█▓▒ ▒@
▒██▒ ░  ░@
▒▓▒░ ░  ░@
░▒ ░     @
░░       @
         @
         @@
  █████  @
▒██▓  ██▒@
▒██▒  ██░@
░██  █▀ ░@
░▒███▒█▄ @
░░ ▒▒░ ▒ @
 ░ ▒░  ░ @
   ░   ░ @
    ░    @
         @@
 ██▀███  @
▓██ ▒ ██▒@
▓██ ░▄█ ▒@
▒██▀▀█▄  @
░██▓ ▒██▒@
░ ▒▓ ░▒▓░@
  ░▒ ░ ▒░@
  ░░   ░ @
   ░     @
         @@
  ██████ @
▒██    ▒ @
░ ▓██▄   @
  ▒   ██▒@
▒██████▒▒@
▒ ▒▓▒ ▒ ░@
░ ░▒  ░ ░@
░  ░  ░  @
      ░  @
         @@
▄▄▄█████▓@
▓  ██▒ ▓▒@
▒ ▓██░ ▒░@
░ ▓██▓ ░ @
  ▒██▒ ░ @
  ▒ ░░   @
    ░    @
  ░      @
         @
         @@
 █    ██ @
 ██  ▓██▒@
▓██  ▒██░@
▓▓█  ░██░@
▒▒█████▓ @
░▒▓▒ ▒ ▒ @
░░▒░ ░ ░ @
 ░░░ ░ ░ @
   ░     @
         @@
 ██▒   █▓@
▓██░   █▒@
 ▓██  █▒░@
  ▒██ █░░@
   ▒▀█░  @
   ░ ▐░  @
   ░ ░░  @
     ░░  @
      ░  @
     ░   @@
 █     █░@
▓█░ █ ░█░@
▒█░ █ ░█ @
░█░ █ ░█ @
░░██▒██▓ @
░ ▓░▒ ▒  @
  ▒ ░ ░  @
  ░   ░  @
    ░    @
         @@
▒██   ██▒@
▒▒ █ █ ▒░@
░░  █   ░@
 ░ █ █ ▒ @
▒██▒ ▒██▒@
▒▒ ░ ░▓ ░@
░░   ░▒ ░@
 ░    ░  @
 ░    ░  @
         @@
▓██   ██▓@
 ▒██  ██▒@
  ▒██ ██░@
  ░ ▐██▓░@
  ░ ██▒▓░@
   ██▒▒▒ @
 ▓██ ░▒░ @
 ▒ ▒ ░░  @
 ░ ░     @
 ░ ░     @@
▒███████▒@
▒ ▒ ▒ ▄▀░@
░ ▒ ▄▀▒░ @
  ▄▀▒   ░@
▒███████▒@
░▒▒ ▓░▒░▒@
░░▒ ▒ ░ ▒@
░ ░ ░ ░ ░@
  ░ ░    @
░        @@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@
@
@
@
@
@
@
@
@
@
@@"})


(def big
 {
  :font-name      "Big"
  :example        [
                   " ____   _"
                   "|  _ \\ (_)"
                   "| |_) | _   __ _"
                   "|  _ < | | / _` |"
                   "| |_) || || (_| |"
                   "|____/ |_| \\__, |"
                   "            __/ |"
                   "           |___/"
                   ]
  :author         "Glenn Chappell, Bruce Jakeway"
  :desc           
  "Big by Glenn Chappell 4/93 -- based on Standard
Includes ISO Latin-1
Greek characters by Bruce Jakeway <pbjakeway@neumann.uwaterloo.ca>
figlet release 2.2 -- November 1996
Permission is hereby given to modify this font, as long as the
modifier's name is placed on a comment line.

Modified by Paul Burton <solution@earthlink.net> 12/96 to include new parameter
supported by FIGlet and FIGWin.  May also be slightly modified for better use
of new full-width/kern/smush alternatives, but default output is NOT changed."
  :figlet-metrics "flf2a$ 8 6 59 15 10 0 24463"
  :figlet-string  
"$@
 $@
 $@
 $@
 $@
 $@
 $@
 $@@
 _ @
| |@
| |@
| |@
|_|@
(_)@
   @
   @@
 _ _ @
( | )@
 V V @
     @
     @
     @
     @
     @@
   _  _   @
 _| || |_ @
|_  __  _|@
 _| || |_ @
|_  __  _|@
  |_||_|  @
          @
          @@
  _  @
 | | @
/ __)@
\\__ \\@
(   /@
 |_| @
     @
     @@
 _   __@
(_) / /@
   / / @
  / /  @
 / / _ @
/_/ (_)@
       @
       @@
        @
  ___   @
 ( _ )  @
 / _ \\/\\@
| (_>  <@
 \\___/\\/@
        @
        @@
 _ @
( )@
|/ @
   @
   @
   @
   @
   @@
  __@
 / /@
| | @
| | @
| | @
| | @
 \\_\\@
    @@
__  @
\\ \\ @
 | |@
 | |@
 | |@
 | |@
/_/ @
    @@
    _    @
 /\\| |/\\ @
 \\ ` ' / @
|_     _|@
 / , . \\ @
 \\/|_|\\/ @
         @
         @@
       @
   _   @
 _| |_ @
|_   _|@
  |_|  @
       @
       @
       @@
   @
   @
   @
   @
 _ @
( )@
|/ @
   @@
        @
        @
 ______ @
|______|@
        @
        @
        @
        @@
   @
   @
   @
   @
 _ @
(_)@
   @
   @@
     __@
    / /@
   / / @
  / /  @
 / /   @
/_/    @
       @
       @@
  ___  @
 / _ \\ @
| | | |@
| | | |@
| |_| |@
 \\___/ @
       @
       @@
 __ @
/_ |@
 | |@
 | |@
 | |@
 |_|@
    @
    @@
 ___  @
|__ \\ @
   ) |@
  / / @
 / /_ @
|____|@
      @
      @@
 ____  @
|___ \\ @
  __) |@
 |__ < @
 ___) |@
|____/ @
       @
       @@
 _  _   @
| || |  @
| || |_ @
|__   _|@
   | |  @
   |_|  @
        @
        @@
_____ @
| ____|@
| |__  @
|___ \\ @
 ___) |@
|____/ @
       @
       @@
   __  @
  / /  @
 / /_  @
| '_ \\ @
| (_) |@
 \\___/ @
       @
       @@
 ______ @
|____  |@
    / / @
   / /  @
  / /   @
 /_/    @
        @
        @@
  ___  @
 / _ \\ @
| (_) |@
 > _ < @
| (_) |@
 \\___/ @
       @
       @@
  ___  @
 / _ \\ @
| (_) |@
 \\__, |@
   / / @
  /_/  @
       @
       @@
   @
 _ @
(_)@
   @
 _ @
(_)@
   @
   @@
   @
 _ @
(_)@
   @
 _ @
( )@
|/ @
   @@
   __@
  / /@
 / / @
< <  @
 \\ \\ @
  \\_\\@
     @
     @@
        @
 ______ @
|______|@
 ______ @
|______|@
        @
        @
        @@
__   @
\\ \\  @
 \\ \\ @
  > >@
 / / @
/_/  @
     @
     @@
 ___  @
|__ \\ @
   ) |@
  / / @
 |_|  @
 (_)  @
      @
      @@
         @
   ____  @
  / __ \\ @
 / / _` |@
| | (_| |@
 \\ \\__,_|@
  \\____/ @
         @@
          @
    /\\    @
   /  \\   @
  / /\\ \\  @
 / ____ \\ @
/_/    \\_\\@
          @
          @@
 ____  @
|  _ \\ @
| |_) |@
|  _ < @
| |_) |@
|____/ @
       @
       @@
  _____ @
 / ____|@
| |     @
| |     @
| |____ @
 \\_____|@
        @
        @@
 _____  @
|  __ \\ @
| |  | |@
| |  | |@
| |__| |@
|_____/ @
        @
        @@
 ______ @
|  ____|@
| |__   @
|  __|  @
| |____ @
|______|@
        @
        @@
 ______ @
|  ____|@
| |__   @
|  __|  @
| |     @
|_|     @
        @
        @@
  _____ @
 / ____|@
| |  __ @
| | |_ |@
| |__| |@
 \\_____|@
        @
        @@
 _    _ @
| |  | |@
| |__| |@
|  __  |@
| |  | |@
|_|  |_|@
        @
        @@
 _____ @
|_   _|@
  | |  @
  | |  @
 _| |_ @
|_____|@
       @
       @@
      _ @
     | |@
     | |@
 _   | |@
| |__| |@
 \\____/ @
        @
        @@
 _  __@
| |/ /@
| ' / @
|  <  @
| . \\ @
|_|\\_\\@
      @
      @@
 _      @
| |     @
| |     @
| |     @
| |____ @
|______|@
        @
        @@
 __  __ @
|  \\/  |@
| \\  / |@
| |\\/| |@
| |  | |@
|_|  |_|@
        @
        @@
 _   _ @
| \\ | |@
|  \\| |@
| . ` |@
| |\\  |@
|_| \\_|@
       @
       @@
  ____  @
 / __ \\ @
| |  | |@
| |  | |@
| |__| |@
 \\____/ @
        @
        @@
 _____  @
|  __ \\ @
| |__) |@
|  ___/ @
| |     @
|_|     @
        @
        @@
  ____  @
 / __ \\ @
| |  | |@
| |  | |@
| |__| |@
 \\___\\_\\@
        @
        @@
 _____  @
|  __ \\ @
| |__) |@
|  _  / @
| | \\ \\ @
|_|  \\_\\@
        @
        @@
  _____ @
 / ____|@
| (___  @
 \\___ \\ @
 ____) |@
|_____/ @
        @
        @@
 _______ @
|__   __|@
   | |   @
   | |   @
   | |   @
   |_|   @
         @
         @@
 _    _ @
| |  | |@
| |  | |@
| |  | |@
| |__| |@
 \\____/ @
        @
        @@
__      __@
\\ \\    / /@
 \\ \\  / / @
  \\ \\/ /  @
   \\  /   @
    \\/    @
          @
          @@
__          __@
\\ \\        / /@
 \\ \\  /\\  / / @
  \\ \\/  \\/ /  @
   \\  /\\  /   @
    \\/  \\/    @
              @
              @@
__   __@
\\ \\ / /@
 \\ V / @
  > <  @
 / . \\ @
/_/ \\_\\@
       @
       @@
__     __@
\\ \\   / /@
 \\ \\_/ / @
  \\   /  @
   | |   @
   |_|   @
         @
         @@
 ______@
|___  /@
   / / @
  / /  @
 / /__ @
/_____|@
       @
       @@
 ___ @
|  _|@
| |  @
| |  @
| |  @
| |_ @
|___|@
     @@
__     @
\\ \\    @
 \\ \\   @
  \\ \\  @
   \\ \\ @
    \\_\\@
       @
       @@
 ___ @
|_  |@
  | |@
  | |@
  | |@
 _| |@
|___|@
     @@
 /\\ @
|/\\|@
    @
    @
    @
    @
    @
    @@
        @
        @
        @
        @
        @
        @
 ______ @
|______|@@
 _ @
( )@
 \\|@
   @
   @
   @
   @
   @@
       @
       @
  __ _ @
 / _` |@
| (_| |@
 \\__,_|@
       @
       @@
 _     @
| |    @
| |__  @
| '_ \\ @
| |_) |@
|_.__/ @
       @
       @@
      @
      @
  ___ @
 / __|@
| (__ @
 \\___|@
      @
      @@
     _ @
    | |@
  __| |@
 / _` |@
| (_| |@
 \\__,_|@
       @
       @@
      @
      @
  ___ @
 / _ \\@
|  __/@
 \\___|@
      @
      @@
  __ @
 / _|@
| |_ @
|  _|@
| |  @
|_|  @
     @
     @@
       @
       @
  __ _ @
 / _` |@
| (_| |@
 \\__, |@
  __/ |@
 |___/ @@
 _     @
| |    @
| |__  @
| '_ \\ @
| | | |@
|_| |_|@
       @
       @@
 _ @
(_)@
 _ @
| |@
| |@
|_|@
   @
   @@
   _ @
  (_)@
   _ @
  | |@
  | |@
  | |@
 _/ |@
|__/ @@
 _    @
| |   @
| | __@
| |/ /@
|   < @
|_|\\_\\@
      @
      @@
 _ @
| |@
| |@
| |@
| |@
|_|@
   @
   @@
           @
           @
 _ __ ___  @
| '_ ` _ \\ @
| | | | | |@
|_| |_| |_|@
           @
           @@
       @
       @
 _ __  @
| '_ \\ @
| | | |@
|_| |_|@
       @
       @@
       @
       @
  ___  @
 / _ \\ @
| (_) |@
 \\___/ @
       @
       @@
       @
       @
 _ __  @
| '_ \\ @
| |_) |@
| .__/ @
| |    @
|_|    @@
       @
       @
  __ _ @
 / _` |@
| (_| |@
 \\__, |@
    | |@
    |_|@@
      @
      @
 _ __ @
| '__|@
| |   @
|_|   @
      @
      @@
     @
     @
 ___ @
/ __|@
\\__ \\@
|___/@
     @
     @@
 _   @
| |  @
| |_ @
| __|@
| |_ @
 \\__|@
     @
     @@
       @
       @
 _   _ @
| | | |@
| |_| |@
 \\__,_|@
       @
       @@
       @
       @
__   __@
\\ \\ / /@
 \\ V / @
  \\_/  @
       @
       @@
          @
          @
__      __@
\\ \\ /\\ / /@
 \\ V  V / @
  \\_/\\_/  @
          @
          @@
      @
      @
__  __@
\\ \\/ /@
 >  < @
/_/\\_\\@
      @
      @@
       @
       @
 _   _ @
| | | |@
| |_| |@
 \\__, |@
  __/ |@
 |___/ @@
     @
     @
 ____@
|_  /@
 / / @
/___|@
     @
     @@
   __@
  / /@
 | | @
/ /  @
\\ \\  @
 | | @
  \\_\\@
     @@
 _ @
| |@
| |@
| |@
| |@
| |@
| |@
|_|@@
__   @
\\ \\  @
 | | @
  \\ \\@
  / /@
 | | @
/_/  @
     @@
 /\\/|@
|/\\/ @
     @
     @
     @
     @
     @
     @@
  _   _  @
 (_)_(_) @
   / \\   @
  / _ \\  @
 / ___ \\ @
/_/   \\_\\@
         @
         @@
 _   _ @
(_)_(_)@
 / _ \\ @
| | | |@
| |_| |@
 \\___/ @
       @
       @@
 _   _ @
(_) (_)@
| | | |@
| | | |@
| |_| |@
 \\___/ @
       @
       @@
 _   _ @
(_) (_)@
  __ _ @
 / _` |@
| (_| |@
 \\__,_|@
       @
       @@
 _   _ @
(_) (_)@
  ___  @
 / _ \\ @
| (_) |@
 \\___/ @
       @
       @@
 _   _ @
(_) (_)@
 _   _ @
| | | |@
| |_| |@
 \\__,_|@
       @
       @@
  ___  @
 / _ \\ @
| | ) |@
| |< < @
| | ) |@
| ||_/ @
|_|    @
       @@"

;; 160  NO-BREAK SPACE
;;  $@
;;  $@
;;  $@
;;  $@
;;  $@
;;  $@
;;  $@
;;  $@@
;; 161  INVERTED EXCLAMATION MARK
;;   _ @
;;  (_)@
;;  | |@
;;  | |@
;;  | |@
;;  |_|@
;;     @
;;     @@
;; 162  CENT SIGN
;;        @
;;     _  @
;;    | | @
;;   / __)@
;;  | (__ @
;;   \   )@
;;    |_| @
;;        @@
;; 163  POUND SIGN
;;      ___   @
;;     / ,_\  @
;;   _| |_    @
;;  |__ __|   @
;;    | |____ @
;;   (_,_____|@
;;            @
;;            @@
;; 164  CURRENCY SIGN
;;         @
;;  /\___/\@
;;  \  _  /@
;;  | (_) |@
;;  / ___ \@
;;  \/   \/@
;;         @
;;         @@
;; 165  YEN SIGN
;;   __   __ @
;;   \ \ / / @
;;   _\ V /_ @
;;  |___ ___|@
;;  |___ ___|@
;;     |_|   @
;;           @
;;           @@
;; 166  BROKEN BAR
;;   _ @
;;  | |@
;;  | |@
;;  |_|@
;;   _ @
;;  | |@
;;  | |@
;;  |_|@@
;; 167  SECTION SIGN
;;     __ @
;;   _/ _)@
;;  / \ \ @
;;  \ \\ \@
;;   \ \_/@
;;  (__/  @
;;        @
;;        @@
;; 168  DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;   $   $ @
;;   $   $ @
;;   $   $ @
;;   $   $ @
;;         @
;;         @@
;; 169  COPYRIGHT SIGN
;;     ________   @
;;    /  ____  \  @
;;   /  / ___|  \ @
;;  |  | |       |@
;;  |  | |___    |@
;;   \  \____|  / @
;;    \________/  @
;;                @@
;; 170  FEMININE ORDINAL INDICATOR
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;  |_____|@
;;     $   @
;;         @
;;         @@
;; 171  LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
;;     ____@
;;    / / /@
;;   / / / @
;;  < < <  @
;;   \ \ \ @
;;    \_\_\@
;;         @
;;         @@
;; 172  NOT SIGN
;;          @
;;          @
;;   ______ @
;;  |____  |@
;;       |_|@
;;      $   @
;;          @
;;          @@
;; 173  SOFT HYPHEN
;;         @
;;         @
;;   _____ @
;;  |_____|@
;;     $   @
;;     $   @
;;         @
;;         @@
;; 174  REGISTERED SIGN
;;     ________   @
;;    /  ____  \  @
;;   /  |  _ \  \ @
;;  |   | |_) |  |@
;;  |   |  _ <   |@
;;   \  |_| \_\ / @
;;    \________/  @
;;                @@
;; 175  MACRON
;;   ______ @
;;  |______|@
;;      $   @
;;      $   @
;;      $   @
;;      $   @
;;          @
;;          @@
;; 176  DEGREE SIGN
;;    __  @
;;   /  \ @
;;  | () |@
;;   \__/ @
;;     $  @
;;     $  @
;;        @
;;        @@
;; 177  PLUS-MINUS SIGN
;;     _   @
;;   _| |_ @
;;  |_   _|@
;;    |_|  @
;;   _____ @
;;  |_____|@
;;         @
;;         @@
;; 178  SUPERSCRIPT TWO
;;   ___ @
;;  |_  )@
;;   / / @
;;  /___|@
;;    $  @
;;    $  @
;;       @
;;       @@
;; 179  SUPERSCRIPT THREE
;;   ____@
;;  |__ /@
;;   |_ \@
;;  |___/@
;;    $  @
;;    $  @
;;       @
;;       @@
;; 180  ACUTE ACCENT
;;   __@
;;  /_/@
;;   $ @
;;   $ @
;;   $ @
;;   $ @
;;     @
;;     @@
;; 181  MICRO SIGN
;;         @
;;         @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;  | ._,_|@
;;  | |    @
;;  |_|    @@
;; 182  PILCROW SIGN
;;    ______ @
;;   /      |@
;;  | (| || |@
;;   \__ || |@
;;     | || |@
;;     |_||_|@
;;           @
;;           @@
;; 183  MIDDLE DOT
;;     @
;;     @
;;   _ @
;;  (_)@
;;   $ @
;;   $ @
;;     @
;;     @@
;; 184  CEDILLA
;;     @
;;     @
;;     @
;;     @
;;     @
;;   _ @
;;  )_)@
;;     @@
;; 185  SUPERSCRIPT ONE
;;   _ @
;;  / |@
;;  | |@
;;  |_|@
;;   $ @
;;   $ @
;;     @
;;     @@
;; 186  MASCULINE ORDINAL INDICATOR
;;    ___  @
;;   / _ \ @
;;  | (_) |@
;;   \___/ @
;;  |_____|@
;;     $   @
;;         @
;;         @@
;; 187  RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
;;  ____   @
;;  \ \ \  @
;;   \ \ \ @
;;    > > >@
;;   / / / @
;;  /_/_/  @
;;         @
;;         @@
;; 188  VULGAR FRACTION ONE QUARTER
;;   _   __   @
;;  / | / /   @
;;  | |/ / _  @
;;  |_/ / | | @
;;   / /|_  _|@
;;  /_/   |_| @
;;            @
;;            @@
;; 189  VULGAR FRACTION ONE HALF
;;   _   __  @
;;  / | / /  @
;;  | |/ /__ @
;;  |_/ /_  )@
;;   / / / / @
;;  /_/ /___|@
;;           @
;;           @@
;; 190  VULGAR FRACTION THREE QUARTERS
;;   ____  __   @
;;  |__ / / /   @
;;   |_ \/ / _  @
;;  |___/ / | | @
;;     / /|_  _|@
;;    /_/   |_| @
;;              @
;;              @@
;; 191  INVERTED QUESTION MARK
;;     _  @
;;    (_) @
;;    | | @
;;   / /  @
;;  | (__ @
;;   \___|@
;;        @
;;        @@
;; 192  LATIN CAPITAL LETTER A WITH GRAVE
;;     __    @
;;     \_\   @
;;     / \   @
;;    / _ \  @
;;   / ___ \ @
;;  /_/   \_\@
;;           @
;;           @@
;; 193  LATIN CAPITAL LETTER A WITH ACUTE
;;      __   @
;;     /_/   @
;;     / \   @
;;    / _ \  @
;;   / ___ \ @
;;  /_/   \_\@
;;           @
;;           @@
;; 194  LATIN CAPITAL LETTER A WITH CIRCUMFLEX
;;     //\   @
;;    |/_\|  @
;;     / \   @
;;    / _ \  @
;;   / ___ \ @
;;  /_/   \_\@
;;           @
;;           @@
;; 195  LATIN CAPITAL LETTER A WITH TILDE
;;     /\/|  @
;;    |/\/   @
;;     / \   @
;;    / _ \  @
;;   / ___ \ @
;;  /_/   \_\@
;;           @
;;           @@
;; 196  LATIN CAPITAL LETTER A WITH DIAERESIS
;;    _   _  @
;;   (_)_(_) @
;;     / \   @
;;    / _ \  @
;;   / ___ \ @
;;  /_/   \_\@
;;           @
;;           @@
;; 197  LATIN CAPITAL LETTER A WITH RING ABOVE
;;      _    @
;;     (o)   @
;;     / \   @
;;    / _ \  @
;;   / ___ \ @
;;  /_/   \_\@
;;           @
;;           @@
;; 198  LATIN CAPITAL LETTER AE
;;       _______ @
;;      /   ____|@
;;     /   |__   @
;;    / /|  __|  @
;;   / ___ |____ @
;;  /_/  |______|@
;;               @
;;               @@
;; 199  LATIN CAPITAL LETTER C WITH CEDILLA
;;    _____ @
;;   / ____|@
;;  | | $   @
;;  | | $   @
;;  | |____ @
;;   \_____|@
;;     )_)  @
;;          @@
;; 200  LATIN CAPITAL LETTER E WITH GRAVE
;;    __   @
;;   _\_\_ @
;;  | ____|@
;;  |  _|  @
;;  | |___ @
;;  |_____|@
;;         @
;;         @@
;; 201  LATIN CAPITAL LETTER E WITH ACUTE
;;     __  @
;;   _/_/_ @
;;  | ____|@
;;  |  _|  @
;;  | |___ @
;;  |_____|@
;;         @
;;         @@
;; 202  LATIN CAPITAL LETTER E WITH CIRCUMFLEX
;;    //\  @
;;   |/ \| @
;;  | ____|@
;;  |  _|  @
;;  | |___ @
;;  |_____|@
;;         @
;;         @@
;; 203  LATIN CAPITAL LETTER E WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;  | ____|@
;;  |  _|  @
;;  | |___ @
;;  |_____|@
;;         @
;;         @@
;; 204  LATIN CAPITAL LETTER I WITH GRAVE
;;   __  @
;;   \_\ @
;;  |_ _|@
;;   | | @
;;   | | @
;;  |___|@
;;       @
;;       @@
;; 205  LATIN CAPITAL LETTER I WITH ACUTE
;;    __ @
;;   /_/ @
;;  |_ _|@
;;   | | @
;;   | | @
;;  |___|@
;;       @
;;       @@
;; 206  LATIN CAPITAL LETTER I WITH CIRCUMFLEX
;;   //\ @
;;  |/_\|@
;;  |_ _|@
;;   | | @
;;   | | @
;;  |___|@
;;       @
;;       @@
;; 207  LATIN CAPITAL LETTER I WITH DIAERESIS
;;   _   _ @
;;  (_)_(_)@
;;   |_ _| @
;;    | |  @
;;    | |  @
;;   |___| @
;;         @
;;         @@
;; 208  LATIN CAPITAL LETTER ETH
;;     _____  @
;;    |  __ \ @
;;   _| |_ | |@
;;  |__ __|| |@
;;    | |__| |@
;;    |_____/ @
;;            @
;;            @@
;; 209  LATIN CAPITAL LETTER N WITH TILDE
;;    /\/| @
;;   |/\/_ @
;;  | \ | |@
;;  |  \| |@
;;  | |\  |@
;;  |_| \_|@
;;         @
;;         @@
;; 210  LATIN CAPITAL LETTER O WITH GRAVE
;;    __   @
;;    \_\  @
;;   / _ \ @
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 211  LATIN CAPITAL LETTER O WITH ACUTE
;;     __  @
;;    /_/  @
;;   / _ \ @
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 212  LATIN CAPITAL LETTER O WITH CIRCUMFLEX
;;    //\  @
;;   |/_\| @
;;   / _ \ @
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 213  LATIN CAPITAL LETTER O WITH TILDE
;;    /\/| @
;;   |/\/  @
;;   / _ \ @
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 214  LATIN CAPITAL LETTER O WITH DIAERESIS
;;   _   _ @
;;  (_)_(_)@
;;   / _ \ @
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 215  MULTIPLICATION SIGN
;;      @
;;      @
;;  /\/\@
;;  >  <@
;;  \/\/@
;;    $ @
;;      @
;;      @@
;; 216  LATIN CAPITAL LETTER O WITH STROKE
;;    _____ @
;;   / __// @
;;  | | // |@
;;  | |//| |@
;;  | //_| |@
;;   //___/ @
;;          @
;;          @@
;; 217  LATIN CAPITAL LETTER U WITH GRAVE
;;    __   @
;;   _\_\_ @
;;  | | | |@
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 218  LATIN CAPITAL LETTER U WITH ACUTE
;;     __  @
;;   _/_/_ @
;;  | | | |@
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 219  LATIN CAPITAL LETTER U WITH CIRCUMFLEX
;;    //\  @
;;   |/ \| @
;;  | | | |@
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 220  LATIN CAPITAL LETTER U WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;  | | | |@
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 221  LATIN CAPITAL LETTER Y WITH ACUTE
;;     __  @
;;  __/_/__@
;;  \ \ / /@
;;   \ V / @
;;    | |  @
;;    |_|  @
;;         @
;;         @@
;; 222  LATIN CAPITAL LETTER THORN
;;   _      @
;;  | |___  @
;;  |  __ \ @
;;  | |__) |@
;;  |  ___/ @
;;  |_|     @
;;          @
;;          @@
;; 223  LATIN SMALL LETTER SHARP S
;;    ___  @
;;   / _ \ @
;;  | | ) |@
;;  | |< < @
;;  | | ) |@
;;  | ||_/ @
;;  |_|    @
;;         @@
;; 224  LATIN SMALL LETTER A WITH GRAVE
;;    __   @
;;    \_\  @
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;         @
;;         @@
;; 225  LATIN SMALL LETTER A WITH ACUTE
;;     __  @
;;    /_/  @
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;         @
;;         @@
;; 226  LATIN SMALL LETTER A WITH CIRCUMFLEX
;;    //\  @
;;   |/ \| @
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;         @
;;         @@
;; 227  LATIN SMALL LETTER A WITH TILDE
;;    /\/| @
;;   |/\/  @
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;         @
;;         @@
;; 228  LATIN SMALL LETTER A WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;         @
;;         @@
;; 229  LATIN SMALL LETTER A WITH RING ABOVE
;;     __  @
;;    (()) @
;;    __ _ @
;;   / _` |@
;;  | (_| |@
;;   \__,_|@
;;         @
;;         @@
;; 230  LATIN SMALL LETTER AE
;;            @
;;            @
;;    __ ____ @
;;   / _`  _ \@
;;  | (_|  __/@
;;   \__,____|@
;;            @
;;            @@
;; 231  LATIN SMALL LETTER C WITH CEDILLA
;;        @
;;        @
;;    ___ @
;;   / __|@
;;  | (__ @
;;   \___|@
;;    )_) @
;;        @@
;; 232  LATIN SMALL LETTER E WITH GRAVE
;;    __  @
;;    \_\ @
;;    ___ @
;;   / _ \@
;;  |  __/@
;;   \___|@
;;        @
;;        @@
;; 233  LATIN SMALL LETTER E WITH ACUTE
;;     __ @
;;    /_/ @
;;    ___ @
;;   / _ \@
;;  |  __/@
;;   \___|@
;;        @
;;        @@
;; 234  LATIN SMALL LETTER E WITH CIRCUMFLEX
;;    //\ @
;;   |/ \|@
;;    ___ @
;;   / _ \@
;;  |  __/@
;;   \___|@
;;        @
;;        @@
;; 235  LATIN SMALL LETTER E WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;    ___  @
;;   / _ \ @
;;  |  __/ @
;;   \___| @
;;         @
;;         @@
;; 236  LATIN SMALL LETTER I WITH GRAVE
;;  __ @
;;  \_\@
;;   _ @
;;  | |@
;;  | |@
;;  |_|@
;;     @
;;     @@
;; 237  LATIN SMALL LETTER I WITH ACUTE
;;   __@
;;  /_/@
;;   _ @
;;  | |@
;;  | |@
;;  |_|@
;;     @
;;     @@
;; 238  LATIN SMALL LETTER I WITH CIRCUMFLEX
;;   //\ @
;;  |/ \|@
;;    _  @
;;   | | @
;;   | | @
;;   |_| @
;;       @
;;       @@
;; 239  LATIN SMALL LETTER I WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;     _   @
;;    | |  @
;;    | |  @
;;    |_|  @
;;         @
;;         @@
;; 240  LATIN SMALL LETTER ETH
;;   /\/\  @
;;   >  <  @
;;   \/\ \ @
;;   / _` |@
;;  | (_) |@
;;   \___/ @
;;         @
;;         @@
;; 241  LATIN SMALL LETTER N WITH TILDE
;;    /\/| @
;;   |/\/  @
;;   _ __  @
;;  | '_ \ @
;;  | | | |@
;;  |_| |_|@
;;         @
;;         @@
;; 242  LATIN SMALL LETTER O WITH GRAVE
;;    __   @
;;    \_\  @
;;    ___  @
;;   / _ \ @
;;  | (_) |@
;;   \___/ @
;;         @
;;         @@
;; 243  LATIN SMALL LETTER O WITH ACUTE
;;     __  @
;;    /_/  @
;;    ___  @
;;   / _ \ @
;;  | (_) |@
;;   \___/ @
;;         @
;;         @@
;; 244  LATIN SMALL LETTER O WITH CIRCUMFLEX
;;    //\  @
;;   |/ \| @
;;    ___  @
;;   / _ \ @
;;  | (_) |@
;;   \___/ @
;;         @
;;         @@
;; 245  LATIN SMALL LETTER O WITH TILDE
;;    /\/| @
;;   |/\/  @
;;    ___  @
;;   / _ \ @
;;  | (_) |@
;;   \___/ @
;;         @
;;         @@
;; 246  LATIN SMALL LETTER O WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;    ___  @
;;   / _ \ @
;;  | (_) |@
;;   \___/ @
;;         @
;;         @@
;; 247  DIVISION SIGN
;;      _    @
;;     (_)   @
;;   _______ @
;;  |_______|@
;;      _    @
;;     (_)   @
;;           @
;;           @@
;; 248  LATIN SMALL LETTER O WITH STROKE
;;          @
;;          @
;;    ____  @
;;   / _//\ @
;;  | (//) |@
;;   \//__/ @
;;          @
;;          @@
;; 249  LATIN SMALL LETTER U WITH GRAVE
;;    __   @
;;    \_\  @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \__,_|@
;;         @
;;         @@
;; 250  LATIN SMALL LETTER U WITH ACUTE
;;     __  @
;;    /_/  @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \__,_|@
;;         @
;;         @@
;; 251  LATIN SMALL LETTER U WITH CIRCUMFLEX
;;    //\  @
;;   |/ \| @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \__,_|@
;;         @
;;         @@
;; 252  LATIN SMALL LETTER U WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \__,_|@
;;         @
;;         @@
;; 253  LATIN SMALL LETTER Y WITH ACUTE
;;     __  @
;;    /_/  @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \__, |@
;;    __/ |@
;;   |___/ @@
;; 254  LATIN SMALL LETTER THORN
;;   _     @
;;  | |    @
;;  | |__  @
;;  | '_ \ @
;;  | |_) |@
;;  | .__/ @
;;  | |    @
;;  |_|    @@
;; 255  LATIN SMALL LETTER Y WITH DIAERESIS
;;   _   _ @
;;  (_) (_)@
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \__, |@
;;    __/ |@
;;   |___/ @@
;; 0x02BC  MODIFIER LETTER APOSTROPHE
;;    @
;;    @
;;  ))@
;;    @
;;    @
;;    @
;;    @
;;    @@
;; 0x02BD  MODIFIER LETTER REVERSED COMMA
;;    @
;;    @
;;  ((@
;;    @
;;    @
;;    @
;;    @
;;    @@
;; 0x037A  GREEK YPOGEGRAMMENI
;;    @
;;    @
;;    @
;;    @
;;    @
;;    @
;;    @
;;  ||@@
;; 0x0387  GREEK ANO TELEIA
;;     @
;;   $ @
;;   _ @
;;  (_)@
;;     @
;;   $ @
;;     @
;;     @@
;; 0x0391  GREEK CAPITAL LETTER ALPHA
;;    ___  @
;;   / _ \ @
;;  | |_| |@
;;  |  _  |@
;;  | | | |@
;;  |_| |_|@
;;         @
;;         @@
;; 0x0392  GREEK CAPITAL LETTER BETA
;;   ____  @
;;  |  _ \ @
;;  | |_) )@
;;  |  _ ( @
;;  | |_) )@
;;  |____/ @
;;         @
;;         @@
;; 0x0393  GREEK CAPITAL LETTER GAMMA
;;   _____ @
;;  |  ___)@
;;  | |$   @
;;  | |$   @
;;  | |    @
;;  |_|    @
;;         @
;;         @@
;; 0x0394  GREEK CAPITAL LETTER DELTA
;;            @
;;      /\    @
;;     /  \   @
;;    / /\ \  @
;;   / /__\ \ @
;;  /________\@
;;            @
;;            @@
;; 0x0395  GREEK CAPITAL LETTER EPSILON
;;   _____ @
;;  |  ___)@
;;  | |_   @
;;  |  _)  @
;;  | |___ @
;;  |_____)@
;;         @
;;         @@
;; 0x0396  GREEK CAPITAL LETTER ZETA
;;   ______@
;;  (___  /@
;;     / / @
;;    / /  @
;;   / /__ @
;;  /_____)@
;;         @
;;         @@
;; 0x0397  GREEK CAPITAL LETTER ETA
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;  |  _  |@
;;  | | | |@
;;  |_| |_|@
;;         @
;;         @@
;; 0x0398  GREEK CAPITAL LETTER THETA
;;    ____  @
;;   / __ \ @
;;  | |__| |@
;;  |  __  |@
;;  | |__| |@
;;   \____/ @
;;          @
;;          @@
;; 0x0399  GREEK CAPITAL LETTER IOTA
;;   ___ @
;;  (   )@
;;   | | @
;;   | | @
;;   | | @
;;  (___)@
;;       @
;;       @@
;; 0x039A  GREEK CAPITAL LETTER KAPPA
;;   _   __@
;;  | | / /@
;;  | |/ / @
;;  |   <  @
;;  | |\ \ @
;;  |_| \_\@
;;         @
;;         @@
;; 0x039B  GREEK CAPITAL LETTER LAMDA
;;            @
;;      /\    @
;;     /  \   @
;;    / /\ \  @
;;   / /  \ \ @
;;  /_/    \_\@
;;            @
;;            @@
;; 0x039C  GREEK CAPITAL LETTER MU
;;   __   __ @
;;  |  \ /  |@
;;  |   v   |@
;;  | |\_/| |@
;;  | |   | |@
;;  |_|   |_|@
;;           @
;;           @@
;; 0x039D  GREEK CAPITAL LETTER NU
;;   _   _ @
;;  | \ | |@
;;  |  \| |@
;;  |     |@
;;  | |\  |@
;;  |_| \_|@
;;         @
;;         @@
;; 0x039E  GREEK CAPITAL LETTER XI
;;   _____ @
;;  (_____)@
;;    ___  @
;;   (___) @
;;   _____ @
;;  (_____)@
;;         @
;;         @@
;; 0x039F  GREEK CAPITAL LETTER OMICRON
;;    ___  @
;;   / _ \ @
;;  | | | |@
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 0x03A0  GREEK CAPITAL LETTER PI
;;   _______ @
;;  (   _   )@
;;   | | | | @
;;   | | | | @
;;   | | | | @
;;   |_| |_| @
;;           @
;;           @@
;; 0x03A1  GREEK CAPITAL LETTER RHO
;;   ____  @
;;  |  _ \ @
;;  | |_) )@
;;  |  __/ @
;;  | |    @
;;  |_|    @
;;         @
;;         @@
;; 0x03A3  GREEK CAPITAL LETTER SIGMA
;;  ______ @
;;  \  ___)@
;;   \ \   @
;;    > >  @
;;   / /__ @
;;  /_____)@
;;         @
;;         @@
;; 0x03A4  GREEK CAPITAL LETTER TAU
;;   _____ @
;;  (_   _)@
;;    | |  @
;;    | |  @
;;    | |  @
;;    |_|  @
;;         @
;;         @@
;; 0x03A5  GREEK CAPITAL LETTER UPSILON
;;   __   __ @
;;  (_ \ / _)@
;;    \ v /  @
;;     | |   @
;;     | |   @
;;     |_|   @
;;           @
;;           @@
;; 0x03A6  GREEK CAPITAL LETTER PHI
;;      _    @
;;    _| |_  @
;;   /     \ @
;;  ( (| |) )@
;;   \_   _/ @
;;     |_|   @
;;           @
;;           @@
;; 0x03A7  GREEK CAPITAL LETTER CHI
;;  __   __@
;;  \ \ / /@
;;   \ v / @
;;    > <  @
;;   / ^ \ @
;;  /_/ \_\@
;;         @
;;         @@
;; 0x03A8  GREEK CAPITAL LETTER PSI
;;   _  _  _ @
;;  | || || |@
;;  | \| |/ |@
;;   \_   _/ @
;;     | |   @
;;     |_|   @
;;           @
;;           @@
;; 0x03A9  GREEK CAPITAL LETTER OMEGA
;;     ____   @
;;    / __ \  @
;;   | |  | | @
;;   | |  | | @
;;   _\ \/ /_ @
;;  (___||___)@
;;            @
;;            @@
;; 0x03B1  GREEK SMALL LETTER ALPHA
;;          @
;;          @
;;    __  __@
;;   /  \/ /@
;;  ( ()  < @
;;   \__/\_\@
;;          @
;;          @@
;; 0x03B2  GREEK SMALL LETTER BETA
;;    ___  @
;;   / _ \ @
;;  | |_) )@
;;  |  _ < @
;;  | |_) )@
;;  |  __/ @
;;  | |    @
;;  |_|    @@
;; 0x03B3  GREEK SMALL LETTER GAMMA
;;         @
;;         @
;;   _   _ @
;;  ( \ / )@
;;   \ v / @
;;    | |  @
;;    | |  @
;;    |_|  @@
;; 0x03B4  GREEK SMALL LETTER DELTA
;;     __  @
;;    / _) @
;;    \ \  @
;;   / _ \ @
;;  ( (_) )@
;;   \___/ @
;;         @
;;         @@
;; 0x03B5  GREEK SMALL LETTER EPSILON
;;       @
;;       @
;;   ___ @
;;  / __)@
;;  > _) @
;;  \___)@
;;       @
;;       @@
;; 0x03B6  GREEK SMALL LETTER ZETA
;;  _____  @
;;  \__  ) @
;;    / /  @
;;   / /   @
;;  | |__  @
;;   \__ \ @
;;      ) )@
;;     (_/ @@
;; 0x03B7  GREEK SMALL LETTER ETA
;;         @
;;         @
;;   _ __  @
;;  | '_ \ @
;;  | | | |@
;;  |_| | |@
;;      | |@
;;      |_|@@
;; 0x03B8  GREEK SMALL LETTER THETA
;;    ___  @
;;   / _ \ @
;;  | |_| |@
;;  |  _  |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 0x03B9  GREEK SMALL LETTER IOTA
;;      @
;;      @
;;   _  @
;;  | | @
;;  | | @
;;   \_)@
;;      @
;;      @@
;; 0x03BA  GREEK SMALL LETTER KAPPA
;;        @
;;        @
;;   _  __@
;;  | |/ /@
;;  |   < @
;;  |_|\_\@
;;        @
;;        @@
;; 0x03BB  GREEK SMALL LETTER LAMDA
;;  __     @
;;  \ \    @
;;   \ \   @
;;    > \  @
;;   / ^ \ @
;;  /_/ \_\@
;;         @
;;         @@
;; 0x03BC  GREEK SMALL LETTER MU
;;         @
;;         @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;  | ._,_|@
;;  | |    @
;;  |_|    @@
;; 0x03BD  GREEK SMALL LETTER NU
;;        @
;;        @
;;   _  __@
;;  | |/ /@
;;  | / / @
;;  |__/  @
;;        @
;;        @@
;; 0x03BE  GREEK SMALL LETTER XI
;;  \=\__  @
;;   > __) @
;;  ( (_   @
;;   > _)  @
;;  ( (__  @
;;   \__ \ @
;;      ) )@
;;     (_/ @@
;; 0x03BF  GREEK SMALL LETTER OMICRON
;;         @
;;         @
;;    ___  @
;;   / _ \ @
;;  ( (_) )@
;;   \___/ @
;;         @
;;         @@
;; 0x03C0  GREEK SMALL LETTER PI
;;          @
;;          @
;;   ______ @
;;  (  __  )@
;;   | || | @
;;   |_||_| @
;;          @
;;          @@
;; 0x03C1  GREEK SMALL LETTER RHO
;;         @
;;         @
;;    ___  @
;;   / _ \ @
;;  | |_) )@
;;  |  __/ @
;;  | |    @
;;  |_|    @@
;; 0x03C2  GREEK SMALL LETTER FINAL SIGMA
;;         @
;;         @
;;    ____ @
;;   / ___)@
;;  ( (__  @
;;   \__ \ @
;;     _) )@
;;    (__/ @@
;; 0x03C3  GREEK SMALL LETTER SIGMA
;;         @
;;         @
;;    ____ @
;;   /  ._)@
;;  ( () ) @
;;   \__/  @
;;         @
;;         @@
;; 0x03C4  GREEK SMALL LETTER TAU
;;       @
;;       @
;;   ___ @
;;  (   )@
;;   | | @
;;    \_)@
;;       @
;;       @@
;; 0x03C5  GREEK SMALL LETTER UPSILON
;;         @
;;         @
;;   _   _ @
;;  | | | |@
;;  | |_| |@
;;   \___/ @
;;         @
;;         @@
;; 0x03C6  GREEK SMALL LETTER PHI
;;      _    @
;;     | |   @
;;    _| |_  @
;;   /     \ @
;;  ( (| |) )@
;;   \_   _/ @
;;     | |   @
;;     |_|   @@
;; 0x03C7  GREEK SMALL LETTER CHI
;;         @
;;         @
;;  __   __@
;;  \ \ / /@
;;   \ v / @
;;    > <  @
;;   / ^ \ @
;;  /_/ \_\@@
;; 0x03C8  GREEK SMALL LETTER PSI
;;           @
;;           @
;;   _  _  _ @
;;  | || || |@
;;  | \| |/ |@
;;   \_   _/ @
;;     | |   @
;;     |_|   @@
;; 0x03C9  GREEK SMALL LETTER OMEGA
;;             @
;;             @
;;    __   __  @
;;   / / _ \ \ @
;;  | |_/ \_| |@
;;   \___^___/ @
;;             @
;;             @@
;; 0x03D1  GREEK THETA SYMBOL
;;      ___    @
;;     / _ \   @
;;    ( (_| |_ @
;;   _ \ _   _)@
;;  | |___| |  @
;;   \_____/   @
;;             @
;;             @@
;; 0x03D5  GREEK PHI SYMBOL
;;           @
;;           @
;;   _   __  @
;;  | | /  \ @
;;  | || || )@
;;   \_   _/ @
;;     | |   @
;;     |_|   @@
;; 0x03D6  GREEK PI SYMBOL
;;             @
;;             @
;;   _________ @
;;  (  _____  )@
;;  | |_/ \_| |@
;;   \___^___/ @
;;             @
;;             @@
  })

(def miniwi
 {
 :font-name "Miniwi"
 :example
[
"▖  ▖▘  ▘   ▘"
"▛▖▞▌▌▛▌▌▌▌▌▌"
"▌▝ ▌▌▌▌▌▚▚▘▌"
""
"   ▘  ▘   ▘"
"▛▛▌▌▛▌▌▌▌▌▌"
"▌▌▌▌▌▌▌▚▚▘▌"
 ]
 :author "Nick Miners"
 :desc "miniwi font
        after the miniwi bitmap font"
 :url "http://github.com/sshbio/miniwi"
 :figlet-metrics "flf2a$ 4 4 3 -1 5 0 1 0"
 :figlet-string
"$@
 $@
 $@
 $@@
▌@
▌@
▖@
 @@
▌▌@
  @
  @
  @@
▗▗ @
▜▜▘@
▜▜▘@
   @@
▗ @
█▘@
▟▌@
▝ @@
▖▖@
▗▘@
▌▖@
  @@
▗ @
▚▘@
▚▌@
  @@
▌@
 @
 @
 @@
▗▘@
▐ @
▐ @
▝▖@@
▝▖@
 ▌@
 ▌@
▗▘@@
▖▖@
▟▖@
▞▖@
  @@
  @
▟▖@
▝ @
  @@
  @
  @
▗ @
▘ @@
  @
▄▖@
  @
  @@
  @
  @
▗ @
  @@
 ▌@
▐ @
▞ @
▘ @@
▄▖@
▛▌@
█▌@
  @@
▗ @
▜ @
▟▖@
  @@
▄▖@
▄▌@
▙▖@
  @@
▄▖@
▄▌@
▄▌@
  @@
▖▖@
▙▌@
 ▌@
  @@
▄▖@
▙▖@
▄▌@
  @@
▄▖@
▙▖@
▙▌@
  @@
▄▖@
 ▌@
 ▌@
  @@
▄▖@
▙▌@
▙▌@
  @@
▄▖@
▙▌@
▄▌@
  @@
 @
▖@
▖@
 @@
 @
▖@
▖@
▘@@
 ▖@
▞ @
▝▖@
  @@
  @
▀▘@
▀▘@
  @@
▖ @
▝▖@
▞ @
  @@
▄▖@
▗▘@
▗ @
  @@
▗▄▖@
▌▄▐@
▌▀▀@
 ▀ @@
▄▖@
▌▌@
▛▌@
  @@
▄ @
▙▘@
▙▘@
  @@
▄▖@
▌ @
▙▖@
  @@
▄ @
▌▌@
▙▘@
  @@
▄▖@
▙▖@
▙▖@
  @@
▄▖@
▙▖@
▌ @
  @@
▄▖@
▌ @
▙▌@
  @@
▖▖@
▙▌@
▌▌@
  @@
▄▖@
▐ @
▟▖@
  @@
 ▖@
 ▌@
▙▌@
  @@
▖▖@
▙▘@
▌▌@
  @@
▖ @
▌ @
▙▖@
  @@
▖  ▖@
▛▖▞▌@
▌▝ ▌@
    @@
▖ ▖@
▛▖▌@
▌▝▌@
   @@
▄▖@
▌▌@
▙▌@
  @@
▄▖@
▙▌@
▌ @
  @@
▄▖@
▌▌@
█▌@
 ▘@@
▄▖@
▙▘@
▌▌@
  @@
▄▖@
▚ @
▄▌@
  @@
▄▖@
▐ @
▐ @
  @@
▖▖@
▌▌@
▙▌@
  @@
▖▖@
▌▌@
▚▘@
  @@
▖  ▖@
▌▞▖▌@
▛ ▝▌@
    @@
▖▖@
▚▘@
▌▌@
  @@
▖▖@
▌▌@
▐ @
  @@
▄▖@
▗▘@
▙▖@
  @@
▐▘@
▐ @
▐ @
▝▘@@
▌ @
▐ @
▝▖@
 ▘@@
▜ @
▐ @
▐ @
▀ @@
▗ @
▘▘@
  @
  @@
  @
  @
▄▖@
  @@
▚ @
 ▘@
  @
  @@
  @
▀▌@
█▌@
  @@
▌ @
▛▌@
▙▌@
  @@
  @
▛▘@
▙▖@
  @@
 ▌@
▛▌@
▙▌@
  @@
  @
█▌@
▙▖@
  @@
▐▘@
▜▘@
▐ @
  @@
  @
▛▌@
▙▌@
▄▌@@
▌ @
▛▌@
▌▌@
  @@
▘@
▌@
▌@
 @@
 ▘@
 ▌@
 ▌@
▙▌@@
▌ @
▙▘@
▛▖@
  @@
▜ @
▐ @
▐▖@
  @@
   @
▛▛▌@
▌▌▌@
   @@
  @
▛▌@
▌▌@
  @@
  @
▛▌@
▙▌@
  @@
  @
▛▌@
▙▌@
▌ @@
  @
▛▌@
▙▌@
 ▌@@
  @
▛▘@
▌ @
  @@
  @
▛▘@
▄▌@
  @@
▗ @
▜▘@
▐▖@
  @@
  @
▌▌@
▙▌@
  @@
  @
▌▌@
▚▘@
  @@
   @
▌▌▌@
▚▚▘@
   @@
  @
▚▘@
▞▖@
  @@
  @
▌▌@
▙▌@
▄▌@@
  @
▀▌@
▙▖@
  @@
▐▘@
▐ @
▐ @
▐▖@@
▌@
▌@
▌@
▘@@
▜ @
▐▖@
▐ @
▀ @@
   @
▖▄ @
▝▘▘@
   @@"})


(def rounded 
{
 :font-name "Rounded"
 :example
[
" ______                            _             _"
"(_____ \\                          | |           | |"
" _____) )  ___   _   _  ____    __| | _____   __| |"
"|  __  /  / _ \\ | | | ||  _ \\  / _  || ___ | / _  |"
"| |  \\ \\ | |_| || |_| || | | |( (_| || ____|( (_| |"
"|_|   |_| \\___/ |____/ |_| |_| \\____||_____) \\____|"
""
" ______   _______  _     _  _______  ______   _______  ______"
"(_____ \\ (_______)(_)   (_)(_______)(______) (_______)(______)"
" _____) ) _     _  _     _  _     _  _     _  _____    _     _"
"|  __  / | |   | || |   | || |   | || |   | ||  ___)  | |   | |"
"| |  \\ \\ | |___| || |___| || |   | || |__/ / | |_____ | |__/ /"
"|_|   |_| \\_____/  \\_____/ |_|   |_||_____/  |_______)|_____/"
 ]
 :author "Nick Miners"
 :desc "Rounded by Nick Miners N.M.Miners@durham.ac.uk
        May 1994"
 :figlet-metrics "flf2a$ 7 6 20 15 2"
 :figlet-string
"$$@
$$@
$$@
$$@
$$@
$$@
$$@@
 _ @
| |@
| |@
|_|@
 _ @
|_|@
   @@
 _  _ @
( )( )@
|/  \\|@
      @
      @
      @
      @@
   _ _   @
 _| U |_ @
(_     _)@
 _| O |_ @
(_     _)@
  |_n_|  @
         @@
   _   @
 _| |_ @
|  ___)@
|___  |@
(_   _|@
  |_|  @
       @@
 _   _ @
(_) | |@
   / / @
  / /  @
 / / _ @
|_| (_)@
       @@
  ___   @
 / _ \\  @
( (_) ) @
 ) _ (  @
( (/  \\ @
 \\__/\\_)@
        @@
 _ @
( )@
|/ @
   @
   @
   @
   @@
  _ @
 / )@
| | @
| | @
| | @
 \\_)@
    @@
 _  @
( \\ @
 | |@
 | |@
 | |@
(_/ @
    @@
    _    @
 _ | | _ @
( \\| |/ )@
 )     ( @
(_/| |\\_)@
   |_|   @
         @@
       @
   _   @
 _| |_ @
(_   _)@
  |_|  @
       @
       @@
   @
   @
   @
   @
 _ @
( )@
|/ @@
       @
       @
 _____ @
(_____)@
       @
       @
       @@
   @
   @
   @
   @
 _ @
(_)@
   @@
     _ @
    | |@
   / / @
  / /  @
 / /   @
|_|    @
       @@
  _____  @
 (_____) @
 _  __ _ @
| |/ /| |@
|   /_| |@
 \\_____/ @
         @@
  ___   @
 (___)  @
    _   @
   | |  @
  _| |_ @
 (_____)@
        @@
 ______  @
(_____ \\ @
  ____) )@
 / ____/ @
| (_____ @
|_______)@
         @@
 ______  @
(_____ \\ @
 _____) )@
(_____ ( @
 _____) )@
(______/ @
         @@
 _     _ @
| |   (_)@
| |_____ @
|_____  |@
      | |@
      |_|@
         @@
 _______ @
(_______)@
 ______  @
(_____ \\ @
 _____) )@
(______/ @
         @@
 _______ @
(_______)@
 ______  @
|  ___ \\ @
| |___) )@
|______/ @
         @@
 _______ @
(_______)@
      _  @
     / ) @
    / /  @
   (_/   @
         @@
  _____  @
 (_____) @
  _____  @
 / ___ \\ @
( (___) )@
 \\_____/ @
         @@
 _______ @
(_______)@
 _______ @
(_____  |@
      | |@
      |_|@
         @@
   @
   @
 _ @
(_)@
 _ @
(_)@
   @@
   @
   @
 _ @
(_)@
 _ @
( )@
|/ @@
    @
  _ @
 / )@
( ( @
 \\_)@
    @
    @@
       @
 _____ @
(_____)@
 _____ @
(_____)@
       @
       @@
    @
 _  @
( \\ @
 ) )@
(_/ @
    @
    @@
  ___  @
 / _ \\ @
(_( ) )@
   (_/ @
   _   @
  (_)  @
       @@
  _____  @
 / __  \\ @
| | /   )@
| | \\__/ @
| |____  @
 \\_____) @
         @@
 _______ @
(_______)@
 _______ @
|  ___  |@
| |   | |@
|_|   |_|@
         @@
 ______  @
(____  \\ @
 ____)  )@
|  __  ( @
| |__)  )@
|______/ @
         @@
 _______ @
(_______)@
 _       @
| |      @
| |_____ @
 \\______)@
         @@
 ______  @
(______) @
 _     _ @
| |   | |@
| |__/ / @
|_____/  @
         @@
 _______ @
(_______)@
 _____   @
|  ___)  @
| |_____ @
|_______)@
         @@
 _______ @
(_______)@
 _____   @
|  ___)  @
| |      @
|_|      @
         @@
 _______ @
(_______)@
 _   ___ @
| | (_  |@
| |___) |@
 \\_____/ @
         @@
 _     _ @
(_)   (_)@
 _______ @
|  ___  |@
| |   | |@
|_|   |_|@
         @@
 _ @
| |@
| |@
| |@
| |@
|_|@
   @@
 _______ @
(_______)@
     _   @
 _  | |  @
| |_| |  @
 \\___/   @
         @@
 _     _ @
(_)   | |@
 _____| |@
|  _   _)@
| |  \\ \\ @
|_|   \\_)@
         @@
 _       @
(_)      @
 _       @
| |      @
| |_____ @
|_______)@
         @@
 _______ @
(_______)@
 _  _  _ @
| ||_|| |@
| |   | |@
|_|   |_|@
         @@
 _______ @
(_______)@
 _     _ @
| |   | |@
| |   | |@
|_|   |_|@
         @@
 _______ @
(_______)@
 _     _ @
| |   | |@
| |___| |@
 \\_____/ @
         @@
 ______  @
(_____ \\ @
 _____) )@
|  ____/ @
| |      @
|_|      @
         @@
 _______ @
(_______)@
 _    _  @
| |  | | @
| |__| | @
 \\______)@
         @@
 ______  @
(_____ \\ @
 _____) )@
|  __  / @
| |  \\ \\ @
|_|   |_|@
         @@
  ______ @
 / _____)@
( (____  @
 \\____ \\ @
 _____) )@
(______/ @
         @@
 _______ @
(_______)@
    _    @
   | |   @
   | |   @
   |_|   @
         @@
 _     _ @
(_)   (_)@
 _     _ @
| |   | |@
| |___| |@
 \\_____/ @
         @@
 _     _ @
(_)   (_)@
 _     _ @
| |   | |@
 \\ \\ / / @
  \\___/  @
         @@
 _  _  _ @
(_)(_)(_)@
 _  _  _ @
| || || |@
| || || |@
 \\_____/ @
         @@
 _     _ @
(_)   (_)@
   ___   @
  |   |  @
 / / \\ \\ @
|_|   |_|@
         @@
 _     _ @
| |   | |@
| |___| |@
|_____  |@
 _____| |@
(_______|@
         @@
 _______ @
(_______)@
   __    @
  / /    @
 / /____ @
(_______)@
         @@
 ___ @
|  _)@
| |  @
| |  @
| |_ @
|___)@
     @@
 _     @
| |    @
 \\ \\   @
  \\ \\  @
   \\ \\ @
    |_|@
       @@
 ___ @
(_  |@
  | |@
  | |@
 _| |@
(___|@
     @@
   __  @
  /  \\ @
 (_/\\_)@
       @
       @
       @
       @@
         @
         @
         @
         @
 _______ @
(_______)@
         @@
 _ @
( )@
 \\|@
   @
   @
   @
   @@
       @
       @
 _____ @
(____ |@
/ ___ |@
\\_____|@
       @@
 _     @
| |    @
| |__  @
|  _ \\ @
| |_) )@
|____/ @
       @@
       @
       @
  ____ @
 / ___)@
( (___ @
 \\____)@
       @@
     _ @
    | |@
  __| |@
 / _  |@
( (_| |@
 \\____|@
       @@
       @
       @
 _____ @
| ___ |@
| ____|@
|_____)@
       @@
    ___ @
   / __)@
 _| |__ @
(_   __)@
  | |   @
  |_|   @
        @@
       @
       @
  ____ @
 / _  |@
( (_| |@
 \\___ |@
(_____|@@
 _     @
| |    @
| |__  @
|  _ \\ @
| | | |@
|_| |_|@
       @@
 _ @
(_)@
 _ @
| |@
| |@
|_|@
   @@
   _ @
  (_)@
   _ @
  | |@
  | |@
 _| |@
(__/ @@
 _     @
| |    @
| |  _ @
| |_/ )@
|  _ ( @
|_| \\_)@
       @@
 _  @
| | @
| | @
| | @
| | @
 \\_)@
    @@
       @
       @
 ____  @
|    \\ @
| | | |@
|_|_|_|@
       @@
       @
       @
 ____  @
|  _ \\ @
| | | |@
|_| |_|@
       @@
       @
       @
  ___  @
 / _ \\ @
| |_| |@
 \\___/ @
       @@
       @
       @
 ____  @
|  _ \\ @
| |_| |@
|  __/ @
|_|    @@
       @
       @
  ____ @
 / _  |@
| |_| |@
 \\__  |@
    |_|@@
       @
       @
  ____ @
 / ___)@
| |    @
|_|    @
       @@
      @
      @
  ___ @
 /___)@
|___ |@
(___/ @
      @@
       @
   _   @
 _| |_ @
(_   _)@
  | |_ @
   \\__)@
       @@
       @
       @
 _   _ @
| | | |@
| |_| |@
|____/ @
       @@
       @
       @
 _   _ @
| | | |@
 \\ V / @
  \\_/  @
       @@
       @
       @
 _ _ _ @
| | | |@
| | | |@
 \\___/ @
       @@
       @
       @
 _   _ @
( \\ / )@
 ) X ( @
(_/ \\_)@
       @@
       @
       @
 _   _ @
| | | |@
| |_| |@
 \\__  |@
(____/ @@
       @
       @
 _____ @
(___  )@
 / __/ @
(_____)@
       @@
   __ @
  / _)@
 | |  @
( (   @
 | |_ @
  \\__)@
      @@
 _ @
| |@
|_|@
 _ @
| |@
|_|@
   @@
 __   @
(_ \\  @
  | | @
   ) )@
 _| | @
(__/  @
      @@
  __  _ @
 /  \\/ )@
(_/\\__/ @
        @
        @
        @
        @@
 _______ @
(_)___(_)@
 _______ @
|  ___  |@
| |   | |@
|_|   |_|@
         @@
 _______ @
(_)___(_)@
 _     _ @
| |   | |@
| |___| |@
 \\_____/ @
         @@
 _     _ @
(_)   (_)@
 _     _ @
| |   | |@
| |___| |@
 \\_____/ @
         @@
 _   _ @
(_) (_)@
 _____ @
(____ |@
/ ___ |@
\\_____|@
       @@
 _   _ @
(_) (_)@
  ___  @
 / _ \\ @
| |_| |@
 \\___/ @
       @@
 _   _ @
(_) (_)@
 _   _ @
| | | |@
| |_| |@
|____/ @
       @@
  ___  @
 / _ \\ @
| ( ) )@
| |( ( @
| | ) )@
|_|(_/ @
       @@"
})

;; Functions to generate font maps for use during dev --------------------------

(defn- figlet-font-multiple-character-heights-warning!
  [font-name m]
  (let [m+                    m #_(update-in m ["o" :height] + 2)
        height-frequencies    (some->> m+
                                       (mapv (fn [[_ v]] (:height v)))
                                       frequencies) ]
    (when (< 1 (count height-frequencies))
       (let [least-frequent-height (some->> height-frequencies
                                            (sort-by val <)
                                            ffirst)
             most-frequent-height (some->> height-frequencies
                                           (sort-by val >)
                                           ffirst)
             chars (keep (fn [[_ {:keys [height character]}]]
                           (when (not= height most-frequent-height)
                             character)) 
                         m+)
             ]
         (println 
          (str "WARNING!\n"
               "Figlet font may contain chars of varying heights:\n\n"
               "font-name:\n"
               "\"" font-name  "\""
               "\n\n"
               "height frequencies:"
               "\n"
               (with-out-str (println height-frequencies))
               "\n"
               "chars found with lesser frequent heights:\n"
               "\"" (string/join "\n" chars) "\""
               "\n"))
         (doseq [char chars]
           (pprint (get m char)))))))

(defn- space-width [widths]
  (let [pos-widths (filter pos? widths)]
    (Math/round 
     (Math/ceil 
      (double 
       (/ (/ (apply + pos-widths)
             (count pos-widths))
          2))))))

(defn- space-char [widths max-height]
  (let [space-width (space-width widths)]
    {:bands     (->> (repeat space-width " ")
                     string/join
                     (repeat max-height)
                     vec)
     :i         0
     :character (ascii-chars-by-index-map 0)
     :width     space-width
     :height    max-height}))

(defn- char-bands-coll-inner [i s]
  (when (pos? i)
    (let [[_ & bands]       
          (string/split-lines s)

          [!-char :as bands]
          (mapv #(string/replace % #"@$" "") bands)

          width              
          (count !-char)]
      (merge {:bands     bands
              :i         i
              :character (ascii-chars-by-index-map i)
              :width     width
              :height    (count bands)}
             (when (every? (fn [s] (= "" s)) bands)  
                   {:missing? true})))))

(defn- chars-array-map* [coll]
  (let [m 
        (reduce (fn [acc m]
                  {:coll    (conj (:coll acc) (:character m) m)
                   :missing (if (:missing? m)
                              (conj (:missing acc) (:character m))
                              (:missing acc))})
                {:coll    []
                 :missing []}
                coll)

        chars-array-map
        (apply array-map (:coll m)) 

        missing-chars
        (->> chars-array-map
             keys
             (clojure.data/diff ascii-chars)
             first
             (remove nil?)
             (concat (:missing m))
             (into []))]
     (keyed [chars-array-map missing-chars])))

(defn- font-metrics [coll]
  (reduce 
   (fn [{:keys [widest-char widest-char-width widths max-height]
         :as   acc}
        {:keys [width height character]}]
     (assoc acc
            :widths 
            (conj widths width)

            :widest-char-width
            (if (> width widest-char-width)
              width
              widest-char-width)

            :widest-char
            (if (> width widest-char-width)
              character
              widest-char)
            
            :max-height
            (if (> height max-height)
              height
              max-height)))
   {:widths            []
    :widest-char-width 0
    :max-height        0
    :widest-char       ""}
   coll))


(defn- missing-char-str [width s]
  (case width
    0 ""
    1 s
    2 (str " " s)
    3 (str " " s " ")
    (if (odd? width)
      (let [pad (sjr (/ (dec width) 2) " ")]
        (str pad s pad))
      (let [pad (sjr (/ width 2) " ")]
        (str pad s (subs pad 1))))))

(defn- replacement-char-vec 
  [{:keys [width height]}
   {s :character}]
  (into []
        (let [s        (missing-char-str width s)
              band-str (sjr width " ")]
          (if (odd? height)
            (concat (repeat (dec (/ (dec height) 2)) band-str)
                    [s]
                    (repeat (inc (/ (dec height) 2)) band-str))
            (concat (repeat (dec (/ height 2)) band-str)
                    [s]
                    (repeat (/ height 2) band-str))))))

(defn- replacement-char-fn [space-char %]
  (if (:missing? %)
    (let [vc (replacement-char-vec space-char %)]
      (assoc %
             :bands
             vc
             :height
             (count vc)))
    %))

(def font-maps-by-sym
  {'big-money   big-money
   'ansi-shadow ansi-shadow
   'drippy      drippy
   'big         big
   'miniwi      miniwi
   'isometric-1 isometric-1
   'rounded     rounded})

(def font-maps-by-name
  (reduce (fn [acc [_ v]]
            (assoc acc (:font-name v) v))
          {}
          font-maps-by-sym))

(def font-names-by-font
  (clojure.set/map-invert font-maps-by-name))

(defn banner-font-array-map [font-sym]
  (when-let [{:keys [figlet-string 
                     font-name
                     author-name
                     desc
                     more-info
                     url
                     example
                     figlet-metrics
                     figlet-string]
              :as   font-map}
             (get font-maps-by-sym font-sym)]
   (let [char-strings-coll
         (string/split figlet-string #"@@")

         char-bands-coll
         (keep-indexed char-bands-coll-inner char-strings-coll)

         {:keys [widest-char widths max-height]}
         (font-metrics char-bands-coll)

         space-char
         (space-char widths max-height)

         char-bands-coll
         (mapv (partial replacement-char-fn space-char) char-bands-coll)

         char-bands-coll
         (cons space-char char-bands-coll)
         
         {:keys [chars-array-map missing-chars]}
         (chars-array-map* char-bands-coll)]

     (figlet-font-multiple-character-heights-warning! 
      font-name
      chars-array-map)

     (apply array-map
             (reduce
              (fn [acc [k v]]
                (if k
                  (conj acc k v)
                  acc))
              []
              [[:font-name font-name]
               [:example example]
               (some->> author-name (vector :author-name))
               (some->> desc (vector :desc))
               (some->> more-info (vector :more-info))
               (some->> url (vector :url))
               [:font-sym font-sym]
               [:widest-char widest-char]
               [:char-height max-height]
               [:max-char-width (apply max widths)]
               [:missing-chars missing-chars]
               [:chars-array-map chars-array-map]])))))



;; This is called during development of the fonts themselves, by code in
;; bling.test that is calling bling.banner/banner to try out new / changed fonts
(def fonts-by-sym
 (apply array-map
        (reduce (fn [vc sym]
                  (conj vc sym (banner-font-array-map sym)))
                []
                defs/banner-fonts-vec)))



;; For manually generating bling.fonts namespace from babashka -----------------
#?(:clj
   (do
     (defn- font-def-source-code [m font-sym]
       (let [commented-example-of-output    (some->> m
                                                     :example
                                                     (string/join "\n;;  ")
                                                     (str ";;  ")) 

             quoted-pretty-printed-font-map (-> m
                                                pprint
                                                with-out-str
                                                (string/replace #"\n$" "")
                                                (->> (str "'")))]
         
         (str commented-example-of-output
              "\n"
              "\n"
              "(def " (name font-sym) "\n"
              quoted-pretty-printed-font-map ")")))

     (defn- font-defs [path]
       (let [banner-fonts defs/banner-fonts-vec
             banner-fonts-count (count banner-fonts)]
         (string/join 
          "\n\n\n"
          (mapv #(font-def-source-code 
                  (banner-font-array-map %)
                  %)
                (do (println 
                     (str
                      "Creating bling.fonts namespace...\n\n"
                      "Writing defs for the following " banner-fonts-count " fonts to\n"
                      path
                      "\n\n"
                      (string/join "\n"
                                   (mapv #(symbol (str "bling.fonts/" %))
                                         banner-fonts)))
                     )
                    banner-fonts)))))

     (defn write-fonts-ns!
       "This updates/generates a bling.fonts namespace.
      
        Used ocasionally during dev, when fonts are modified or added.
  
        This is intended to be called from repl or babashka script in
        dev/bb-script.cljc.
        
        To call from bb (from the bling project root dir):
        bb bb-script.cljc"
       []
       (spit "./src/bling/fonts.cljc"
             (str "(ns bling.fonts)"
                  "\n"
                  "\n"
                  "\n"
                  "\n"
                  (font-defs "./src/bling/fonts.cljc"))
             :append false))))
