(ns missinterpret.storage.utils.s3
  (:require [clojure.pprint :refer [pprint]]
            [clojure.string :as st]
            [clojurewerkz.urly.core :as url]
            [cognitect.aws.client.api :as aws]
            [missinterpret.anomalies.anomaly :refer [throw+ throw-if-cognitect-anomaly]]
            [missinterpret.storage.block.predicate :as pred.block]))

;; Fns --------------------------------------------------------------
;;

(defn client [config]
  (let [s3 (if (some? config)
             (aws/client (merge {:api :s3} config))
             (aws/client {:api :s3}))]
    (aws/validate-requests s3 true)
    s3))


(defn list-buckets [config]
  (-> (client config)
       (aws/invoke {:op :ListBuckets})
       throw-if-cognitect-anomaly))


(defn bucket-exists? [config bucket]
  (contains?
    (reduce
      (fn [coll b]
        (conj coll (:Name b)))
      #{}
      (-> (list-buckets config)
          :Buckets))
    bucket))


(defn create-bucket [config bucket]
  (-> (client config)
       (aws/invoke {:op :CreateBucket :request {:Bucket bucket}})
       throw-if-cognitect-anomaly))


(defn delete-bucket [config bucket]
  (-> (client config)
       (aws/invoke {:op :DeleteBucket :request {:Bucket bucket}})
       throw-if-cognitect-anomaly))


(defn bucket-objects [config bucket]
  (-> (client config)
      (aws/invoke {:op :ListObjectsV2 :request {:Bucket bucket}})
      throw-if-cognitect-anomaly))


(defn get-object
  "Get's the object from the given bucket. Supports range requests returning
   a cognitect anomaly map if the aws call fails.

  NOTE: `:storage.block.read/start` and `:storage.block.read/end` follow Clojure indexing
         but s3 range is 1 indexed. This fn adds 1 to a given start parameter."
  [{:keys [client-configuration bucket key]
    :source.range/keys [start end]
    :as args}]
  (when-not (pred.block/s3-args? args)
    (throw+
      {:from     ::get-object
       :category :anomaly.category/invalid
       :message  {:readable "Invalid S3 args"
                  :reasons  [:invalid/s3.arguments]
                  :data {:arg1 args}}}))
  (let [base-req {:op :GetObject :request {:Bucket bucket :Key key}}
        req (cond
              (and (some? start) (some? end))
              (let [s (inc start)
                    range-string (str "bytes=" s "-" end)]
                (assoc-in base-req [:request :Range] range-string))

              (or (some? start) (some? end))
              (throw+
                {:from      ::get-object
                 :category  :anomaly.category/invalid
                 :message   {:readable "Invalid s3 range request"
                             :reasons  [:invalid/s3.range-request]
                             :data     {:arg1 args}}})
              :else base-req)]
    (-> (client client-configuration)
        (aws/invoke req)
        throw-if-cognitect-anomaly)))


(defn object-last-modified [{:keys [client-configuration bucket key] :as args}]
  ;; Looks like it only has last-modified?
  ;; https://github.com/cognitect-labs/aws-api
  ;; https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingMetadata.html
  "implement"
  )

(defn object-size [{:keys [client-configuration bucket key] :as args}]
  "0")


(defn put-object
  "Adds the object to the given object with the key. Supports setting the
   storage tier that maps to availability via the optional
   `:storage.access/availablility` argument:

   Throws an anomaly if the put object call fails."
  ;; TODO: Check that the spec isn't strict...
  [{:keys [client-configuration bucket key]
    :storage/keys [input-stream]
    :provider/keys [availability]
    :as args}]
  (when-not (pred.block/s3-args? args)
    (throw+
      {:from     ::out-object
       :category :anomaly.category/invalid
       :message  {:readable "Invalid S3 arguments"
                  :reasons  [:invalid.s3/arguments]
                  :data {:argument args}}}))
  (let [base-req {:op :PutObject :request {:Bucket bucket :Key key :Body input-stream}}
        req (if (some? availability)
              (cond
                :else base-req) ;; TODO: Map availabilty to storage class
              base-req)
        resp (-> (client client-configuration)
                 (aws/invoke req)
                 throw-if-cognitect-anomaly)]
    (when (not (contains? resp :ETag))
      (throw+
        {:from      ::put-object
         :category  :anomaly.category/fault
         :message   {:readable (str "Invalid GetObject response: " resp)
                     :reasons  [:invalid.provider.s3/response]
                     :data {:argument args :request req :resp resp}}}))
    resp))


(defn delete-object
  "Deletes the object from the bucket with the key. Throws an anomaly
   if the object does not delete."
  [{:keys [client-configuration bucket key] :as args}]
  (when-not (pred.block/s3-args? args)
    (throw+
      {:from     ::delete-object
       :category :anomaly.category/invalid
       :message  {:readable "Invalid S3 arguments"
                  :reasons  [:invalid/s3.arguments]
                  :data {:arg1 args}}}))
  (let [req {:op :DeleteObject :request {:Bucket bucket :Key key}}]
    (-> (client client-configuration)
        (aws/invoke req)
        throw-if-cognitect-anomaly)))


(defn parse-s3-uri [uri]
  (let [s3u (url/url-like uri)]
    (if (= "s3" (url/protocol-of s3u))
      (let [key (-> (url/path-of s3u) (subs 1))]
        {:bucket (url/host-of s3u)
         :key    key})
      (throw+
        {:from     ::parse-s3-uri
         :category :anomaly.category/invalid
         :message {:readable (str "Invalid s3 uri: " uri)
                   :reasons  [:invalid.provider.s3/uri]
                   :data {:argument uri}}}))))

(defn s3-ref->uri [bucket key]
  (let [b (if (st/starts-with? bucket "/")
            (subs bucket 1)
            bucket)
        k (if (st/starts-with? key "/")
            (subs key 1)
            key)]
    (str "s3://" b "/" k)))

(defn uri-exists? [config s3-uri]
  (let [args  (parse-s3-uri s3-uri)
        key   (:key args)
        objs  (bucket-objects config (:bucket args))]
    (->> (:Contents objs)
         (filter #(st/includes? (:Key %) key))
         first
         some?)))

(defn object-exists? [{:keys [client-configuration bucket key]}]
  (uri-exists? client-configuration (s3-ref->uri bucket key)))

(defn key-id [{:keys [key]}]
  (-> key (st/split #"\.") first))