(ns missinterpret.storage.block.file-system
  (:require [clojure.java.io :as io]
            [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.address.core :as core.address]
            [missinterpret.storage.address.predicate :as pred.address]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.edn-io.data-readers :as data-readers] ;; tag literals
            [missinterpret.edn-io.edn :as edn]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.protocols.block :as prot.block]
            [missinterpret.storage.provider.file-system :as provider.fs]
            [missinterpret.storage.utils.byte-arrays :as utils.bytes]
            [missinterpret.storage.utils.file :as utils.file]
            ))

;; Definitions -------------------------------------------------------------
;;

(def kind    :block.kind/file-system)
(def version "1.0.0")
(def hash :sha2-256)

(def block-info #:block.info{:kind    kind
                             :version version
                             :hash    hash})


;; Block Impl ----------------------------------------------------
;;

(defrecord FileSystemBlock [arguments block-cache]
  prot.block/Block

  (id [this]
    (let [{:keys [file validate-path-address? use-path-address?]} arguments]
      (if (true? use-path-address?)
        (let [fname-id (utils.file/filename file)
              address #:address{:id            fname-id
                                :algorithm     hash
                                :size          (utils.file/size file)
                                :calculated-at (utils.file/created-on file)}]
          (when (and (true? validate-path-address?)
                     (not= fname-id (-> (core.block/compute-id-address this) :address/id)))
            (throw+
              {:from     ::id
               :category :anomaly.category/incorrect
               :message  {:readable "File address id does not match computed address id"
                          :reasons  [:incorrect/file.address]
                          :data     {:arg1 arguments}}}))
          #:block.id{:address address
                     :arguments arguments})
        (let [address (core.block/compute-id-address this)]
          (when (and (true? validate-path-address?)
                     (not= (utils.file/filename file) (:address/id address)))
            (throw+
              {:from     ::id
               :category :anomaly.category/incorrect
               :message  {:readable "File address id does not match computed address id"
                          :reasons  [:incorrect/file.address]
                          :data     {:arg1 arguments}}}))
          #:block.id{:address address
                     :arguments arguments}))))

  (info [_] block-info)

  (data [_]
    (if (some? @block-cache)
      @block-cache
      (let [{:keys [file]} arguments
            ;; TODO: replace with edn reader
            sb (->> file slurp edn/read)]
        (reset! block-cache sb)
        sb)))

  (address [this] (core.block/address this))
  (metadata [this] (core.block/metadata this))
  (sources [this] (core.block/sources this)))


;; Factory ----------------------------------------------------------------
;;

(defn load
  "Loads a block on a local file system."
  [{:keys [file validate-id-address? use-path-address?] :as arguments}]
  (when-not (pred.block/fs-args? arguments)
    (throw+
      {:from     ::load
       :category :anomaly.category/invalid
       :message  {:readable "Block ID invalid"
                  :reasons  [:invalid/block.arguments]
                  :data     {:arg1 arguments}}}))
  {:storage/block
   (map->FileSystemBlock {:arguments arguments
                          :block-cache (atom nil)})})

(defn new
  "Creates a new block from the given file by computing a new address."
  [{:keys [file name metadata content-type]
    :as arguments}]
  (when-not (and (pred.block/fs-args? arguments)
                 (some? name))
    (throw+

      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Block arguments invalid"
                  :reasons  [:invalid/block.arguments]
                  :data     {:arg1 arguments}}}))
  (let [address (-> (io/input-stream file)
                    utils.bytes/to-byte-array
                    core.address/byte-array->address)
        ext (utils.file/ext file)
        uri (.toURI (io/file file))
        source (provider.fs/new-source {:name         name
                                        :content-type content-type
                                        :ext          ext
                                        :uri          uri})
        meta (if (some? metadata) metadata {})]
    {:storage/block (core.block/direct-block meta address source)}))




;; Utils ----------------------------------------------------------------
;;

(defn id-files
  "Returns the files which have a valid address id as their filename"
  [files]
  (filterv
    (fn [f]
      (let [filename (utils.file/filename f)]
        (pred.address/valid-address-id? filename)))
    files))

