(ns missinterpret.storage.block.core
  (:require [clojure.pprint :refer [pprint]]
            [clojure.set :as set]
            [clojurewerkz.urly.core :as url]
            [missinterpret.anomalies.anomaly :refer [throw+ anomaly?]]
            [missinterpret.storage.protocols.block :as prot.block]
            [missinterpret.storage.address.predicate :as pred.address]
            [missinterpret.storage.source.predicate :as pred.source]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.address.core :as core.address]
            [missinterpret.storage.utils.core :as core.utils]))

(defn block [b]
  (let [blk (if (and (map? b) (contains? b :storage/block))
              (:storage/block b)
              b)]
    (when-not (pred.block/block? blk)
      (throw+
        {:from     ::block
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Block"
                    :reasons  [:invalid/block]
                    :data {:arg1 b :block blk}}}))
    blk))

(defn id [blk]
  (let [id (-> blk block prot.block/id)]
    (when-not (pred.block/id? id)
      (throw+
        {:from     ::id
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Id"
                    :reasons  [:invalid/block.id]
                    :data {:arg1 blk :id id}}}))
    id))

(defn block-address-id [blk]
  (let [id (-> (id blk)
               (get-in [:block.id/address :address/id]))]
    (when-not (pred.address/valid-address-id? id)
      (throw+
        {:from     ::block-address-id
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Id"
                    :reasons  [:invalid/block.address-id]
                    :data {:arg1 blk :id id}}}))
    id))

(defn data [blk]
  (let [data (-> blk block prot.block/data)]
    (when-not (pred.block/data? data)
      (throw+
        {:from     ::data
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Block Data"
                    :reasons  [:invalid/block.data]
                    :data {:arg1 blk :data data}}}))
    data))

(defn compute-id-address [blk]
  (-> (data blk) core.address/data->address))

(defn address [blk]
  (let [address (-> (data blk)
                    (get-in [:block/content :content/address]))]
    (when-not (pred.address/address? address)
      (throw+
        {:from     ::address
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Address"
                    :reasons  [:invalid/block.address]
                    :data {:arg1 blk :address address}}}))
    address))

(defn metadata [blk]
  (let [metadata (-> (data blk) :block/metadata)]
    (when-not (pred.block/metadata? metadata)
      (throw+
        {:from     ::metadata
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Block Metadata"
                    :reasons  [:invalid/block.metadata]
                    :data {:arg1 blk :metadata metadata}}}))
    metadata))

(defn sources [blk]
  (if-not (pred.block/block? blk)
    #{}
    (-> (data blk)
        (get-in [:block/content :content/sources]))))

(defn sources-by-kind [blk kind]
  (filterv
    #(= kind (:source/kind %))
    (sources blk)))

(defn sources-by-name [blk source-name]
  (filter
    #(= source-name (:source/name %))
    (sources blk)))

(defn sources-by-uri-match [blk {:keys [regex protocol]}]
  (reduce
    (fn [coll s]
      (let [uri (:source/uri s)
            path (when (some? uri)
                   (-> uri core.utils/to-uri str))]
        (if (and (some? uri) (= protocol (url/protocol-of uri)) (re-find regex path))
          (conj coll s)
          coll)))
    #{}
    (sources blk)))


;; Adapter Blocks -----------------------------------------------------------

(defrecord DirectBlock [metadata address source]
  prot.block/Block

  (id [this]
    (let [addr (core.address/data->address (prot.block/data this))]
      #:block.id{:address    addr
                 :arguments  {:metadata metadata
                              :address address
                              :source source}}))

  (info [_] #:block.info{:kind    :storage.block.kind/direct
                         :version "1.0.0"
                         :hash    (:address/algorithm address)})

  (data [_]
    #:block{:metadata metadata
            :content
            {:content/address address
             :content/sources #{source}}})

  (address [_] address)
  (metadata [_] metadata)
  (sources [_] #{source}))

(defn direct-block
  ([address source]
   (direct-block {} address source))
  ([metadata address source]
   (if (and (map? metadata)
            (pred.address/address? address)
            (pred.source/source? source))
     (map->DirectBlock {:metadata metadata
                        :address address
                        :source source})
     (throw+
       {:from     ::direct-block
        :category :anomaly.category/invalid
        :message  {:readable "Arguments invalid"
                   :reasons  [:or :invalid.block/metadata :invalid/address :invalid/source]
                   :data     {:arg1 metadata :arg2 address :arg3 source}}}))))

(defrecord AddSourcesBlock [block add-sources]
  prot.block/Block

  (id [this]
    (let [addr (core.address/data->address (data this))]
      #:block.id{:address   addr
                 :arguments {:block block
                             :add-sources add-sources}}))

  (info [_]
    (let [info (prot.block/info block)]
      (assoc info :block.info/kind :block.kind/add-sources)))

  (data [this]
    (let [data (data block)
          this-sources (prot.block/sources this)]
      (assoc-in data [:block/content :content/sources] this-sources)))

  (address [_] (address block))
  (metadata [_] (metadata block))

  (sources [_]
    (let [b-sources (sources block)]
      (set/union b-sources add-sources))))

(defn add-sources [blk add-sources]
  (cond
    (anomaly? add-sources) add-sources

    (not (and (pred.block/block? blk) (pred.source/are-sources? add-sources)))
    (throw+
      {:from     ::add-sources
       :category :anomaly.category/invalid
       :message  {:readable "Arguments invalid"
                  :reasons  [:or :invalid/block :invalid.provider/sources]
                  :data     {:arg1 blk :arg2 add-sources}}})

    :else (map->AddSourcesBlock {:block blk :add-sources add-sources})))


(defrecord RmSourcesBlock [block rm-sources]
  prot.block/Block

  (id [this]
    (let [addr (core.address/data->address (data this))]
      #:block.id{:address addr
                 :arguments {:block block
                             :rm-sources rm-sources}}))

  (info [_]
    (let [info (prot.block/info block)]
      (assoc info :block.info/kind :block.kind/rm-sources)))

  (data [this]
    (let [metadata (metadata block)
          address (address block)
          this-sources (prot.block/sources this)]
      #:block{:metadata metadata
              :content {:content/address address
                        :content/sources this-sources}}))

  (address [_] (address block))
  (metadata [_] (metadata block))

  (sources [_]
    (let [b-sources (sources block)]
      (set/difference b-sources rm-sources))))

(defn rm-sources [blk rm-sources]
  (cond
    (anomaly? rm-sources) rm-sources

    (not (and (pred.block/block? blk) (pred.source/are-sources? rm-sources)))
    (throw+
      {:from     ::rm-sources
       :category :anomaly.category/invalid
       :message  {:readable "Arguments invalid"
                  :reasons  [:or :invalid/block :invalid.provider/sources]
                  :data     {:arg1 blk :arg2 rm-sources}}})

    :else (map->RmSourcesBlock {:block blk :rm-sources rm-sources})))


(def metadata-actions
  #{:metadata/replace :metadata/merge-first :metadata/merge-last})

(defn metadata-action? [action]
  (contains? metadata-actions action))

(defrecord ApplyMetadataBlock [block metadata action]
  prot.block/Block

  (id [this]
    (let [addr (core.address/data->address (data this))]
      #:block.id{:address   addr
                 :arguments {:block    block
                             :metadata metadata
                             :action   action}}))

  (info [_]
    (let [info (prot.block/info block)]
      (assoc info :block.info/kind :block.kind/apply-metadata)))

  (data [this]
    (let [data (prot.block/data block)]
      (assoc data :block/metadata (prot.block/metadata this))))

  (address [_] (address block))

  (metadata [_]
    (let [md (prot.block/metadata block)
          action (if (metadata-action? action)
                   action
                   :metadata/merge-last)]
      (cond
        (= :metadata/replace action)     metadata
        (= :metadata/merge-first action) (merge metadata md)
        (= :metadata/merge-last action)  (merge md metadata)
        :else md)))

  (sources [_] (sources block)))


(defn apply-metadata [blk metadata action]
  (if (and (pred.block/block? blk) (pred.block/metadata? metadata))
    (map->ApplyMetadataBlock {:block blk
                              :metadata metadata
                              :action action})
    (throw+
      {:from     ::apply-metadata
       :category :anomaly.category/invalid
       :message  {:readable "Arguments invalid"
                  :reasons  [:or :invalid/block :invalid.block/metadata :invalid.block.metadata/action]
                  :data     {:arg1 blk :arg2 metadata :arg3 action}}})))

