(ns missinterpret.storage.store.predicate
  (:require [clojure.pprint :refer [pprint]]
            [malli.core :as m]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.provider.core :as core.prov]
            [missinterpret.storage.store.spec :as spec]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.protocols.store :as prot.store]
            [missinterpret.storage.protocols.block-store :as prot.block-store]
            [missinterpret.storage.protocols.content-store :as prot.content-store]))

;; IDs -------------------------------------------------

(defn block-store-id? [args]
  (m/validate spec/BlockStoreId args))

(defn memory-block-store-id? [args]
  (and (block-store-id? args)
       (m/validate spec/MemoryBlockStoreId args)))

(defn file-block-store-id? [args]
  (and (block-store-id? args)
       (m/validate spec/FileBlockStoreId args)))


(defn content-store-id? [args]
  (m/validate spec/ContentStoreId args))

(defn memory-content-store-id? [args]
  (and (content-store-id? args)
       (m/validate spec/MemoryContentStoreId args)))

(defn file-content-store-id? [args]
  (and (content-store-id? args)
       (m/validate spec/FileContentStoreId args)))


(defn block-store? [block-store]
  (try
    (let [id (prot.block-store/id block-store)]
      (block-store-id? id))
    (catch Exception _ false)))


(defn content-store? [content-store]
  (try
    (let [id (prot.content-store/id content-store)]
      (content-store-id? id))
    (catch Exception _ false)))


;; NOTE: This spec needs to be here to avoid a circular dependency
(def StoreId
  [:map
   [:store.id/name    some?]
   [:store.id/block   [:fn block-store?]]
   [:store.id/content [:fn content-store?]]])

(defn store-id? [id]
  (m/validate StoreId id))

;; Runtime ------------------------------------------------

(defn store? [store]
  (try
    (let [id (prot.store/id store)]
      (m/validate StoreId id))
    (catch Exception _ false)))

(defn consistent?
  "Does the provider's content address id match the block
   content address id?

   If either provider or block are not valid entities - returns true"
  [{:storage/keys [provider block]}]
  (try
    (cond
      (not (pred.prov/provider? provider)) true
      (not (pred.block/block? block)) true
      :else
      (let [addr-id-1 (-> (core.prov/address provider) :address/id)
            addr-id-2 (-> (core.block/address block) :address/id)]
        (= addr-id-1 addr-id-2)))
    (catch Exception _ false)))


;; Arguments -----------------------------------------------

(defn store-arguments? [args]
  (m/validate spec/StoreArgs args))


;; Fn Options ----------------------------------------------

(defn flow-store-opts?
  [opts]
  (m/validate spec/FlowStoreOpts opts))

(defn query-opts?
  [opts]
  (m/validate spec/QueryOpts opts))

(defn add-opts?
  [opts]
  (m/validate spec/AddOpts opts))

(defn remove-opts?
  [opts]
  (m/validate spec/RemoveOpts opts))

;; TODO: Availability Opts