(ns missinterpret.storage.provider.s3
  #_(:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+ throw-if-cognitect-anomaly]]
            [missinterpret.storage.protocols.provider :as prot.provider]
            [missinterpret.storage.protocols.block :as prot.block]
            [missinterpret.storage.utils.s3 :as utils.s3]
            [missinterpret.storage.block.core :as block]
            [missinterpret.storage.block.predicate :refer [is-block?]]
            [missinterpret.storage.source.core :as source])
  (:import [java.net URI]))


;; Definitions -------------------------------------------------------------
;;

(def version      "1.0.0")
(def kind         :storage.provider.kind/s3)

(def provider-info #:storage.provider{:kind         kind
                                      :version      version})


;; Support Fns -------------------------------------------------------------
;;

#_(defn s3-config [block]
  (-> (prot.block/id block)
      :storage.block/arguments))


#_(defn available-sources [block source-name]
  (let [s3-config (s3-config block)]
    (reduce
      (fn [coll source]
        (let [uri (:source/uri source)]
          (if (and (some? uri) (utils.s3/uri-exists? s3-config (URI. uri)))
            (conj coll source)
            coll)))
      #{}
      (if (some? source-name)
        (block/sources-by-name block source-name)
        (block/sources-by-kind block kind)))))



#_(defn new-source [{:keys [uri] :as arg}]
  (source/new-source (-> arg
                         (assoc :kind kind)
                         (assoc :uri  (str uri)))))


;; S3 Provider Impl ------------------------------------------------------
;;

#_(defrecord S3Provider [block source-name]
   prot.provider/ContentProvider

   (info [_] provider-info)

   (block [_] {:storage/block block})

  (available-sources [_] (available-sources block source-name))

  (content [this {:storage/keys [source]
                  :storage.action.read/keys [start end] :as args}]
     (let [sources (prot.provider/available-sources this)
           out-source (if (and (some? source) (contains? sources source))
                        source
                        (first sources))]
       (cond
         (nil? out-source)
         (throw+
           {:from     ::perform-action
            :category :anomaly.category/invalid
            :message  {:readable "No content provider source found."
                       :reasons  [:missing.provider/source]
                       :data     {:provider this :argument args}}})
         :else
         (let [{:storage.block/keys [content]} (prot.block/data block)
               read-size (get-in content [:storage/address :address/size])
               read-args (if (or (some? start) (some? end))
                           (cond-> {}
                                   (some? start) (assoc :storage.action.read/start start)
                                   (nil? start)  (assoc :storage.action.read/start 0)
                                   (some? end)   (assoc :storage.action.read/end end)
                                   (or (nil? end) (> end read-size))
                                   (assoc :storage.action.read/end read-size))
                           {})
               uri-string (:source/uri out-source)
               uri (URI. uri-string)
               s3-config (s3-config block)
               s3-args (-> (utils.s3/parse-s3-uri uri)
                           (merge s3-config)
                           (merge read-args))]
           {:storage/source                out-source
            :storage.provider/input-stream (-> (utils.s3/get-object s3-args)
                                               throw-if-cognitect-anomaly
                                               :Body)})))))


;; Factory ---------------------------------------------------------
;;

#_(defn new-provider [{:storage/keys [block]
                     :storage.source/keys [name] :as args}]
  (if (and (block? block) (not-empty (available-sources block name)))
    (merge args {:storage/provider (map->S3Provider {:block block :source-name name})})
    (throw+
      {:from     ::new-provider
       :category :anomaly.category/invalid
       :message  {:readable "Block invalid"
                  :reasons  [:invalid/block]
                  :data     {:argument args}}})))

