(ns missinterpret.storage.store.spec
  (:require [malli.util :as mu]
            [missinterpret.storage.source.spec :as spec.source]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.address.predicate :as pred.addr]
            [missinterpret.storage.utils.file :as utils.file]))

;; Block Store -----------------------------------------------

(def BlockStoreId
  [:map
   [:block-store.id/name                       some?]
   [:block-store.id/uuid                       uuid?]
   [:block-store.id/arguments {:optional true} map?]])

(def FileBlockStoreId
  [:map
   [:block-store.id/name some?]
   [:block-store.id/uuid uuid?]
   [:block-store.id/arguments
    [:map
     [:directory [:fn utils.file/dir-exists?]]]]])

(def MemoryBlockStoreId
  [:map
   [:block-store.id/name some?]
   [:block-store.id/uuid uuid?]
   [:block-store.id/arguments {:optional true}
    [:map
     [:store-atm [:fn #(-> (type %) (= clojure.lang.Atom))]]]]])


;; Content Store ---------------------------------------------

(def ContentStoreId
  [:map
   [:content-store.id/name                       some?]
   [:content-store.id/uuid                       uuid?]
   [:content-store.id/arguments {:optional true} map?]])

(def FileContentStoreId
  [:map
   [:content-store.id/name some?]
   [:content-store.id/uuid uuid?]
   [:content-store.id/arguments
    [:map
     [:directory [:fn utils.file/dir-exists?]]]]])

(def MemoryContentStoreId
  [:map
   [:content-store.id/name some?]
   [:content-store.id/uuid uuid?]
   [:content-store.id/arguments {:optional true}
    [:map
     [:format keyword?]]]])


;; Store ----------------------------------------------------------

(def StoreArgs
  [:map
   [:store/name                     some?]
   [:store/kind    {:optional true} keyword?]
   [:store/version                  string?]
   [:store/arguments
    [:map
     [:block   BlockStoreId]
     [:content ContentStoreId]]]])

(def FlowStoreArgs
  (mu/merge
    StoreArgs
    [:map
     [:store/kind [:fn #(= % :store.kind/flow)]]]))

(def SyncStoreArgs
  (mu/merge
    StoreArgs
    [:map
     [:store/kind [:fn #(= % :store.kind/synchronous)]]]))


;; Runtime Element ------------------------------------------------

(def Element
  [:map
   [:storage/block                     [:fn pred.block/block?]]
   [:storage/provider                  [:fn pred.prov/provider?]]
   [:storage/address  {:optional true} [:fn pred.addr/address?]]])


;; Options --------------------------------------------------------

(def FlowStoreOpts
  [:map
   [:id                            keyword?]
   [:store-prefix {:optional true} boolean?]])


;; Protocol Method Options -----------------------------------------
;; Note:
;;  1. These are all entirely optional.
;;  2. All default-fn are arity-1 which is applied to each block being processed.
;;
;; Query Notes:
;; - The query-fn is arity-1. Its usage depends on the implementing
;;   store types. For the following implementations:
;;
;;   *Memory, file system and s3*
;;   Operate on the collection of all blocks, needs to return a collection
;;
;; - The ordering-fn depends on the implementation but for those listed
;;   above it is an arity-2 function which must be a comparator. See the
;;   guide: https://clojure.org/guides/comparators
;;
(def QueryOpts
  [:map
   [:store.query/default-fn  {:optional true} fn?]
   [:store.query/query-fn    {:optional true} fn?]
   [:store.query/ordering-fn {:optional true} fn?]])


;; Add Notes:
;;  - enforce-1-to-1
;;
(def AddOpts
  [:map
   [:store.add/default-fn                 {:optional true} fn?]
   [:store.add/metadata                   {:optional true} map?]
   [:store.add/metadata-action            {:optional true} keyword?]
   [:store.add/rm-sources                 {:optional true} spec.source/Sources]
   [:store.add/error-when-content-exists? {:optional true} boolean?]])


(def RemoveOpts
  [:map
   [:store.remove/default-fn {:optional true} fn?]])


