(ns missinterpret.storage.provider.file-system
  (:require [clojure.pprint :refer [pprint]]
            [clojure.java.io :as io]
            [missinterpret.anomalies.anomaly :refer [throw+ anomaly]]
            [missinterpret.storage.protocols.provider :as prot.prov]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.source.core :as core.source]
            [missinterpret.storage.utils.core :as utils.core]
            [missinterpret.storage.utils.file :as utils.file])
  (:import (java.io RandomAccessFile)))

;; Definitions -------------------------------------------------------------
;;

(def kind         :provider.kind/file-system)
(def version      "1.0.0")
(def availability :source.availability/file-system)

(def provider-info #:provider.info{:kind    kind
                                   :version version})


;; Support Fns -------------------------------------------------------------
;;

(defn new-source [arguments]
  (let [{:keys [uri content-type ext] :as args} (utils.core/strip-ns arguments)]
    (core.source/new (-> args
                         (assoc :availability availability)
                         (assoc :content-type (if (some? content-type) content-type "data"))
                         (assoc :ext          (if (some? ext) ext "dat"))
                         (assoc :kind         kind)
                         (assoc :uri          (str uri))))))


;; FileSystemProvider Impl ------------------------------------------------------
;;

(defrecord FileSystemProvider [block source-name]
   prot.prov/ContentProvider

   (info [_] provider-info)

   (block [_] {:storage/block block})

   (available-sources [_] (core.block/sources-by-kind block kind))

   (content [this {:storage/keys      [source]
                   :source/keys       [name]
                   :source.range/keys [start end] :as args}]
     (let [sources (core.block/sources-by-kind block kind)
           out-source (core.source/content-source sources source-name args)]
       (let [content-file (-> (:source/uri out-source)
                              utils.core/to-uri
                              utils.file/uri->file)]
         (with-open [raf (RandomAccessFile. content-file "r")]
           (let [len (.length raf)
                 s (if (some? start) start 0)
                 e (if (some? end) end len)
                 diff (- e s)]
             (if (and (<= diff len) (> e s))
               (let [bytes (byte-array diff)]
                 ;; Seek the file pointer to the desired start
                 (when (> s 0) (.seek raf s))
                 (.readFully raf bytes)
                 {:storage/source                out-source
                  :storage/input-stream (io/input-stream bytes)})
               (throw+
                 {:from     ::content
                  :category :anomaly.category/invalid
                  :message  {:readable "Invalid range request"
                             :reasons  [:invalid.provider/range-request]
                             :data     {:arg1     args
                                        :provider this
                                        :file     (str content-file)
                                        :length   len}}}))))))))

;; Factory ---------------------------------------------------------
;;

(defn new [{:storage/keys [block]
            :source/keys [name]
            :keys [throw-when-no-sources?]
            :as args}]
  (when-not (and (pred.prov/provider-args? args)
                 (if (true? throw-when-no-sources?)
                   (seq (core.block/sources-by-kind block kind))
                   true))
    (throw+
      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Invalid provider arguments or no sources available"
                  :reasons  [:or
                             :invalid/provider.block
                             :invalid/provider.no-sources]
                  :data     {:arg1 args}}}))
  (assoc args :storage/provider (map->FileSystemProvider {:block block :source-name name})))

