(ns missinterpret.storage.provider.memory
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.protocols.provider :as prot.prov]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.source.core :as core.source]
            [missinterpret.storage.utils.core :as utils.core]
            [missinterpret.storage.utils.byte-arrays :as utils.bytes]))

;; Definitions -------------------------------------------------------------
;;

(def kind         :provider.kind/memory)
(def version      "1.0.0")
(def availability :source.availability/memory)

(def provider-info #:provider.info{:kind    kind
                                   :version version})


;; Support Fns -------------------------------------------------------------
;;

(defn new-source [arguments]
  (let [{:keys [content-type ext format data] :as args} (utils.core/strip-ns arguments)]
    (-> args
        (assoc :kind         kind)
        (assoc :availability availability)
        (assoc :content-type (if (some? content-type) content-type "data"))
        (assoc :ext          (if (some? ext) ext "dat"))
        (assoc :context       #:source.memory{:format format :data data})
        (dissoc :data)
        (dissoc :format)
        core.source/new)))

;; MemoryProvider Impl -----------------------------------------------------
;;

(defrecord MemoryProvider [block source-name]
   prot.prov/ContentProvider

  (info [_] provider-info)

  (block [_] {:storage/block block})

  (available-sources [_] (core.block/sources-by-kind block kind))

  (content [_ {:storage/keys      [source]
               :source/keys       [name]
               :source.range/keys [start end]
               :as args}]
    (let [sources (core.block/sources-by-kind block kind)
          out-source (core.source/content-source sources source-name args)
          {:source.memory/keys [data format]} (:source/context out-source)
          content (-> (utils.bytes/decode data format) utils.core/to-input-stream)]
      (if (utils.core/input-stream? content)
        ;; Optionally return the subset in the range.
        ;; semantics [)
        (cond-> {:storage/source       out-source
                 :storage/input-stream content}

                (or (some? start) (some? end))
                (assoc :storage/input-stream (-> (utils.core/input-stream->byte-array content)
                                                 (utils.bytes/range-bytes start end)
                                                 utils.core/to-input-stream)))
        {:storage/source out-source
         :storage/data   content}))))


;; Factory -----------------------------------------------------------------
;;

(defn new [{:storage/keys [block]
            :source/keys [name]
            :keys [throw-when-no-sources?]
            :as args}]
  (when-not (and (pred.prov/provider-args? args)
                 (if (true? throw-when-no-sources?)
                   (seq (core.block/sources-by-kind block kind))
                   true))
    (throw+
      {:from     ::new
       :category :anomaly.category/invalid
       :message  {:readable "Invalid provider arguments or no sources available"
                  :reasons  [:or
                             :invalid/provider.block
                             :invalid/provider.no-sources]
                  :data     {:arg1 args}}}))
  (assoc args :storage/provider (map->MemoryProvider {:block block :source-name name})))

