(ns missinterpret.storage.utils.file
  (:require [clojure.pprint :refer [pprint]]
            [clojure.string :as s]
            [clojure.java.io :as io]
            [clojurewerkz.urly.core :as url]
            [missinterpret.anomalies.anomaly :refer [throw+]])
  (:import (java.nio.file Files LinkOption Paths)
           (missinterpret.storage FileUtils)))

(defn cwd-path []
  (str (.getCanonicalPath (io/file ".")) "/"))

(defn file->uri [file]
  (.toURI (io/file file)))

(defn file-uri? [file-uri]
  (= "file" (url/protocol-of file-uri)))

(defn exists? [file]
  (let [f (io/file file)]
    (and (.exists f) (.canRead f) (.isFile f))))

(defn uri-exists? [file-uri]
  (when (and (some? file-uri) (file-uri? file-uri))
    (let [f (-> file-uri
                url/path-of
                io/file)]
      (and (.exists f) (.canRead f) (.isFile f)))))

(defn uri->file
  "Converts the file uri into an io/file. If it does not exist or the uri is not
   a file uri an anomly is thrown."
  [file-uri]
  (cond
    (not (file-uri? file-uri))
    (throw+
      {:from      ::uri->file
       :category  :anomaly.category/unsupported
       :message   {:readable "URI unsupported protocol"
                   :reasons  [:invalid.file/uri]
                   :data     {:argument file-uri}}})

    (not (uri-exists? file-uri))
    (throw+
      {:from      ::uri->file
       :category  :anomaly.category/invalid
       :message   {:readable "File either does not exist, is not readable, or a directory"
                   :reasons  [:invalid/file]
                   :data     {:argument file-uri}}}))
  (-> file-uri
      url/path-of
      io/file))

(defn filename [file]
  (-> file
      io/file
      .getName
      (s/split #"\.")
      first))


(defn ext [file]
  (-> file
      io/file
      .getName
      (s/split #"\.")
      last))

(defn created-on
  "Returns the creation date of the file as an instant"
  [file]
  (FileUtils/createdOn (io/file file)))

(defn size
  [file]
  (let [f (io/file file)
        path (-> f file->uri Paths/get)]
    (Files/size path)))

(defn dir-exists? [d]
  (when (some? d)
    (let [directory (io/file d)]
      (and (= java.io.File (type directory))
           (.exists directory)
           (.isDirectory directory)))))

(defn mkdir [path]
  (let [f (io/file path)]
    (if (.exists f)
      (throw+
        {:from      ::mkdir
         :category  :anomaly.category/conflict
         :message   {:readable (str f " already exists")
                     :reasons  [:conflict/exists]
                     :data     {:arg1 path}}})
      (.mkdir f))))

(defn rmdir [path]
  (let [f (io/file path)]
    (if-not (dir-exists? f)
      (throw+
        {:from      ::rmdir
         :category  :anomaly.category/invalid
         :message   {:readable (str f " does not exist")
                     :reasons  [:invalid/missing]
                     :data     {:arg1 path}}})
      (.delete f))))

(defn subdir
  "Returns a File of a subdirectory validating it is a directory type and exists.
  Throws an anomly otherwise."
  [parent subname]
  (let [d (io/file parent subname)]
    (cond
      (not (.exists d))
      (throw+
        {:from      ::subdir
         :category  :anomaly.category/fault
         :message   {:readable (str "Does not exist: " d)
                     :reasons  [:missing/directory]
                     :data     {:argument d}}})

      (not (.isDirectory d))
      (throw+
        {:from      ::subdir
         :category  :anomaly.category/fault
         :message   {:readable (str "Not a directory: " d)
                     :reasons  [:invalid/directory]
                     :data {:argument d}}})
      :else d)))


