(ns missinterpret.storage.provider.core
  (:require [clojure.pprint :refer [pprint]]
            [clojurewerkz.urly.core :as url]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.protocols.provider :as prot.prov]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.source.predicate :as pred.source]))

(defn provider [prov]
  (let [p (if (and (map? prov) (contains? prov :storage/provider))
            (:storage/provider prov)
            prov)]
    (when-not (pred.prov/provider? p)
      (throw+
        {:from     ::provider
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Provider"
                    :reasons  [:invalid/provider]
                    :data {:arg1 prov :provider p}}}))
    p))

(defn info [prov]
  (let [i (-> (provider prov) prot.prov/info)]
    (when-not (pred.prov/info? i)
      (throw+
        {:from     ::info
         :category :anomaly.category/invalid
         :message  {:readable "Invalid Provider Info"
                    :reasons  [:invalid/provider.info]
                    :data {:arg1 prov :info i}}}))
    i))

(defn block [prov]
  (let [b (-> (provider prov) prot.prov/block)]
    (when-not (pred.block/block? b)
      (throw+
        {:from     ::block
         :category :anomaly.category/invalid
         :message  {:readable "Invalid provider block"
                    :reasons  [:invalid/provider.block]
                    :data {:arg1 prov :block b}}}))
    b))

(defn available-sources [prov]
  (if-not (pred.prov/provider? prov)
    []
    (let [srcs (-> (provider prov) prot.prov/available-sources)]
      (when-not (pred.source/are-sources? srcs)
        (throw+
          {:from     ::available-sources
           :category :anomaly.category/invalid
           :message  {:readable "Invalid provider sources"
                      :reasons  [:invalid/provider.sources]
                      :data {:arg1 prov}}}))
      srcs)))

(defn sources-by-name [p name]
  (cond
    (not (pred.prov/provider? p)) []
    (some? name) (filterv #(= name (:source/name %)) (available-sources p))
    :else
    (throw+
      {:from     ::sources-by-name
       :category :anomaly.category/invalid
       :message  {:readable "Missing source name"
                  :reasons  [:missing.source/name]
                  :data {:argument p}}})))

(defn sources-by-kind [p kind]
  (cond
    (not (pred.prov/provider? p)) []
    (some? kind) (filterv #(= kind (:source/kind %)) (available-sources p))
    :else
    (throw+
      {:from     ::sources-by-kind
       :category :anomaly.category/invalid
       :message  {:readable "Missing source kind"
                  :reasons  [:missing.source/kind]
                  :data {:argument p}}})))

(defn sources-by-uri-match [p {:keys [regex protocol] :as args}]
  (cond
    (not (pred.prov/provider? p)) []

    (and (some? regex) (some? protocol))
    (reduce
      (fn [coll s]
        (let [uri (:source/uri s)]
          (if (and (some? uri) (= protocol (url/protocol-of uri)) (re-find regex uri))
            (conj coll s)
            coll)))
      #{}
      (available-sources p))

    :else
    (throw+
      {:from     ::sources-by-uri-match
       :category :anomaly.category/invalid
       :message  {:readable "Missing source kind"
                  :reasons  [:missing/regex]
                  :data     {:argument args}}})))

(defn content
  ([prov]
   (content prov nil))
  ([prov name]
   (content prov name {}))
  ([prov name args]
   (let [ctx (if (some? name)
               (assoc args :source/name name)
               args)
         cnt (-> (provider prov) (prot.prov/content ctx))]
     (when-not (pred.prov/content? cnt)
       (throw+
         {:from     ::content
          :category :anomaly.category/invalid
          :message  {:readable "Provider content invalid"
                     :reasons  [:invalid/content.data]
                     :data     {:arg1 prov :arg2 name :arg3 args}}}))
     cnt)))

(defn address [p]
  (-> (block p) core.block/address))
