(ns missinterpret.storage.store.core
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.flows.constructors :as flow.cstr]
            [missinterpret.anomalies.anomaly :refer [anomaly]]
            [missinterpret.storage.source.predicate :as pred.source]
            [missinterpret.storage.utils.core :as core.utils]
            [missinterpret.storage.protocols.store :as prot.store]
            [missinterpret.storage.store.internal :as internal]
            [missinterpret.storage.store.predicate :as pred.store]
            [missinterpret.storage.store.predicate-runtime :as pred.store-rt]))

;; Flow -------------------------------------------------------------
;;

(def flow-version "1.0.0")
(def flow-kind    :store.kind/flow)

(def flow-store-info #:store.info{:version  flow-version
                                  :kind     flow-kind})

(defrecord FlowStore [store-id]
  prot.store/Store

  (id [_] store-id)

  (info [_] flow-store-info)

  (query [this {:keys [id store-prefix] :as args}]
    (if-not (pred.store/flow-store-opts? args)
      (anomaly
        ::query
        :anomaly.category/invalid
        {:readable "Invalid argument: missing source or sink stream"
         :reasons  [:missing/stream]
         :data     {:arg1 this :arg2 args}})

      (let [query (internal/query-fn store-id)
            flow-id (if (true? store-prefix)
                      (keyword (str (:store.id/name store-id)) (str id))
                      id)]
        (flow.cstr/fn-flow flow-id query))))

  (add! [this {:keys [id store-prefix] :as args}]
    (if-not (pred.store/flow-store-opts? args)
      (anomaly
        ::add!
        :anomaly.category/invalid
        {:readable "Invalid argument: missing source or sink stream"
         :reasons  [:missing/stream]
         :data     {:arg1 this :arg2 args}})

      (let [add (internal/add-fn store-id)
            flow-id (if (true? store-prefix)
                      (keyword (str (:store.id/name store-id)) (str id))
                      id)]
        (flow.cstr/fn-flow flow-id add))))

  (remove! [this {:keys [id store-prefix] :as args}]
    (if-not (pred.store/flow-store-opts? args)
      (anomaly
        ::remove!
        :anomaly.category/invalid
        {:readable "Invalid argument: missing source or sink stream"
         :reasons  [:missing/stream]
         :data     {:arg1 this :arg2 args}})

      (let [remove (internal/remove-fn store-id)
            flow-id (if (true? store-prefix)
                      (keyword (str (:store.id/name store-id)) (str id))
                      id)]
        (flow.cstr/fn-flow flow-id remove))))

  (availability! [this {:keys [id store-prefix] :as args}]
    (if-not (pred.store/flow-store-opts? args)
      (anomaly
        ::availability!
        :anomaly.category/invalid
        {:readable "Invalid argument: missing source or sink stream"
         :reasons  [:missing/stream]
         :data     {:arg1 this :arg2 args}})

      (let [avail (internal/availability-fn store-id)
            flow-id (if (true? store-prefix)
                      (keyword (str (:store.id/name store-id)) (str id))
                      id)]
        (flow.cstr/fn-flow flow-id avail)))))


;; Synchronous -----------------------------------------------------
;;

(def store-version "1.0.0")
(def store-kind    :store.kind/synchronous)

(def store-info #:store.info{:version  store-version
                             :kind     store-kind})

(defrecord Store [store-id]
  prot.store/Store

  (id [_] store-id)

  (info [_] store-info)

  (query [_ element]
    (let [query (internal/query-fn store-id)] (query element)))

  (add! [_ element]
    (let [add (internal/add-fn store-id)] (add element)))

  (remove! [_ element]
    (let [remove (internal/remove-fn store-id)] (remove element)))

  (availability! [_ element]
    (let [avail (internal/availability-fn store-id)] (avail element))))


;; Factory ---------------------------------------------------------
;;

(defn store-id [name block-store content-store]
  (let [id #:store.id{:name name :uuid (random-uuid) :block block-store :content content-store}]
    (if (not (pred.store-rt/store-id? id))
      (anomaly
        ::id
        :anomaly.category/invalid
        {:readable "Invalid id arguments"
         :reasons  [:or :name :block-store :content-store]
         :data     {:arg1 name
                    :arg2 block-store
                    :arg3 content-store}})
      id)))


(defn new-flow-store
  ([name block-store content-store]
   (-> (store-id name block-store content-store) new-flow-store))
  ([arguments]
    (if (pred.store-rt/store-id? arguments)
      (map->FlowStore {:store-id arguments})
      (anomaly
        ::new-stream-store
        :anomaly.category/invalid
        {:readable "Invalid arguments"
         :reasons  [:invalid/store.id]
         :data     {:argument arguments}}))))


(defn new-store
  ([name block-store content-store]
   (-> (store-id name block-store content-store) new-store))
  ([arguments]
   (if (pred.store-rt/store-id? arguments)
     (map->Store {:store-id arguments})
     (anomaly
       ::new-store
       :anomaly.category/invalid
       {:readable "Invalid arguments"
        :reasons  [:invalid/store.id]
        :data     {:arg1 arguments}}))))


;; Options --------------------------------------------------------
;;

(defn query-options
  "Includes the arguments that can modify a query."
  ([opts]
   (query-options {} opts))
  ([element {:keys [default-fn query-fn ordering-fn] :as opts}]
  (if (and (map? element)
           (or (fn? default-fn) (fn? query-fn) (fn? ordering-fn)))
    (->> (core.utils/upsert-ns opts "store.query")
         (merge element))
    (anomaly
      ::apply-query-options
      :anomaly.category/invalid
      {:readable "Invalid arguments"
       :reasons  [:invalid/options]
       :data     {:arg1 element :arg2 opts}}))))


(defn add-options
  "Includes the arguments that can modify a block during the add operation.

  When metadata is supplied without a metadata-action the action defaults
  to :metadata/merge-first."
  ([opts]
   (add-options {} opts))
  ([element {:keys [default-fn metadata metadata-action rm-sources] :as opts}]
  (if (and (map? element)
           (or (fn? default-fn) (map? metadata) (keyword? metadata-action)
               (and (set? rm-sources) (pred.source/are-sources? rm-sources))))
    (->> (core.utils/upsert-ns opts "store.add")
         (merge element))
    (anomaly
      ::apply-add-options
      :anomaly.category/invalid
      {:readable "Invalid arguments"
       :reasons  [:invalid/options]
       :data     {:arg1 element :arg2 opts}}))))


(defn remove-options
  ([opts]
   (remove-options {} opts))
  ([element {:keys [default-fn allow-content-only
                    omit-rm-block-predicate keep-orig-block-predicate]
             :as opts}]
  (if (and (map? element)
           (or (fn? default-fn) (true? allow-content-only)
               (fn? omit-rm-block-predicate) (fn? keep-orig-block-predicate)))
    (->> (core.utils/upsert-ns opts "store.remove")
         (merge element))
    (anomaly
      ::apply-remove-options
      :anomaly.category/invalid
      {:readable "Invalid arguments"
       :reasons  [:invalid/options]
       :data     {:arg1 element :arg2 opts}}))))

