(ns missinterpret.storage.address.core
  (:require [clojure.pprint :refer [pprint]]
            [clojure.java.io :as io]
            [missinterpret.storage.address.predicate :refer [address?]]
            [clojure.instant :as inst]
            [blocks.core :as block]
            [missinterpret.storage.utils.byte-arrays :as utils.bytes]
            [multiformats.hash :as multihash]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.utils.core :as utils.core]))

(defn byte-array->address
  "Converts bytes into a content addressable representation.

   NOTES:
    - This currently uses a specific addressing function to produce the address info.
    - Algorithm :sha2-256"
  [b]
  (let [blk (block/read! b)
        mh (:id blk)
        address (-> {:id        (str (multihash/hex mh))
                     :algorithm (keyword (:algorithm mh))
                     :size      (str (:size blk))

                     ;; EDN sucks for objects it doesn't already have baked in and it
                     ;; still doesn't have support for Instant. Convert it into a java.util.Date
                     ;; form (i.e. #inst) so convert to a string that can be parsed when read
                     ;;
                     :calculated-at (-> blk
                                        :stored-at
                                        str
                                        inst/read-instant-date
                                        str)}
                    (utils.core/upsert-ns "address"))]
    (if (address? address)
      address
      (throw+
        {:from     ::byte-array->address
         :category :anomaly.category/invalid
         :message  {:readable "Invalid address generated"
                    :reasons  [:invalid/address]
                    :data {:arg1 b
                           :address address}}}))))

(defn data->address
  "Creates an address out of data that can be converted into a byte array"
  [data]
  (let [bytes (if (bytes? data) data (utils.bytes/to-byte-array data))]
    (byte-array->address bytes)))

(defn file->address
  [file]
  (-> (io/input-stream file)
      utils.bytes/to-byte-array
      byte-array->address))