(ns missinterpret.storage.store.content.file-system
  (:require [clojure.pprint :refer [pprint]]
            [clojure.java.io :as io]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.storage.address.predicate :as pred.address]
            [missinterpret.storage.block.core :as core.block]
            [missinterpret.storage.block.predicate :as pred.block]
            [missinterpret.storage.protocols.content-store :as prot.content-store]
            [missinterpret.storage.protocols.provider :as prot.provider]
            [missinterpret.storage.provider.core :as core.prov]
            [missinterpret.storage.provider.file-system :as provider.fs]
            [missinterpret.storage.provider.predicate :as pred.prov]
            [missinterpret.storage.source.predicate :as pred.source]
            [missinterpret.storage.utils.core :as utils.core]
            [missinterpret.storage.utils.file :as utils.file]
            [missinterpret.storage.store.content.core :as core.content-store]
            [missinterpret.storage.provider.core :as provider]
            [missinterpret.storage.store.predicate :refer [file-content-store-id?]])
  (:import (java.nio.file FileSystems)))

;; Definitions -------------------------------------------------------------
;;

(def version "1.0.0")
(def kind    :content-store.kind/file-system)

(def content-store-info #:content-store.info{:version version
                                             :kind    kind})


;; Support Fns ------------------------------------------------------------
;;

(defn store-directory [content-store-id]
  (get-in content-store-id [:content-store.id/arguments :directory]))


(defn source-data
  "Returns a map containing the source of this provider
  in the store along with that sources content file in the storage directory."
  [{:storage/keys [provider block] :as element} content-dir]
  (let [{:address/keys [id] :as addr}
        (cond
          (pred.block/block? block)      (core.block/address block)
          (pred.prov/provider? provider) (core.prov/address provider)
          :else
          (throw+
            {:from    ::source-data
             :category :cognitect.anomalies/conflict
             :message {:readable "No provider or block"
                       :reasons  [:invalid/arguments]
                       :data     {:arg1 element
                                  :arg2 content-dir}}}))
        path-match (re-pattern (str content-dir "/" id))
        sources (if (some? provider)
                  (provider/sources-by-uri-match provider {:regex path-match :protocol "file"})
                  (core.block/sources-by-uri-match block {:regex path-match :protocol "file"}))]
    (if (not= 1 (count sources))
      (throw+
        {:from     ::source-data
         :category :cognitect.anomalies/invalid
         :message {:readable (str (count sources) " sources found - expected 1")
                   :reasons  [:error/source.missing]
                   :data     {:provider provider
                              :content-dir      content-dir
                              :sources          sources}}})
      (let [source (first sources)
            file (-> source :source/uri utils.file/uri->file)]
        {::output-file file
         ::source      source}))))


(defn provider-source
  "Returns the source that matches the content provider for this store."
  [content-store-id element]
  (try
    (-> element
        (source-data (store-directory content-store-id))
        ::source)
    (catch Exception _ nil)))


(defn content-file
  "Returns the file in the content directory which matches the
   address of the provider or nil if it does not exist in the store.

   NOTE: This is wildly inefficient and will fall down hard with
         large numbers of files. Need an index-based solution to scale."
  [content-store-id {:address/keys [id] :as addr}]
  (if-not (some? id)
    nil
    (let [content-dir (store-directory content-store-id)
          glob (str  "glob:" id "*.*")
          grammar-matcher (.getPathMatcher
                            (FileSystems/getDefault)
                            glob)
          matches (->> content-dir
                       io/file
                       file-seq
                       (filter #(.isFile %))
                       (filter #(.matches grammar-matcher (.getFileName (.toPath %))))
                       (mapv #(.getAbsolutePath %)))]
      (-> (first matches)
          io/file))))

(defn element-source
  "Returns a source from a storage asset which has an address that exists
   in this content store.

   Heuristic:
    1. Checks the provider or block for a source from this content store returning it
    2. Constructs a source using the address from the asset in this order:
       a. a storage/address
       b. the block address
       c. the provider address

       Note: In this case the source will have an 'unspecified' content-type"
  [content-store-id {:storage/keys [provider block address] :as element}]
  (if-let [source (provider-source content-store-id element)]
    source
    (loop [addrs [(if (pred.address/address? address) address ::none)
                  (if (pred.block/block? block) (core.block/address block) ::none)
                  (if (pred.prov/provider? provider) (core.prov/address provider) ::none)]]
      (let [addr (first addrs)]
        (cond
          (= ::none addr) (recur (rest addrs))
          (nil? addr)     nil
          :else
          (if-let [file (content-file content-store-id addr)]
            (let [name (:content-store.id/name content-store-id)
                  file-ext (utils.file/ext file)]
              (provider.fs/new-source {:name         name
                                       :content-type "unspecified"
                                       :ext          file-ext
                                       :uri          (utils.file/file->uri file)}))
            (recur (rest addrs))))))))


(defn add-content!
  "Writes the content to the storage directory returning the new
   provider source."
  [content-store-id {:storage/keys [provider] :as element}]
  (when-not (pred.prov/provider? provider)
    (throw+
      {:from ::add-content!
       :reason :anomaly.category/invalid
       :message {:readable "Invalid provider"
                 :reasons  [:invalid/provider]
                 :data     {:arg1 content-store-id :arg2 element}}}))
  (let [content-dir (store-directory content-store-id)
        {:storage/keys [input-stream data source] :as content} (prot.provider/content provider element)
        id (get (provider/address provider) :address/id)
        {:source/keys [ext content-type]} source
        name (:content-store.id/name content-store-id)
        file-ext (if (some? ext) ext "dat")
        file-type (if (some? content-type) content-type "data")
        output-file (io/file content-dir (str id "." file-ext))]
    (cond
      (some? data) (spit output-file data)

      (utils.core/input-stream? input-stream)
      (do
        (io/copy input-stream output-file)
        (.close input-stream))

      :else
      (throw+
        {:from ::add-content!
         :reason :anomaly.category/error
         :message {:readable "Content from provider did not return any data"
                   :reasons  [:error/provider.content]
                   :data     {:arg1 content-store-id :arg2 element :content content}}}))
    (provider.fs/new-source {:name         name
                             :content-type file-type
                             :ext          file-ext
                             :uri          (-> output-file utils.file/file->uri str)})))


(defn remove-content!
  "Removes the content from the file storage directory returning the source
   used."
  [content-store-id {:storage/keys [provider] :as element}]
  (when-not (pred.prov/provider? provider)
    (throw+
      {:from ::remove-content!
       :reason :anomaly.category/invalid
       :message {:readable "Invalid provider"
                 :reasons  [:invalid/provider]
                 :data     {:arg1 content-store-id :arg2 element}}}))
  (let [content-dir (store-directory content-store-id)
        {:keys [::output-file ::source]} (source-data element content-dir)]
    (io/delete-file output-file)
    source))


;; Implementation ---------------------------------------------------------
;;

;; TODO: Documentation - formalize that this interface returns a source for
;;       any content with the id provided by an element for
;;       exists, provider and source
;;
;;  1. Content exists in store, block backing the provider has a source
;;  2. Content exists in store, block backing the provider does not have a source
;;  3. Content exists in store by address, create new provider
;;
(defrecord FileSystemContentStore [content-store-id]
  prot.content-store/ContentStore

  (id [_] content-store-id)

  (info [_] content-store-info)

  (exists? [_ {:storage/keys [provider block address] :as element}]
    (-> (element-source content-store-id element)
        pred.source/source?))

  (provider [this {:storage/keys [provider block address] :as element}]
    (if (pred.source/source? (provider-source content-store-id element))
      (cond
        (pred.prov/provider? provider) provider
        (pred.block/block? block)      (provider.fs/new element))
      (utils.core/when-let* [source (prot.content-store/source this element)
                             addr (core.content-store/address element)]

        (-> {:storage/block (core.block/direct-block addr source)}
            provider.fs/new))))

  (source [_ {:storage/keys [provider block address] :as element}]
    (element-source content-store-id element))

  (add! [_ {:storage/keys [provider] :as element}]
    (add-content! content-store-id element))

  (remove! [_ {:storage/keys [provider] :as element}]
    (remove-content! content-store-id element))

  (availability! [this arguments]
    (throw+
      {:from     ::availability!
       :category :anomaly.category/unavailable
       :message  {:readable "Not implemented"
                  :reasons  [:not-implemented]
                  :data {:arg1 this :arg2 arguments}}})))


;; Factory ---------------------------------------------------------
;;

(defn compatible? [id]
  (file-content-store-id? id))


(defn id [name dir]
  #:content-store.id{:name name
                     :arguments {:directory dir}})

(defn new [id]
  (if (file-content-store-id? id)
    (map->FileSystemContentStore {:content-store-id id})
    (throw+
      {:from     ::new-store
       :category :anomaly.category/invalid
       :message  {:readable "Invalid ID for store"
                  :reasons  [:invalid/content-store.id]
                  :data     {:arg1 id}}})))

