(ns missinterpret.flows.system.flows
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+ wrap-exception anomaly anomaly?]]
            [missinterpret.flows.spec :as spec]
            [missinterpret.flows.workflow :as flows]
            [missinterpret.flows.system.catalog.workflow :as cat.workflow]
            [missinterpret.flows.system.catalog.flow :as cat.flow]
            [missinterpret.flows.system.workflow :as wf]
            [missinterpret.flows.utils :refer [extract opts] :as utils.flows]
            [missinterpret.flows.predicates :refer [workflow-unloaded?]]))

(defn load
  "Retrieves a workflow by id from the workflow catalog and generates a 'loaded' version.
   A workflow in a 'loaded' state means that the workflow has
   a `:workflow/source`, a `:workflow/sink`, and is ready to be passed to `run`.

   [Optionsl Arguments]
   :throw-invalid   When true will return the invalid workflow instead of throwing an exception.
   :return-error    When true any exception is returned with an anomaly and
                    invalid workflows are returned.

   Note
   Options are activated by passing them through the runtime argument map. If any are
   used they are not passed through to other functions."
  [id & {:workflow/keys [return-error] :or {return-error false} :as op}]
  (try
    (let [workflow (-> (cat.workflow/lookup id :throw-missing true)
                       (utils.flows/merge-opts op))
          loaded (wf/try-load (cat.flow/catalog) (cat.workflow/catalog) workflow)
          error {:from     ::load
                 :category :anomaly.category/fault
                 :message  {:readable "Workflow is unable to be loaded"
                            :reasons  [:fault.load/workflow]
                            :data     {:id id :opts op}}}]
      (cond
        (and (workflow-unloaded? loaded) (false? return-error))
        (throw+ error)

        (and (workflow-unloaded? loaded) (true? return-error))
        (anomaly error)

        :else loaded))

    (catch Exception e (if (true? return-error)
                          (wrap-exception e ::load)
                          (throw e)))))


(defn run
  "Executes a loaded workflow returning the results.

   Options:

   This function supports a number of options to control the execution context of a workflow,
   the caching behaviour of the catalog and workflow re-use. These can be useful, among
   others, when:

    o Directly managing the manifold streams in the underlying workflow.
    o Using flows which support caching and re-use (for example, pedestal routes).
    o The configuration context needed to load the workflow is only available at call-time.

   Opts

   Options can be assigned either in the workflow definition or passed. All workflow
   attributes can be passed and when specified will over-ride any attributes from
   it's definition.

   - throw-on-error   When true any exception is returned as an anomaly.

   - new              Returns a new loaded workflow based on the workflow with the id
                      that exists in the catalog. This will always try to load the workflow
                      and over-rides the `:workflow/defer-load` attribute if present.

   - return-n         Returns exactly n results from the workflow or fails if unable to do so
                      when timeouts are specified, otherwise run will hang.

                      NOTES:
                       - If no put/take timeout is set and the workflow stream has a 1-1 relationship
                         between input and output run will hang until the stream has been provided
                         the required input arguments.

   - repeat           When true and `return-n` is specified, `run` will `put!` n times before calling `take`.

   - expand           When true `run` will `put-all!` when the runtime argument is a seq.

   - put-timeout      Sets the amount of time `run` will wait when executing `put!/put-all!`
                      before considering the operation failed.

   - take-timeout     Sets the amount of time `run` will wait when executing `take!`
                      before considering the operation failed."
  [id args & {:workflow/keys [new return-error] :as op}]
  (let [workflow (-> (cat.workflow/lookup id :throw-missing true)
                     (utils.flows/merge-opts op))
        lazy? (true? (:workflow/lazy-load workflow))
        rt-wf (cond
                (true? new) (load id (-> workflow
                                         (extract spec/WorkflowOpts)
                                         (assoc :workflow/force true)))

                (and (workflow-unloaded? workflow) lazy?)
                (let [loaded (load id (-> workflow
                                          (extract spec/WorkflowOpts)
                                          (assoc :workflow/lazy-load false)))]
                  (if (anomaly? loaded)
                    loaded
                    (do
                      (cat.workflow/add! loaded)
                      loaded)))

                (workflow-unloaded? workflow)
                (let [anom {:from     ::run
                            :category :anomaly.category/invalid
                            :message  {:readable "Workflow is not loaded"
                                       :reasons  [:invalid.run/workflow]
                                       :data     {:id id :workflow workflow :opts op}}}]
                  (if (true? return-error)
                    (anomaly anom)
                    (throw+ anom)))

                :else workflow)]
    (cond
      (and (anomaly? rt-wf) (true? return-error)) rt-wf
      (anomaly? rt-wf)                            (throw+ rt-wf)
      :else
      (flows/run rt-wf args))))


