(ns missinterpret.flows.macros
  (:require [clojure.pprint :refer [pprint]]
            [clojure.spec.alpha :as s]
            [malli.core :as m]
            [missinterpret.flows.macros-spec :as spec]
            [missinterpret.anomalies.anomaly :refer [anomaly anomaly? throw+]]))

;; TODO: Important !!!
;;  Any future macros that do not rely on this macro need to come *before*
;;
;; Format of spec conform with changes from defn-spec
;;
;; Single Arity
;; {:name bar,
;;  :docstring "bar function",
;;  :meta {:private true},
;;  :bs [:arity-1
;;       {:args {},
;;        :body [:body [(+ 1 2)]]}]}
;;
;; Multiple Arity
;; {:name bar,
;;  :docstring "bar function",
;;  :meta {:private true},
;;  :bs [:arity-n
;;        {:bodies
;;          [{:args {},
;;            :body [:body [(+ 1 2)]]}
;;
;;        {:args {:args [[:sym a] [:sym b]]},
;;          :body [:body [(+ a b)]]}]}]}
;;
(defn update-body [{:keys [bs]:as conf} body-update-fn]
  (cond
    (= (first bs) :arity-1)
    (let [body-location [:bs 1 :body 1]
          args (get-in conf [:bs 1 :args])]
      (update-in conf body-location (partial body-update-fn args)))

    (= (first bs) :arity-n)
    (let [bodies (get-in conf [:bs 1 :bodies]),
          new-bodies
          (mapv (fn [body]
                  (let [body-location [:body 1]
                        args (:args body)]
                    (update-in body body-location (partial body-update-fn args))))
                bodies)]
      (assoc-in conf [:bs 1 :bodies] new-bodies))))


;; macro that adds support for argument
;; validation. Performs the following checks:
;;
;;  1. Wraps body with a try-catch
;;     a. If ex-data does not match the input schema and anomaly is returned
;;        (TODO: It would be nice to be able to also throw an exception
;;               and for it to be 'turned off' -- via meta parsing... see schema above
;;     b. The body is wrapped in a let of the output of evaluation
;;        and if the output value matches the output schema it is passed
;;        through.
;;

;; https://www.codementor.io/@yehonathansharvit/how-to-write-custom-defn-macros-with-clojure-spec-supk887tw

(defn argument-invalid? [schema arg skip]
  (and
    (not (true? skip))
    (some? schema) (not (m/validate schema arg))))

(defn unpack-validate-args [name rt-args in-schema throw skip]
  (let [all-args (map #(last %) (get-in rt-args [:args]))]
    (loop [args all-args
           idx 0]
      (let [arg (first args)
            schema (cond
                     (not (vector? in-schema))  nil
                     (>= idx (count in-schema)) nil
                     :else
                     (nth in-schema idx))]
        (cond
          (nil? arg)     all-args
          (anomaly? arg) arg

          (argument-invalid? schema arg skip)
          (cond
            (true? throw)
            (throw+
              {:from     (keyword name)
               :category :anomaly.category/error
               :message {:readable "Input does not match schema"
                         :reasons  [:error.schema/input]
                         :data     {:arguments all-args
                                    :schema/in in-schema
                                    :schema/throw throw
                                    :idx idx}}})
            :else
            (anomaly
              (keyword name)
              :anomaly.category/error
              {:readable "Input does not match schema"
               :reasons  [:error.schema/input]
               :data     {:arguments all-args
                          :schema/in in-schema
                          :schema/throw throw
                          :idx idx}}))
          :else
          (recur (rest args) (inc idx)))))))

(defn validate
  "Validates that the input arguments to the function
   by matching any provided schema to its input arguments
   and output as well as pipelining anomalies.

   Arguments are passed via function metadata:

    :schema/in [s1 s2 ...]  Schema are matched against the args in order of appearance
    :schema/out s           Output Schema

    :schema/throw?

     By Default error conditions, including Exceptions, are returned as anomalies.

     When `true`, validation errors are thrown as exceptions and exceptions are allowed
     to escape the function scope.

    :schema/skip? true  - Validation skipped; anomalies pipelined"
  [name {:schema/keys [in out throw? skip?] :as meta} args body]
  `((let [body-fn#
          (fn []
            (let [~'validated-arguments (unpack-validate-args ~name ~args ~in ~throw? ~skip?)]
              (if (anomaly? ~'validated-arguments)
                ~'validated-arguments
                (let [output# (do ~@body)]
                  (if (and (not (true? ~skip?))
                           (some? ~out) (not (m/validate ~out output#)))
                    (cond
                      (anomaly? output#) output#

                      (true? ~throw?)
                      (throw+
                        {:from     (keyword ~name)
                         :category :anomaly.category/error
                         :message {:readable "Output does not match schema"
                                   :reasons  [:fault.schema/output]
                                   :data     {:arguments ~args
                                              :output output#
                                              :schema/out ~out
                                              :schema/throw ~throw?}}})
                      :else
                      (anomaly
                        (keyword ~name)
                        :anomaly.category/error
                        {:readable "Output does not match schema"
                         :reasons  [:error.schema/output]
                         :data     {:arguments ~args
                                    :output output#
                                    :schema/out ~out
                                    :schema/throw ~throw?}}))
                    output#)))))]
      (if (true? ~throw?)
        (body-fn#)
        (try
          (body-fn#)
          (catch Exception e#
            (let [data# (ex-data e#)
                  msg# (.getMessage e#)]
              (cond
                (anomaly? data#) data#

                :else
                (anomaly
                  (keyword ~name)
                  :anomaly.category/fault
                  {:readable "Exception thrown"
                   :data {:ex        e#
                          :msg       msg#
                          :arguments (map #(last %) (get-in ~args [:args]))}})))))))))

(defmacro defn* [& args]
  (let [{:keys [name meta] :as conf} (s/conform ::spec/defn-args args)
        new-conf (update-body conf (partial validate (str name) meta))
        new-args (s/unform ::spec/defn-args new-conf)]
    (cons `clojure.core/defn new-args)))
