(ns missinterpret.flows.system.state
  (:require [clojure.pprint :refer [pprint]]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.predicates :as p]
            [missinterpret.flows.system.workflow :as wf]
            [missinterpret.flows.pedestal.workflow :as p.wf]))

(def state-atom (atom {:flow.catalog/loaded {}
                       :workflow.catalog/loaded {}}))

(defn load-flow-catalog [definitions]
  (reduce
    (fn [coll f]
      (let [id (:flow/id f)]
        (cond
          (not (p/flow? f))
          (throw+
            {:from     ::load-flow-catalog
             :category :anomaly.category/invalid
             :message  {:readable "Invalid flow definition"
                        :reasons  [:fault.flow/invalid]
                        :data     {:flow f
                                   :flow-defs definitions}}})

          (contains? coll id)
          (throw+
            {:from     ::load-flow-catalog
             :category :anomaly.category/conflict
             :message  {:readable "Flow already exists in catalog"
                        :reasons  [:fault.start/duplicate-flow]
                        :data     {:flow f
                                   :flow-defs definitions}}})
          :else
          (assoc coll id f))))
    {}
    definitions))


(defn load-wf-catalog
  ([flow-catalog definitions]
   (load-wf-catalog flow-catalog definitions {}))
  ([flow-catalog definitions args]
   ;; 1. Constructs the catalog skeleton by checking
   ;;    each workflow in the definitions for integrity.
   (let [skel-catalog (wf/wf-skeleton definitions)]
     ;; 2. Loads all workflows which are not marked as
     ;;    lazy or defer. Validates they are loaded.
     (reduce
       (fn [loaded-catalog [id wf]]
         (let [loaded (wf/load flow-catalog skel-catalog wf args)]
           (assoc loaded-catalog id loaded)))
       {}
       skel-catalog))))


(defn start
  "Start function used by both mount and component.
   It stores the atom that is used by the flow and workflow
   catalogs.

   Flows and workflows are validated and workflows which
   are not marked as lazy-load or defer-load are loaded.

  Note: Loading Order
   1. Compile Time: Flows pedestal routes are processed; Workflows are not loaded
   2. System startup: Flows and all workflows are validated, are added to the catalogs,
                      and the result is stored to state-atom."
  [{:flows.workflow.catalog/keys [definitions args]
    :flows.flow.catalog/keys [definitions]
    :flows.pedestal/keys [routes]
    :as flows-definition}]
  (let [flow-defs (:flows.flow.catalog/definitions flows-definition)
        flow-catalog (load-flow-catalog flow-defs)
        wf-defs (:flows.workflow.catalog/definitions flows-definition)
        base-catalog (load-wf-catalog flow-catalog wf-defs args)
        workflow-catalog (if (set? routes)
                           (reduce
                             (fn [coll r]
                               (let [wf (p.wf/route-workflow r)]
                                 (cond
                                   ;; If the route embeds a keyword instead of a
                                   ;; workflow validates that it exists.
                                   (keyword? wf)
                                   (if-not (get base-catalog wf)
                                     (throw+
                                       {:from     ::start
                                        :category :anomaly.category/invalid
                                        :message  {:readable "Workflow failed to load when not marked defer or lazy"
                                                   :reasons  [:fault.start/unloaded]
                                                   :data     {:workflow wf
                                                              :base-catalog base-catalog
                                                              :routes routes
                                                              :flow-catalog flow-catalog}}})
                                     coll)

                                   :else
                                   (do
                                     (wf/throw-wf-invalid base-catalog wf)
                                     (assoc coll (:workflow/id wf)
                                                 (wf/load flow-catalog coll wf args))))))
                             base-catalog
                             routes)
                          base-catalog)]

    (if (and (p/flow-catalog? flow-catalog) (p/workflow-catalog? workflow-catalog))
      (let [new-state (-> flows-definition
                          (assoc :flow.catalog/loaded flow-catalog)
                          (assoc :workflow.catalog/loaded workflow-catalog))]
        (reset! state-atom new-state)
        @state-atom)
      (throw+
        {:from     ::start
         :category :anomaly.category/fault
         :message  {:readable "Invalid flow or workflow catalog"
                    :reasons  [:fault.state.start/invalid-catalogs]
                    :data     {:flows/definition flows-definition
                               :flow.catalog/loaded flow-catalog
                               :workflow.catalog/loaded workflow-catalog}}}))))


(defn dissoc-in!
  "Removes a workflow from the catalog"
  [path id]
  (swap! state-atom update-in path dissoc id))


(defn reset-state!
  "Resets the runtime state unloading all entities from the catalogs."
  []
  (reset! state-atom {:flow.catalog/loaded {}
                      :workflow.catalog/loaded {}}))