(ns missinterpret.flows.utils
  (:require [clojure.pprint :refer [pprint]]
            [malli.core :as m]
            [malli.util :as mu]
            [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.spec :as spec]))

;; Map Fns -----------------------------------------------------------

(defn merge-if
  "Slightly safer version of merge. Merges b into a if they are both maps.
   Caution - does not throw an exception if they aren't."
  [a b]
  (if (and (map? a) (map? b)) (merge a b) a))

(defn keys [schema]
  (when (some? schema)
    (->> (mu/keys schema)
         (into #{}))))

(defn exclude
  "Returns a map which excludes all keywords from the malli schema."
  [m schema]
  (let [keys (mu/keys schema)]
    (loop [schema-keys keys
           dissoc-map m]
      (let [k (first schema-keys)]
        (if (nil? k)
          dissoc-map
          (recur (rest schema-keys) (dissoc dissoc-map k)))))))


(defn extract
  "Returns a map of all keywords from the malli schema"
  [m schema]
  (let [keys (mu/keys schema)]
    (reduce
      (fn [coll k]
        (if (contains? m k)
          (assoc coll k (get m k))
          coll))
      {}
      keys)))


(defn only?
  "Does the argument map only contain the keys of the given schema?"
  [m schema]
  (= (->> m keys (into #{})) (->> (mu/keys schema) (into #{}))))


(defn either
  "Returns x if not nil, default otherwise."
  [x default]
  (if (some? x)
    x
    default))


(defn opts
  "Converts a seq into a map.

   Intended for options to a function. i.e. [a b & opts]
   Handles nested options. i.e. [[:a :A]]

  Throws
   - If arg is not a seq
   - If 'root' vector does not conform to missinterpret.flows.spec/Options
     spec."
  [o]
  (cond

    (nil? o) {}

    ;; Maps are passed. i.e. {:a :A}
    (map? o) o

    ;; Wrapped maps are passed i.e. [{:a :A}]
    (map? (first o)) (first o)

    ;; Nested? i.e. [[:a :A]]
    (and (= 1 (count o)) (sequential? (first o)))
    (opts (first o))

    ;; conform to spec?
    (m/validate spec/Options o) (apply hash-map o)

    :else {}))

(defn merge-opts [m o]
  (if (some? o)
    (->> (opts o)
         (merge m))
    m))

(defn set-ns
  "Sets the namespace of the top-level keys of the map to the argument ns."
  [m nspace & {:keys [keep-qualified]}]
  (reduce
    (fn [coll [k v]]
      (if (and (true? keep-qualified) (namespace k))
        (assoc coll k v)
        (let [n (name k)]
          (assoc coll (keyword nspace n) v))))
    {}
    m))


;; Fns -----------------------------------------------------------

(defn take-range
  [coll range]
  (let [start (if (integer? range)
                range
                (first range))
        end (when (vector? range)
              (last range))
        v (vec coll)]
    (if (integer? end)
      (subvec v start end)
      (subvec v start))))


(defn try-put-all!
  "Takes a sequence argument and attempts to put
   each. Matches the semantics of stream/try-put!"
  [sink arg-seq timeout timeout-val]
  (if (not (and (s/stream? sink) (some? arg-seq)
                (integer? timeout) (some? timeout-val)))
    (throw+
      {:from     ::try-put-all!
       :category :anomaly.category/invalid
       :message  {:readable "Invalid arguments"
                  :reasons  [:invalid/arguments]
                  :data     {:arg1 sink :arg2 arg-seq :arg3 timeout :arg4 timeout-val}}})

    (loop [args arg-seq]
      (let [arg (first args)]
        (cond
          (nil? arg) true

          (= timeout-val @(s/try-put! sink arg timeout timeout-val))
          timeout-val

          :else (recur (rest args)))))))


(defn safe-log
  [log-fn message]
  (when (fn? log-fn)
    (try
      (log-fn message)
      (catch Exception _ nil))))