(ns missinterpret.flows.system.catalog.flow
  (:require [missinterpret.flows.system.state :refer [state-atom] :as s]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.utils :refer [opts]]))

;; Editing Fns ----------------------------------------------------------
;;

(defn catalog []
  (get @state-atom :flow.catalog/loaded))


(defn lookup [id & {:keys [throw-missing] :as op}]
  (let [f (get-in @state-atom [:flow.catalog/loaded id])]
    (if (and (true? throw-missing) (nil? f))
      (throw+
        {:from     ::lookup
         :category :anomaly.category/invalid
         :message  {:readable (str id  " not in catalog")
                    :reasons  [:invalid/id]
                    :data     {:id id :opts op}}})
      f)))


(defn lookup-in [id path & {:keys [throw-missing] :as op}]
  (let [f (get-in @state-atom (concat [:flow.catalog/loaded id] path))]
    (if (and (true? throw-missing) (nil? f))
      (throw+
        {:from     ::lookup-in
         :category :anomaly.category/invalid
         :message  {:readable (str id " not in catalog")
                    :reasons  [:invalid/id]
                    :data     {:id id :opts op}}})
      f)))


(defn add!
  "Upserts a workflow to the catalog"
  [id value & {:keys [insert] :as op}]
  (let [insert? (-> (opts op) :insert true?)]
    (if (and insert? (some? (lookup id)))
      (throw+
        {:from     ::add!
         :category :anomaly.category/conflict
         :message  {:readable "Duplicate flow encountered that conflicts with catalog"
                    :reasons  [:invalid/flow]
                    :data     {:id id
                               :value value
                               :flow.catalog/value (lookup id)
                               :opts op}}})
      (swap! state-atom assoc-in [:flow.catalog/loaded id] value))))


(defn remove!
  "Removes a workflow from the catalog"
  [id]
  (s/dissoc-in! [:flow.catalog/loaded id]))




