(ns missinterpret.flows.core
  (:require [clojure.pprint :refer [pprint]]
            [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [throw+]]
            [missinterpret.flows.predicates :refer [flow? workflow? workflow-loaded? flow-catalog?]]
            [missinterpret.flows.utils :refer [try-put-all!]]
            [missinterpret.flows.xflow :as xf]))

;; Catalog ----------------------------------------------------------------

(defn lookup-flow
  "Looks up a flow from the catalog by its :flow/id"
  [catalog id & {:keys [throw-missing] :or {throw-missing false}}]
  (if (flow-catalog? catalog)
    (let [flow (get catalog id)]
      (cond
        (and (nil? flow) (true? throw-missing))
        (throw+
          {:from     ::lookup-flow
           :category :anomaly.category/unavailable
           :message  {:readable (str id " not found")
                      :reasons  [:invalid.flow/id]
                      :data     {:flow-catalog catalog
                                 :id id}}})

        (and (some? flow) (not (flow? flow)))
        (throw+
          {:from     ::lookup-flow
           :category :anomaly.category/fault
           :message  {:readable (str id " not a valid flow")
                      :reasons  [:invalid/flow]
                      :data     {:flow-catalog catalog
                                 :flow flow
                                 :id id}}})
        :else flow))
    (throw+
      {:from     ::lookup-flow
       :category :anomaly.category/fault
       :message  {:readable "Not a valid flow catalog"
                  :reasons  [:invalid.flow/catalog]
                  :data     {:flow-catalog catalog
                             :id id}}})))


;; Flow  ---------------------------------------------------------

(defn default-flow
  "Returns a flow map which wraps the provided `default-fn` to decorate
   the argument (as a map) with map values returned by the function."
  [{:flow/keys [id default-fn]}]
  (let [flow-id (-> id name (str "-default-fn") keyword)]
    (xf/xflow flow-id (xf/default default-fn))))


(defn fn-flow
  "Returns a flow map for a fn which wraps its execution to conform to the
   stream model."
  [fn-arg & {:flow/keys [expand id] :or {expand false id nil} :as opts}]
  (let [flow-id (keyword (if (some? id) id (str fn-arg)))]
    (if (true? expand)
      (xf/xflow flow-id (xf/fn-expand fn-arg))
      (xf/xflow flow-id (xf/->fn fn-arg)))))


(defn flowify
  "Converts the argument into an appropriate flow.

   - flow map: pass-through
   - fn: returns a fn-flow (optionally uses result pipelining)
   - single arg: lookup flow in catalog"
  [flow-catalog arg & {:flow/keys [expand] :or {expand false} :as op}]
  (cond
    (flow? arg)     arg
    (workflow? arg) arg
    (fn? arg)       (fn-flow arg :flow/expand expand)
    :else           (lookup-flow flow-catalog arg)))


;; Workflow ------------------------------------------------------------

(defn validate-topology
  "Checks that the entries of the workflow definition to be processed by `workflow-factory`
   is a seq, isn't empty, has a recognized type and exists in the catalog when defined by id."
  [flow-catalog definition]
  (let [validation
        (when (seq definition)
          (map
            (fn [x]
              (if (or (flow? x) (fn? x)
                      (and (workflow? x) (workflow-loaded? x))
                      (and (keyword? x) (lookup-flow flow-catalog x)))
                true
                x))
            definition))]
    (cond
      (nil? validation)         {:valid false :invalid [:empty-definition]}
      (every? true? validation) {:valid true}
      :else
      {:valid false
       :invalid (filterv #(not (boolean? %)) validation)})))


(defn linear-order
  [flow-catalog definition & {:keys [expand omit-leave] :or {expand false omit-leave false} :as opts}]
  (->> definition
       (map #(flowify flow-catalog % :expand expand))
       (reduce
         (fn [coll f]
           (cond-> coll
             (contains? f :flow/default-fn) (conj (default-flow f))
             true                           (conj f)
             (contains? f :flow/leave-fn)   (conj (fn-flow f))))
         [])
       (filterv some?)))


;; http://pedestal.io/pedestal/0.7/guides/what-is-an-interceptor.html#_transition_from_enter_to_leave
;;
;; Calls leave in reverse order passing the context from the last enter fn
(defn interceptor-order [flow-catalog definition & {:keys [expand] :or {expand false} :as opts}]
  ;; 2 passes - first one through omitting leave-fn's then second one
  ;;  adds them at the end
  ;;
  ;;   - Note, not sure how to pass the opts correctly
  ;;   - I don't think this is quite right....
  (let [pass1 (linear-order flow-catalog definition :omit-leave true)]
    (reduce
      (fn [coll flow]
        (let [leave-fn (:flow/leave-fn flow)]
          (if (fn? leave-fn)
            (conj coll (fn-flow leave-fn))
            coll)))
      pass1
      (reverse pass1))))


(defn connect-node
  "Connects the sink of one flow as the source of the next
   updating the workflow.

   Note: This operates on the inverse of the convention
         for streams. i.e. source "
  [workflow node args]
  (try
    (let [source-for-fn (:workflow/source workflow)]
      (if (workflow? node)
        (let [sink (:workflow/sink node)
              source (:workflow/source node)]
          (s/connect source-for-fn sink {:description (:workflow/id node)})
          (assoc workflow :workflow/source source))
        ;; The flow-fn takes args and a sink
        ;; returning its internal source
        (let [flow-fn (:flow/fn node)
              fn-src (flow-fn args source-for-fn)]
          (assoc workflow :workflow/source fn-src))))
    (catch Exception _ #:workflow{:invalid [node]})))


;; Run -----------------------------------------------------------------------

(defn invoke
  "Operates on the sink/source to take up to n results from the
   workflow stream. always returns a seq of values.

   Options
    - return-n     Returns exactly n results from the workflow or fails if unable to do so
                   when timeouts are specified, otherwise run will hang.

                   NOTES:
                    - If no put/take timeout is set and the workflow stream has a 1-1 relationship
                      between input and output run will hang until the stream has been provided
                      the required input arguments.

    - expand       If the runtime argument is a seq, put-all instead of put is used.

    - repeat       When return-n is greater than 1 and expand is false, will call put with the runtime args
                   n times before calling take.

    - put-timeout  Will wait X milliseconds before throwing an anomaly during put operations.

    - take-timeout Will wait X milliseconds before throwing an anomaly during take operations."
  ([workflow]
   (invoke workflow {}))
  ([{:workflow/keys [return-n put-timeout take-timeout expand repeat sink source]
    :or {return-n 1 expand false repeat false} :as workflow}
   rt-arg]

   ;; Put ---------------------------------------------------
   ;;  TODO: Finish implementation and tests. Current will always throw if called
   ;;
   (doseq [n (if (and (> return-n 1) (false? expand) (true? repeat))
               (range return-n)
               [0])]
     (if (integer? put-timeout)
       (if (true? expand)
         (when true
           #_(= ::timeout (try-put-all! sink rt-arg put-timeout ::timeout))
           (throw+ {:from     ::invoke
                    :category :anomaly.category/busy
                    :message  {:readable "Put timed out"
                               :reasons  [:fault.run/busy]
                               :data     {:workflow workflow
                                          :rt-arg rt-arg}}}))
         (when true
           #_(= ::timeout @(s/try-put! sink rt-arg put-timeout ::timeout))
           (throw+ {:from     ::invoke
                    :category :anomaly.category/busy
                    :message  {:readable "Put timed out"
                               :reasons  [:fault.run/busy]
                               :data     {:workflow workflow
                                          :rt-arg rt-arg}}})))
       (if (true? expand)
         (s/put-all! sink rt-arg)
         (s/put! sink rt-arg))))

  ;; Take ----------------------------------------------
  (reduce
    (fn [coll _]
      (if (integer? take-timeout)
        ;;  TODO: Finish implementation and tests. Current will always throw if called
        (let [result ::timeout #_@(s/try-take! source ::drained take-timeout ::timeout)]
          (if (= result ::timeout)
            (throw+ {:from     ::invoke
                     :category :anomaly.category/busy
                     :message  {:readable "Take timed out"
                                :reasons  [:fault.run/busy]
                                :data     {:workflow workflow
                                           :rt-arg rt-arg}}})
            (conj coll result)))

        (conj coll @(s/take! source))))
    []
    (range return-n))))




