(ns missinterpret.flows.xflow
  (:require [manifold.stream :as s]
            [missinterpret.anomalies.anomaly :refer [anomaly anomaly? wrap-exception]]
            [missinterpret.anomalies.macros :refer [defn]]))

(defn default
  "Returns a callback fn compatible with xflow which merges the results of the `default-fn` into
   the argument."
  [default-fn]
  (fn [sink element]
    (if (anomaly? element)
      element
      (try
        (let [result (default-fn element)]
          (if (map? result)
            (->> (merge result element)
                 (s/put! sink))
            (anomaly
              ::default
              :anomaly.category/fault
              {:readable "Default fn did not return a map"
              :data {:element element
                     :default-fn default-fn}})))
        (catch Exception e (s/put! sink (wrap-exception e ::default-call)))))))


(defn ->fn [fn-arg]
  "Returns a callback fn compatible with xflow which adds the result of `fn-arg` to the sink via `put!`.
   If an exception is thrown it is wrapped as an anomaly and is returned. `core/invoke` will re-throw."
  (fn [sink element]
    (if (anomaly? element)
      element
      (try
        (let [result (fn-arg element)]
          (s/put! sink result))
        (catch Exception e (s/put! sink (wrap-exception e ::->fn-call)))))))


(defn fn-expand [fn-arg]
  "Returns a callback fn compatible with xflow which adds the result of `fn-arg` to the sink.
   Supports a single result or a sequence of values which are expanded by being added to
   the sink via `put-all!`"
  (fn [sink element]
    (if (anomaly? element)
      element
      (try
        (let [result (fn-arg element)]
          (if (sequential? result)
            (s/put-all! sink result)
            (s/put! sink result)))
        (catch Exception e (s/put! sink (wrap-exception e ::fn-expand-call)))))))


(defn xflow
  "Returns a flow using `connect-via` with the passed xform functions
   which must have arity 2: [sink element] and return a function
   with the same arity.

    Optionally will add the default-fn."
  ([id xform]
   #:flow{:id id
          :fn (fn [_ sink]
                (let [source (s/stream)]
                  (s/connect-via
                    sink
                    (partial xform source)
                    source
                    {:description id})
                  source))})
  ([id xform default-fn]
   (cond-> (xflow id xform)
           (fn? default-fn) (assoc :flow/default-fn default-fn))))

