(ns missinterpret.edn-io.edn
  (:require [clojure.java.io :as io]
            [missinterpret.anomalies.anomaly :refer [anomaly throw+ wrap-exception]]
            [clojure.edn :as edn]
            [clojure.pprint :refer [pprint]]
            [missinterpret.edn-io.data-readers :as readers]
            [clojure.string :as s])
  (:import (java.io InputStreamReader PushbackReader)))

(defn- input-stream?
  [in]
  (-> in type str (s/ends-with? "InputStream")))


(defn read-input-stream
  "Reads and parses edn data via an input stream. If argument
   is not already a stream, converts."
  [i throw-on-error]
  (try
    (if (input-stream? i)
      (with-open [r (InputStreamReader. i)]
        (edn/read {:readers readers/tags} (PushbackReader. r)))
      (with-open [istream (io/input-stream i)
                  r (InputStreamReader. istream)]
        (edn/read {:readers readers/tags} (PushbackReader. r))))

    (catch Exception e
      (do
        (println "read-input-stream> EXCEPTION ======================")
        (pr-str e)
        (let [wrapped (wrap-exception e ::read-string)]
          (if (true? throw-on-error) (throw+ wrapped) wrapped))))))


(defn read
  "Reads edn data from:

    - String
    - Input stream or anything that can be converted to an input stream
    - Map argument with keys: data, input-stream
      If the data argument is passed, the function passes it through."
  [in & {:keys [throw-on-error]}]
  (let [{:storage/keys [data input-stream]} in]
    (cond
      (nil? in)
      (let [from ::read-string
            category :anomaly.category/invalid
            msg {:readable "Nil argument" :reasons [:invalid/input]}]
        (if (true? throw-on-error)
          (throw+
            {:from     from
             :category category
             :message  msg})
          (anomaly from category msg)))

      (string? in)
      (read-input-stream (.getBytes in) throw-on-error)

      (some? data)
      data

      (some? input-stream)
      (read-input-stream input-stream throw-on-error)

      :else (read-input-stream in throw-on-error))))
