(ns representations.schema.v0.common
  (:require
   [clojure.string :as str]
   [representations.util.malli.registry :as mr]
   [representations.util.malli.common :as mc]))

(mr/def ::ref
  [:and
   {:description "Reference to another representation by name (ref:name)"}
   ::mc/non-blank-string
   [:re #"^ref:[a-zA-Z0-9][a-zA-Z0-9-_]*$"]])

(mr/def ::env-var
  [:and
   {:description "Environment variable reference (env:VAR_NAME)"}
   ::mc/non-blank-string
   [:re #"^env:.+$"]])

(mr/def ::ref-or-string
  [:or
   {:description "Either a ref: to another representation, or a plain string"}
   ::ref
   ::mc/non-blank-string])

(mr/def ::env-or-string
  [:or
   {:description "Either an env: variable reference, or a plain string"}
   ::env-var
   ::mc/non-blank-string])

(mr/def ::display-name
  [:and
   {:description "Human-readable name"}
   ::mc/non-blank-string])

(mr/def ::description
  [:and
   {:description "Documentation explaining the purpose"}
   [:or :nil :string]])

(mr/def ::query
  [:or
   {:description "Query - either a native SQL query string or MBQL/lib query map"}
   ::mc/non-blank-string
   ;; TODO: vector? map? the following represents mbql:
   :any])

(mr/def ::database
  [:or
   {:description "Database reference - either ref:database-name or a plain string"}
   ::ref
   ::mc/non-blank-string])

(mr/def ::collection
  [:and
   {:description "Optional collection path for organizing"}
   any?])

(defn ref?
  "Returns true if x is a reference string (ref:name) with a non-empty name."
  [x]
  (and (string? x)
       (str/starts-with? x "ref:")
       (> (count x) 4)))

(defn env?
  "Returns true if x is an environment variable reference (env:VAR_NAME) with a non-empty var name."
  [x]
  (and (string? x)
       (str/starts-with? x "env:")
       (> (count x) 4)))

(defn unref
  "Strips the 'ref:' prefix from a reference string.
   Returns the name portion, or nil if x is not a ref."
  [x]
  (when (ref? x)
    (subs x 4)))

(defn resolve-env-var
  "Resolves an environment variable reference to its value.
   Returns the value from the environment, or nil if x is not an env var or the var is not set."
  [x]
  (when (env? x)
    (System/getenv (subs x 4))))
