(ns representations.schema.v0.column
  (:require
   [clojure.string :as str]
   [malli.core :as m]
   [representations.util.malli.registry :as mr]
   [representations.util.malli.common :as mc]))

(mr/def ::column-name
  [:and
   {:description "Column name as returned by the query"}
   ::mc/non-blank-string])

(mr/def ::display-name
  [:and
   {:description "Human-friendly display name for the column"}
   ::mc/non-blank-string])

(mr/def ::column-description
  [:and
   {:description "Documentation explaining the column's meaning"}
   [:or
    :nil
    :string]])

(defn normalize-type-string
  "Convert type strings to internal keyword format.
   Expects strings like 'type/Text' or 'type/PK' and converts to :type/Text, :type/PK.
   Also handles already-keywordized values."
  [type-str]
  (when type-str
    (cond
      (keyword? type-str) type-str

      (string? type-str)
      (let [trimmed (str/trim type-str)]
        (cond
          (str/starts-with? trimmed ":")
          (keyword (subs trimmed 1))

          (str/includes? trimmed "/")
          (keyword trimmed)

          :else
          (keyword "type" trimmed)))

      :else nil)))

(mr/def ::base-type
  [:and
   {:description "The actual data type of the column (e.g., Text, Integer, DateTime)"
    :decode/json normalize-type-string}
   [:or
    :string
    :keyword]
   [:fn
    {:error/message "Must be a valid base type (not a semantic type)"
     :error/fn (fn [{:keys [value]} _]
                 (str "Not a valid base type: " (pr-str value)))}
    (fn [x]
      (let [type-kw (if (keyword? x) x (normalize-type-string x))]
        (and type-kw
             (isa? type-kw :type/*))))]])

(mr/def ::effective-type
  [:and
   {:description "How Metabase should treat this column (can override base_type)"
    :decode/json normalize-type-string}
   [:or
    :string
    :keyword]
   [:fn
    {:error/message "Must be a valid effective type"
     :error/fn (fn [{:keys [value]} _]
                 (str "Not a valid effective type: " (pr-str value)))}
    (fn [x]
      (let [type-kw (if (keyword? x) x (normalize-type-string x))]
        (and type-kw
             (isa? type-kw :type/*))))]])

(mr/def ::semantic-type
  [:and
   {:description "Semantic meaning of the column (e.g., Email, Currency, Entity Key)"
    :decode/json normalize-type-string}
   [:or
    :nil
    :string
    :keyword]
   [:fn
    {:error/message "Must be a valid semantic type"
     :error/fn (fn [{:keys [value]} _]
                 (let [normalized (normalize-type-string value)]
                   (str "Not a recognized semantic type: " (pr-str value)
                        ". Got: " normalized
                        " which is not a :Semantic/* or :Relation/* type.")))}
    (fn [x]
      (if (nil? x)
        true
        (let [type-kw (if (keyword? x) x (normalize-type-string x))]
          (when type-kw
            (or (isa? type-kw :Semantic/*)
                (isa? type-kw :Relation/*))))))]])

(mr/def ::visibility
  [:enum
   {:description "Column visibility setting"}
   "normal" "sensitive" "retired" "hidden" "details-only"])

;; TODO: what to do here?
(mr/def ::currency
  [:and
   {:description "Currency code for financial columns (e.g., USD, EUR)"}
   ::mc/non-blank-string])

(mr/def ::column-settings
  [:map
   {:closed true
    :description "User-editable column settings for formatting and display"}
   [:column_title {:optional true} [:maybe :string]]
   [:text_align {:optional true} [:maybe [:enum "left" "right" "middle"]]]
   [:text_wrapping {:optional true} [:maybe :boolean]]
   [:view_as {:optional true} [:maybe [:enum "link" "email_link" "image" "auto"]]]
   [:link_text {:optional true} [:maybe :string]]
   [:link_url {:optional true} [:maybe :string]]
   [:show_mini_bar {:optional true} [:maybe :boolean]]
   [:number_style {:optional true} [:maybe [:enum "decimal" "percent" "scientific" "currency"]]]
   [:currency {:optional true} [:maybe :string]]
   [:currency_style {:optional true} [:maybe :string]]
   [:date_style {:optional true} [:maybe :string]]
   [:date_separator {:optional true} [:maybe [:enum "/" "-" "."]]]
   [:date_abbreviate {:optional true} [:maybe :boolean]]
   [:time_enabled {:optional true} [:maybe [:enum "minutes" "seconds" "milliseconds"]]]
   [:time_style {:optional true} [:maybe :string]]])

(mr/def ::column
  [:map
   {:closed true
    :description "Column metadata definition"}
   [:name ::column-name]
   [:display_name {:optional true} ::display-name]
   [:description {:optional true} ::column-description]
   [:base_type {:optional true} ::base-type]
   [:effective_type {:optional true} ::effective-type]
   [:semantic_type {:optional true} ::semantic-type]
   [:visibility_type {:optional true} ::visibility]
   [:fk_target_field_id {:optional true} [:maybe :int]]
   [:currency {:optional true} ::currency]
   [:settings {:optional true} [:maybe ::column-settings]]])

(mr/def ::columns
  [:sequential
   {:description "Array of column metadata definitions"}
   ::column])

(def column-keys
  (->> (m/schema ::column {:registry mr/registry})
       m/deref
       m/entries
       (into #{} (map first))))
