(ns representations.invariants
  (:require
   [clojure.walk :as walk]
   [representations.read :as read]
   [representations.schema.v0.common :as common]))

(defn- extract-refs
  "Returns all refs present in the representation, recursively walking to discover them.
   Returns a set of unref'd names."
  [representation]
  (let [refs (volatile! [])]
    (walk/postwalk (fn [node]
                     (when (common/ref? node)
                       (vswap! refs conj node))
                     node)
                   representation)
    (set (map common/unref @refs))))

;; Invariant 1: All EDNs must parse successfully

(defn parse-valid
  "Checks that all EDNs parse successfully.
   Returns a seq of violation maps, one per parse failure (or empty seq if all valid).
   Each violation includes :invariant/offending with the EDN that failed to parse."
  [edns]
  (for [edn edns
        :let [result (try
                       {:success true :representation (read/parse edn)}
                       (catch Exception e
                         {:success false
                          :exception e
                          :edn edn}))]
        :when (not (:success result))]
    {:invariant/type :invariant.type/parse-valid
     :invariant/message (.getMessage (:exception result))
     :invariant/offending (:edn result)
     :invariant/data (ex-data (:exception result))}))

;; Invariant 2: All `:name` fields must be unique across the collection

(defn unique-names
  "Checks that all :name fields are unique across the collection.
   Returns a seq of violation maps, one per duplicate name (or empty seq if valid).
   Each violation includes :invariant/offending set of all representations with that name.
   Takes already-parsed representations."
  [representations]
  (let [name-groups (group-by :name representations)
        duplicates (filter (fn [[_name reps]] (> (count reps) 1)) name-groups)]
    (for [[name reps] duplicates]
      {:invariant/type :invariant.type/unique-names
       :invariant/message (str "Name '" name "' is not unique across the collection")
       :invariant/offending (set reps)
       :invariant/data {:name name
                        :count (count reps)}})))

;; Invariant 3: No representation may have a dangling `ref:...`. I.e., for every `ref` there must be a corresponding representation with that `:name`.

(defn no-dangling-refs
  "Checks that every ref: has a corresponding representation with that :name.
   Returns a seq of violation maps, one per dangling ref (or empty seq if valid).
   Each violation includes :invariant/offending set with the representation that has the dangling ref.
   Takes already-parsed representations."
  [representations]
  (let [names (set (map :name representations))]
    (for [rep representations
          dangling-ref (remove names (extract-refs rep))]
      {:invariant/type :invariant.type/no-dangling-refs
       :invariant/message (str "Reference 'ref:" dangling-ref "' does not match any representation's :name")
       :invariant/offending #{rep}
       :invariant/data {:ref dangling-ref
                        :from-representation (:name rep)}})))

;; Invariant 4: All representations are of the same version

(defn same-version
  "Checks that all representations have the same :version.
   Returns a seq with one violation if versions differ (or empty seq if valid).
   Takes already-parsed representations."
  [representations]
  (let [versions (set (map :version representations))]
    (if (> (count versions) 1)
      [{:invariant/type :invariant.type/same-version
        :invariant/message "All representations must be of the same version"
        :invariant/data {:versions versions}}]
      [])))

;; Main entry point

(defn check-invariants
  "Checks all invariants on a collection of EDNs.
   Always runs parse-valid first. For successfully parsed representations,
   runs the additional invariants provided (defaults to all remaining invariants).
   
   Returns a seq of violation maps (or empty seq if all valid).
   
   Each violation has:
   - :invariant/type - keyword identifying the invariant
   - :invariant/message - human-readable description
   - :invariant/offending - the offending EDN(s) or representation(s)
   - :invariant/data - additional data about the violation"
  ([edns]
   (check-invariants edns [unique-names no-dangling-refs same-version]))
  ([edns additional-invariants]
   (let [;; Always run parse-valid first
         parse-violations (parse-valid edns)

         ;; Parse all EDNs, collecting only successful ones
         parsed-reps (keep (fn [edn]
                             (try
                               (read/parse edn)
                               (catch Exception _ nil)))
                           edns)

         ;; Run additional invariants on successfully parsed representations
         other-violations (when (seq parsed-reps)
                            (mapcat (fn [invariant-fn]
                                      (invariant-fn parsed-reps))
                                    additional-invariants))

         ;; Combine all violations
         all-violations (concat parse-violations other-violations)]

     (vec all-violations))))
