(ns representations.core
  (:require
   [representations.invariants :as invariants]
   [representations.read :as read]
   [representations.schema.v0.common :as common]))

(def latest-version
  "The most recent stable version for representations"
  :v0)

(defn parse-all
  [edns]
  (map read/parse edns))

(def default-invariants
  "Default invariants to check (parse-valid is always checked first automatically)."
  [invariants/unique-names
   invariants/no-dangling-refs
   invariants/same-version])

(defn invariant-violations
  "Returns invariant violations (or nil if there were none).

   Always checks parse-valid first. For successfully parsed representations,
   runs the additional invariants provided (defaults to all standard invariants).

   There is no way to skip the parse-valid check - it is always run first."
  ([edns] (invariant-violations edns default-invariants))
  ([edns additional-invariants]
   (when-let [violations (not-empty (invariants/check-invariants edns additional-invariants))]
     violations)))

(defn ref?
  "Returns true if x is a reference string (ref:name)."
  [x]
  (common/ref? x))

(defn env?
  "Returns true if x is an environment variable reference (env:VAR_NAME)."
  [x]
  (common/env? x))

(defn unref
  "Strips the 'ref:' prefix from a reference string.
   Returns the name portion, or nil if x is not a ref."
  [x]
  (common/unref x))

(defn resolve-env-var
  "Resolves an environment variable reference to its value.
   Returns the value from the environment, or nil if x is not an env var or the var is not set."
  [x]
  (common/resolve-env-var x))
