(ns intoclj.core
  (:require [clojure.set :as set]
            [clojure.java.javadoc :refer :all]
            [clojure.walk :as walk]
            [clojure.string :as str]
            [clojure.pprint :as pp]
            [clojure.java.io :as io]
            [intoclj.util :as u]
            [intoclj.interop.java :as j]
            [intoclj.interop.javaparser :as p]))

;;;;;;;;;;;;;;;;;;; Reflection  ;;;;;;;;;;;;;;;;;;;

(defn- executable-param-types [e]
  (for [p (j/Executable:getParameters e)]
    (j/Parameter:getType p)))

(def ^:private modifiers
  {:private j/Modifier:isPrivate
   :public j/Modifier:isPublic
   :final j/Modifier:isFinal
   :static j/Modifier:isStatic
   :abstract j/Modifier:isAbstract
   :protected j/Modifier:isProtected
   :interface j/Modifier:isInterface
   :native j/Modifier:isNative
   :strict j/Modifier:isStrict
   :volatile j/Modifier:isVolatile
   :transient j/Modifier:isTransient
   :synchronized j/Modifier:isSynchronized})

(defn- modifier-kws [m]
  (->> modifiers
       (filter (fn [[_ f]] (f m)))
       (map first)
       (set)))

(defn- ->constructor-name [n]
  (str "->" n))

(defn- reflect-methods [c]
  (for [m (j/Class:getDeclaredMethods c)]
    {:name (j/Method:getName m)
     :flags (modifier-kws (j/Method:getModifiers m))
     :declaring-class (j/Method:getDeclaringClass m)
     :parameter-types (executable-param-types m)}))

(defn- reflect-fields [c]
  (for [f (j/Class:getDeclaredFields c)]
    {:name (j/Field:getName f)
     :flags (modifier-kws (j/Field:getModifiers f))
     :declaring-class (j/Field:getDeclaringClass f)
     :type (j/Field:getType f)}))

(defn- reflect-constructors [c]
  (for [co (j/Class:getConstructors c)]
    {:name (->constructor-name (j/Class:getSimpleName c))
     :flags (conj (modifier-kws (j/Constructor:getModifiers co)) :constructor)
     :declaring-class (j/Constructor:getDeclaringClass co)
     :parameter-types (executable-param-types co)}))

(defn- reflect-class [c]
  {:class c
   :name (j/Class:getName c)
   :flags (modifier-kws (j/Class:getModifiers c))
   :fields (reflect-fields c)
   :methods (reflect-methods c)
   :constructors (reflect-constructors c)})

(defn classes-in-package [p]
  (-> (name p)
      (intoclj.JavaClassScanner/getClassesUnderPackage)
      (seq)))

;;;;;;;;;;;;;;;;;;; Resolving Javadoc ;;;;;;;;;;;;;;;;;;;

(defn- drop-path-root [p]
  (->> (str/split p #"/")
       (drop 1)
       (str/join "/")))

(defn- path->class-name [p]
  (-> (u/drop-suffix p ".java")
      (str/replace "/" ".")))

(defn- java-src-entries [f]
  (let [z (j/->ZipFile f)]
    (->> (j/ZipFile:entries z)
         (iterator-seq)
         (map #(hash-map :path (j/ZipEntry:getName %) :zip-entry % :zip-file z))
         (filter #(str/ends-with? (:path %) ".java")))))

(defn- existing-file [f]
  (if (j/File:exists f) f))

(defn- stdlib-src-zips []
  (-> (j/System:getProperty "java.home")
      (j/->File)
      (j/->File "lib/src.zip")
      (existing-file)
      (or (-> (j/System:getProperty "java.home")
              (u/drop-suffix "/jre")
              (u/drop-suffix "\\jre")
              (j/->File)
              (j/->File "src.zip")
              (existing-file)))
      (list (-> (j/System:getProperty "java.home")
              (j/->File)
              (j/->File "javafx-src.zip")
              (existing-file)))
      (filter some?)))

(defn- stdlib-src-entries [z]
  (let [es (java-src-entries z)]
    (if (filter #(str/starts-with? (:path) "java.base") es)
      (map #(update % :path drop-path-root)) es)))

(defn- all-src-entries []
  (->> (str/split (j/System:getProperty "java.class.path") #";")
       (filter #(str/ends-with? % ".jar"))
       (map #(drop-last 4 %))
       (map str/join)
       (map #(str % "-sources.jar"))
       (map j/->File)
       (filter j/File:exists)
       (mapcat java-src-entries)
       (concat (mapcat stdlib-src-entries stdlib-src-zips))
       (map #(assoc % :class-name (path->class-name (:path %))))))

(defn- entry->code [{f :zip-file e :zip-entry}]
  (with-open [is ^java.io.InputStream (j/ZipFile:getInputStream f e)]
    (str/join \newline (line-seq (io/reader is)))))

(defn- class-below-name? [c n]
  (->>
    (j/Class:getName c)
    (take-while #(not= % \$))
    (str/join)
    (= n)))

(defn- class->code [c]
  (->> (all-src-entries)
       (filter #(class-below-name? c (:class-name %)))
       (map entry->code)
       (first)))

(defn- optional->nilable [x]
  (if (instance? java.util.Optional x)
    (when (j/Optional:isPresent x)
      (j/Optional:get x))
    x))

(defn- dec->name [d]
  (str (p/NodeWithSimpleName:getName d)))

(defn- desc->array [d]
  (->> (p/JavadocDescription:getElements d)
       (map p/JavadocDescriptionElement:toText)))

(defn- dec->doc [d]
  (if-let [jd (optional->nilable (p/NodeWithJavadoc:getJavadoc d))]
    {:description (desc->array (p/Javadoc:getDescription jd))
     :tags
     (for [bt (p/Javadoc:getBlockTags jd)]
       {:content (desc->array (p/JavadocBlockTag:getContent bt))
        :tag (p/JavadocBlockTag:getTagName bt)
        :name (str (optional->nilable (p/JavadocBlockTag:getName bt)))})}))

(defn- arr->desc-str [a]
  (->> (mapcat str/split-lines a)
       (map str/trim)
       (str/join " ")))

(defn- doc->str [d]
  (->>
    (for [{t :tag c :content n :name} (:tags d)]
      (str (str/trim (str "- " t " " n)) ": " (arr->desc-str c)))
    (cons (arr->desc-str (:description d)))
    (str/join "\n")))

(defn- dec->flags [d]
  (->> (p/NodeWithModifiers:getModifiers d)
       (map (comp keyword str/trim str))
       (set)))

(defn- classdec->field-javadocs [d]
  (concat
    (for [f (p/NodeWithMembers:getFields d)]
      (let [v (first (p/FieldDeclaration:getVariables f))]
        {:name (dec->name v)
         :doc (dec->doc f)
         :type (str (p/VariableDeclarator:getType v))
         :flags (dec->flags f)}))
    (if (p/EnumDeclaration? d)
      (for [m (p/EnumDeclaration:getEntries d)]
         {:name (dec->name m)
          :doc (dec->doc m)
          :type (dec->name d)
          :flags #{:public :static :final}}))))

(defn- node->param-javadocs [d]
  (for [p (p/NodeWithParameters:getParameters d)]
    {:name (dec->name p)
     :type (str (p/Parameter:getType p))}))

(defn- classdec->method-javadocs [d]
  (for [m (p/NodeWithMembers:getMethods d)]
    {:name (dec->name m)
     :doc (dec->doc m)
     :flags (dec->flags m)
     :parameters (node->param-javadocs m)
     :return-type (str (p/MethodDeclaration:getType m))}))

(defn- classdec->constructor-javadocs [d]
  (for [c (p/NodeWithMembers:getConstructors d)]
    {:name (->constructor-name (dec->name d))
     :doc (dec->doc c)
     :flags (dec->flags c)
     :parameters (node->param-javadocs c)}))

(defn- classdec->javadocs [d]
  {:name (dec->name d)
   :doc (dec->doc d)
   :fields (classdec->field-javadocs d)
   :flags (dec->flags d)
   :methods (classdec->method-javadocs d)
   :constructors (classdec->constructor-javadocs d)})

(defn- class->javadocs [c]
  (some->
   (class->code c)
   (intoclj.JavaFileReader/getDeclarationsInSource)
   (get (j/Class:getSimpleName c))
   (classdec->javadocs)))

;;;;;;;;;;;;;;;;;;; Building model ;;;;;;;;;;;;;;;;;;;

(def ^:private arg-symbols
  (map (comp symbol str)
       (repeat "arg")
       (range 1 Integer/MAX_VALUE)))

(defn- flag? [f x]
  (contains? (:flags x) f))

(defn- flags? [fs x]
  (every? #(flag? % x) fs))

(defn- this-param [met]
  (if-not (or (flag? :static met) (flag? :constructor met))
    {:name 'this :type (:declaring-class met)}))

(defn- matching-doc-methods [jdocs met]
  (if (flag? :constructor met)
    (:constructors jdocs)
    (->> (:methods jdocs)
         (filter #(= (:name %) (:name met))))))

(defn- compute-param-names [jdocs met]
  (->> (matching-doc-methods jdocs met)
       (filter (partial flag? :public))
       (filter #(= (count (:parameters %))
                   (count (:parameter-types met))))
       (map #(map :name (:parameters %)))
       (u/transpose)
       (map set)
       (map #(str/join "|" %))
       (map symbol)))

(defn- compute-params [jdocs met]
  (->> (:parameter-types met)
       (map #(hash-map :name %1 :type %2)
            (concat (compute-param-names jdocs met) arg-symbols))
       (cons (this-param met))
       (filter some?)
       (assoc met :parameters)))

(def ^:private primitive-class-literals
  (u/->map
    #(let [t (symbol (str (j/Class:getName %) "/TYPE"))]
       [(eval t) t])
    [Boolean Byte Short Integer Long Float Double Character]))

(def ^:private primitive-wrapper-class-literals
  (u/mapvals
   (comp symbol str/join #(drop-last 5 %) str)
   primitive-class-literals))

(declare class->sub-literal)

(defn- array-class-literal [c]
  (if-let [t (j/Class:getComponentType c)]
    `(~'array-class ~(class->sub-literal t))))

(defn- class->sub-literal [c]
  (or (primitive-class-literals c)
      (array-class-literal c)
      (symbol (j/Class:getName c))))

(defn- class->literal [c]
  (or (primitive-wrapper-class-literals c)
      (array-class-literal c)
      (symbol (j/Class:getName c))))

(defn- invokation-head-symbol [met]
  (symbol
   (cond
     (flag? :static met)
     (str (j/Class:getName (:declaring-class met)) "/" (:name met))
     (flag? :constructor met)
     (str (j/Class:getName (:declaring-class met)) ".")
     :else (str "." (:name met)))))

(defn- compute-invokation [met]
  (->> (:parameters met)
       (map #(with-meta (:name %) {:tag (j/Class:getName (:type %))}))
       (cons (invokation-head-symbol met))
       (assoc met :invokation)))

(defn- shorten-tree [tr]
  (if (map? tr)
    (if (<= (count tr) 1)
      (shorten-tree (first (vals tr)))
      (u/mapvals shorten-tree tr))
    tr))

(defn- shrink-tree [tr]
  (->> tr
       (walk/postwalk
        #(if (map? %)
           (let [v (distinct (vals %))]
             (if (and (u/single? v) (:name (first v)))
               (first v) %))
           %))
       (shorten-tree)))

(defn- instance-sexps [met]
  (for [{n :name t :type} (:parameters met)]
    `(instance? ~(class->literal t) ~n)))

(defn- compute-expr-tree [mets]
  (-> (->> mets
           (reduce #(assoc-in %1 (instance-sexps %2) (:invokation %2))
                   {})
           (shrink-tree)
           (assoc (first mets) :expr-tree))
      (dissoc :invokation)))

(defn- merge-arities [mets]
 (-> (first mets)
     (dissoc :expr-tree)
     (assoc :arities (sort-by (comp count :parameters) mets))))

(defn- methods->doc-string [mets]
  (if-not (empty? mets)
    (str/join "\n"
      (for [m mets]
        (str
          "-- ("
          (str/join ", "
            (for [p (:parameters m)]
              (str (:type p) " " (:name p))))
          ")"
          (if-let [t (:return-type m)]
            (str " -> " t))
          (if-let [d (:doc m)] (str "\n" (doc->str d))))))))

(defn- method-doc [jdocs met]
  (->> (concat (:methods jdocs)
               (:constructors jdocs))
       (filter #(= (:name %) (:name met)))
       (sort-by (comp count :parameters))
       (methods->doc-string)))

(defn- ->methods [{c :reflection d :javadocs}]
  (->>
   (:constructors c)
   (concat (:methods c))
   (filter (partial flag? :public))
   (map #(compute-params d %))
   (map compute-invokation)
   (group-by (juxt :name (comp count :parameter-types)))
   (vals)
   (map compute-expr-tree)
   (group-by :name)
   (vals)
   (map merge-arities)
   (map #(assoc % :doc (method-doc d %)))))

(defn- ->fields [{c :reflection d :javadocs}]
  (->> (:fields c)
       (filter #(flags? [:public :static :final] %))
       (map #(assoc % :javadocs (get-in d [:fields (:name %)])))))

(defn- ->class [c]
  (let [r (reflect-class c)
        d {:reflection r :javadocs (class->javadocs c)}]
    (merge r d
      {:name (j/Class:getSimpleName c)
       :fq-name (j/Class:getName c)
       :fields (->fields d)
       :methods (->methods d)})))

;;;;;;;;;;;;;;;;;;; Generating code ;;;;;;;;;;;;;;;;;;;

(defn- gen-fn-name [met]
  (if (flag? :constructor met)
    (:name met)
    (str (j/Class:getSimpleName (:declaring-class met))
         ":" (:name met))))

(defn- gen-fn-expr [tr]
  (if (map? tr)
    (->> (u/mapvals gen-fn-expr tr)
         (apply concat)
         (cons `cond))
    tr))

(defn- gen-methods [c]
  (for [met (:methods c)]
    `(defn
       ~(symbol (gen-fn-name met))
       ~@(if-let [d (:doc met)] [d])
       ~@(for [{e :expr-tree p :parameters} (:arities met)]
           `([~@(map :name p)] ~(gen-fn-expr e))))))

(defn- gen-fields [c]
  (for [f (:fields c)]
    `(def
       ~(with-meta
          (symbol
           (str (j/Class:getSimpleName (:declaring-class f)) ":" (:name f)))
          {:tag (j/Class:getName (:type f))})
       ~(str
          "Type: "
          (or (:type (:doc f))
              (j/Class:getName (:type f)))
          (if-let [d (:doc (:javadocs f))] (str "\n" (doc->str d))))
       ~(symbol
         (str (j/Class:getName (:declaring-class f)) "/" (:name f))))))

(defn- gen-class-def [c]
  `(def ~(symbol (str (:name c) ":class"))
     ~@(if-let [d (:doc (:javadocs c))] [(doc->str d)])
     ~(symbol (:fq-name c))))

(defn- gen-class-check [c]
  `(defn ~(symbol (str (:name c) "?"))
     ~(str "Checks if x is an instance of " (:fq-name c) ".")
     [~'x]
     (instance? ~(symbol (:fq-name c)) ~'x)))

(defn- functional-interface-method [c]
  (if (flag? :interface c)
    (let [ms (filter #(flag? :abstract %) (:methods (:reflection c)))]
      (if (u/single? ms) (first ms)))))

(defn- gen-function-constructor [c]
  (if-let [m (functional-interface-method c)]
    `(defn
       ~(with-meta
          (symbol (->constructor-name (:name c)))
          {:tag (symbol (:fq-name c))})
       ~(str "Costructs " (:fq-name c) " that invokes x.")
       [~'x]
       (reify ~(symbol (:fq-name c))
         (~(symbol (:name m))
           [~'_ ~@(take (count (:parameter-types m)) arg-symbols)]
           (~'x ~@(take (count (:parameter-types m)) arg-symbols)))))))

(defn gen-wrappers [c]
  (let [c (->class c)]
    (filter some?
      (concat
        (gen-methods c)
        (gen-fields c)
        [(gen-class-def c)
         (gen-class-check c)
         (gen-function-constructor c)]))))

(defmacro def-wrappers [c]
  `(binding [*warn-on-reflection* true]
     (list ~@(gen-wrappers (resolve c)))))

(defn- pprintln [x]
  (pp/pprint x)
  (newline))

(defn- print-wrappers [c]
 (binding [*print-meta* true]
   (run! pprintln (gen-wrappers c))))

(defn- print-ns-expr [ns]
  (pprintln `(ns ~(symbol ns))))

(defn- print-utils []
  (println "(defmacro ^:private array-class [c]
    (let [s (with-meta (gensym) {:private true})]
      (symbol (eval `(def ~s (class (make-array ~c 0)))))))"))

(defn- ns->file [dir ns]
  (let [n (str (name dir) "/" (str/replace (name ns) "." "/") ".clj")
        f (io/file n)]
    (-> f j/File:getParentFile j/File:mkdirs)
    f))

(defn gen-wrappers-file [dir ns & classes]
  (spit (ns->file dir ns)
    (with-out-str
      (print-ns-expr ns)
      (print-utils)
      (->> (flatten classes)
           (filter (comp j/Modifier:isPublic j/Class:getModifiers))
           (run! print-wrappers)))))

;;;;;;;;;;;;;;;;;;; Dev ;;;;;;;;;;;;;;;;;;;

(comment
  (gen-wrappers-file
    "src" 'intoclj.interop.java
    java.io.File
    java.io.BufferedReader
    java.util.Optional
    java.lang.Class
    java.lang.reflect.Method
    java.lang.reflect.Modifier
    java.lang.reflect.Parameter
    java.lang.reflect.Executable
    java.lang.reflect.Field
    java.lang.reflect.Constructor
    java.lang.System
    java.util.zip.ZipFile
    java.util.zip.ZipEntry
    java.util.function.Function)
  (gen-wrappers-file
    "src" 'intoclj.interop.javaparser
    com.github.javaparser.ast.body.ClassOrInterfaceDeclaration
    com.github.javaparser.ast.body.MethodDeclaration
    com.github.javaparser.ast.body.FieldDeclaration
    com.github.javaparser.ast.Modifier
    com.github.javaparser.ast.body.VariableDeclarator
    com.github.javaparser.ast.body.ConstructorDeclaration
    com.github.javaparser.ast.body.Parameter
    com.github.javaparser.javadoc.Javadoc
    com.github.javaparser.javadoc.JavadocBlockTag
    com.github.javaparser.javadoc.description.JavadocDescription
    com.github.javaparser.javadoc.description.JavadocDescriptionElement
    com.github.javaparser.ast.nodeTypes.NodeWithJavadoc
    com.github.javaparser.ast.nodeTypes.NodeWithParameters
    com.github.javaparser.ast.nodeTypes.NodeWithSimpleName
    com.github.javaparser.ast.nodeTypes.NodeWithModifiers
    com.github.javaparser.ast.body.EnumConstantDeclaration
    com.github.javaparser.ast.body.TypeDeclaration
    com.github.javaparser.ast.nodeTypes.NodeWithMembers
    com.github.javaparser.ast.body.EnumDeclaration))
