;; Copyright (c) Rich Hickey. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file epl-v10.html at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns io.github.frenchy64.fully-satisfies.linear-expansion
  "Macros whose expansions grow linearly rather than exponentially."
  (:refer-clojure :exclude [doseq for])
  (:require [io.github.frenchy64.fully-satisfies.configurable-core-macros.assert-args :refer [assert-args]]))

(defmacro doseq
  "Repeatedly executes body (presumably for side-effects) with
  bindings and filtering as provided by \"for\".  Does not retain
  the head of the sequence. Returns nil.
  
  Unlike clojure.core/doseq, expansion grows linearly with the number
  of nestings of this macro, rather than exponentially."
  [seq-exprs & body]
  (assert-args
    (vector? seq-exprs) "a vector for its binding"
    (even? (count seq-exprs)) "an even number of forms in binding vector")
  (let [step (fn step [recform exprs]
               (if-not exprs
                 [true `(do ~@body)]
                 (let [k (first exprs)
                       v (second exprs)]
                   (if (keyword? k)
                     (let [steppair (step recform (nnext exprs))
                           needrec (steppair 0)
                           subform (steppair 1)]
                       (cond
                         (= k :let) [needrec `(let ~v ~subform)]
                         (= k :while) [false `(when ~v
                                                ~subform
                                                ~@(when needrec [recform]))]
                         (= k :when) [false `(if ~v
                                               (do
                                                 ~subform
                                                 ~@(when needrec [recform]))
                                               ~recform)]))
                     (let [seq- (gensym "seq_")
                           chunk- (with-meta (gensym "chunk_")
                                             {:tag 'clojure.lang.IChunk})
                           count- (gensym "count_")
                           i- (gensym "i_")
                           in-chunk- (gensym "in-chunk_")
                           recform `(if ~in-chunk-
                                      (recur ~seq- ~chunk- ~count- (unchecked-inc ~i-))
                                      (recur (next ~seq-) nil 0 0))
                           steppair (step recform (nnext exprs))
                           needrec (steppair 0)
                           subform (steppair 1)]
                       [true
                        `(loop [~seq- ~v, ~chunk- nil,
                                ~count- 0, ~i- 0]
                           (let [~in-chunk- (< ~i- ~count-)
                                 ~seq- (if ~in-chunk- ~seq- (seq ~seq-))]
                             (when (if ~in-chunk- true ~seq-)
                               (if (if ~in-chunk- false (chunked-seq? ~seq-))
                                 (let [c# (chunk-first ~seq-)]
                                   (recur (chunk-rest ~seq-) c#
                                          (int (count c#)) (int 0)))
                                 (let [~k (if ~in-chunk- (.nth ~chunk- ~i-) (first ~seq-))]
                                   ~subform
                                   ~@(when needrec [recform]))))))])))))]
    (nth (step nil (seq seq-exprs)) 1)))

(defmacro for
  "List comprehension. Takes a vector of one or more
   binding-form/collection-expr pairs, each followed by zero or more
   modifiers, and yields a lazy sequence of evaluations of expr.
   Collections are iterated in a nested fashion, rightmost fastest,
   and nested coll-exprs can refer to bindings created in prior
   binding-forms.  Supported modifiers are: :let [binding-form expr ...],
   :while test, :when test.

  Unlike clojure.core/for, expansion grows linearly with the number
  of nestings of this macro, rather than exponentially.

  (take 100 (for [x (range 100000000) y (range 1000000) :while (< y x)] [x y]))"
  {:added "1.0"}
  [seq-exprs body-expr]
  (assert-args
    (vector? seq-exprs) "a vector for its binding"
    (even? (count seq-exprs)) "an even number of forms in binding vector")
  (let [to-groups (fn [seq-exprs]
                    (reduce (fn [groups [k v]]
                              (if (keyword? k)
                                (conj (pop groups) (conj (peek groups) [k v]))
                                (conj groups [k v])))
                            [] (partition 2 seq-exprs)))
        err (fn [& msg] (throw (IllegalArgumentException. ^String (apply str msg))))
        gbody (gensym "body__")
        emit-bind (fn emit-bind [[[bind expr & mod-pairs]
                                  & [[_ next-expr] :as next-groups]]]
                    (let [giter (gensym "iter__")
                          gxs (gensym "to-process__")
                          gi (gensym "chunk-index__")
                          gb (gensym "chunk-buffer__")
                          gchunked? (gensym "c?__")
                          gchunk (with-meta (gensym "chunk__") {:tag 'clojure.lang.IChunk})
                          gchunk-size (gensym "csize__")
                          gchunk-iter? (gensym "citer__")
                          outer-loop (boolean next-groups)
                          do-outer (fn do-outer [[[k v :as pair] & etc]]
                                     (cond
                                       (= k :let) `(let ~v ~(do-outer etc))
                                       (= k :while) `(when ~v ~(do-outer etc))
                                       (= k :when) `(if ~v
                                                      ~(do-outer etc)
                                                      (recur (rest ~gxs)))
                                       (keyword? k) (err "Invalid 'for' keyword " k)
                                       :else
                                       `(let [iterys# ~(emit-bind next-groups)
                                              fs# (seq (iterys# ~next-expr))]
                                          (if fs#
                                            (concat fs# (~giter (rest ~gxs)))
                                            (recur (rest ~gxs))))))
                          do-inner (fn do-inner [[[k v :as pair] & etc]]
                                     (cond
                                       (= k :let) `(let ~v ~(do-inner etc))
                                       (= k :while) `(if ~v
                                                       ~(do-inner etc)
                                                       (when ~gchunked?
                                                         ; (not (< ~gi ~gchunk-size)) to break loop
                                                         (recur
                                                           ~gxs
                                                           ~gchunk-size
                                                           ~gchunk-size
                                                           ~gchunk
                                                           false ;; drop results
                                                           ~gb
                                                           ~gchunked?)))
                                       (= k :when) `(if ~v
                                                      ~(do-inner etc)
                                                      (if ~gchunked?
                                                        (recur
                                                          ~gxs
                                                          (unchecked-inc ~gi)
                                                          ~gchunk-size
                                                          ~gchunk
                                                          ~gchunk-iter?
                                                          ~gb
                                                          ~gchunked?)
                                                        (recur
                                                          (rest ~gxs)
                                                          ~gi
                                                          ~gchunk-size
                                                          ~gchunk
                                                          ~gchunk-iter?
                                                          ~gb
                                                          ~gchunked?)))
                                       (keyword? k) (err "Invalid 'for' keyword " k)
                                       :else `(let [~gbody ~body-expr]
                                                (if ~gchunked?
                                                  (do (chunk-append ~gb ~gbody)
                                                      (recur ~gxs
                                                             (unchecked-inc ~gi)
                                                             ~gchunk-size
                                                             ~gchunk
                                                             ~gchunk-iter?
                                                             ~gb
                                                             ~gchunked?))
                                                  (cons ~gbody
                                                        (~giter (rest ~gxs)))))))]
                      `(fn ~giter [~gxs]
                         (lazy-seq
                           ~(if outer-loop
                              `(loop [~gxs ~gxs]
                                 (when-first [~bind ~gxs]
                                   ~(do-outer mod-pairs)))
                              `(loop [~gxs ~gxs
                                      ~gi (int 0)
                                      ~gchunk-size (int 1) ;; before newly-chunked, ensure (< ~gi ~gchunk-size)
                                      ~gchunk nil
                                      ~gchunk-iter? true
                                      ~gb nil
                                      ~gchunked? false]
                                 (if (< ~gi ~gchunk-size)
                                   (when-let [~gxs (if ~gchunked? ~gxs (seq ~gxs))]
                                     (let [chunked# (chunked-seq? ~gxs)
                                           newly-chunked?# (if ~gchunked? false chunked#)
                                           ~gi (int (if newly-chunked?# 0 ~gi))
                                           ~gchunk (if newly-chunked?# (chunk-first ~gxs) ~gchunk)
                                           ~gchunk-size (int (if newly-chunked?# (count ~gchunk) ~gchunk-size))
                                           ~gb (if newly-chunked?# (chunk-buffer ~gchunk-size) ~gb)
                                           ~gchunked? (or ~gchunked? chunked#)
                                           ~bind (if ~gchunked? (.nth ~gchunk ~gi) (first ~gxs))]
                                       ~(do-inner mod-pairs)))
                                   (if ~gchunk-iter?
                                     (chunk-cons
                                       (chunk ~gb)
                                       (~giter (chunk-rest ~gxs)))
                                     (chunk-cons (chunk ~gb) nil)))))))))]
    `(let [iter# ~(emit-bind (to-groups seq-exprs))]
       (iter# ~(second seq-exprs)))))
