package stencil.impl;

import stencil.PreparedTemplate;
import stencil.TemplateFactory;

import java.io.File;
import java.io.IOException;
import java.time.ZoneOffset;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Wraps a TemplateFactory instance and proxies calls only iff template file has not been changed since last call.
 */
@SuppressWarnings("unused")
public final class CachingTemplateFactory implements TemplateFactory {
    private final TemplateFactory templateFactory;
    private final Map<File, PreparedTemplate> cache = new ConcurrentHashMap<>();

    /**
     * Constructs a new wrapping instance. Caches in memory.
     *
     * @param templateFactory instance to wrap
     * @throws IllegalArgumentException on null input.
     */
    public CachingTemplateFactory(TemplateFactory templateFactory) {
        if (templateFactory == null)
            throw new IllegalArgumentException("can not wrap null object!");

        this.templateFactory = templateFactory;
    }

    @Override
    public PreparedTemplate prepareTemplateFile(File templateFile) throws IOException {
        if (cache.containsKey(templateFile)) {
            PreparedTemplate stored = cache.get(templateFile);
            if (stored.creationDateTime().toEpochSecond(ZoneOffset.UTC) <= templateFile.lastModified()) {
                stored = templateFactory.prepareTemplateFile(templateFile);
                cache.put(templateFile, stored);
            }
            return stored;
        } else {
            final PreparedTemplate stored = templateFactory.prepareTemplateFile(templateFile);
            cache.put(templateFile, stored);
            return stored;
        }
    }
}
