package io.dropwizard.revolver.optimizer;

import com.google.common.collect.Maps;
import io.dropwizard.revolver.RevolverBundle;
import io.dropwizard.revolver.core.config.ApiLatencyConfig;
import io.dropwizard.revolver.core.config.RevolverConfig;
import io.dropwizard.revolver.core.config.RevolverServiceConfig;
import io.dropwizard.revolver.core.config.hystrix.ThreadPoolConfig;
import io.dropwizard.revolver.http.config.RevolverHttpApiConfig;
import io.dropwizard.revolver.http.config.RevolverHttpServiceConfig;
import io.dropwizard.revolver.optimizer.config.OptimizerConcurrencyConfig;
import io.dropwizard.revolver.optimizer.config.OptimizerConfig;
import io.dropwizard.revolver.optimizer.config.OptimizerTimeConfig;
import io.dropwizard.revolver.optimizer.utils.OptimizerUtils;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

/***
 Created by nitish.goyal on 29/03/19
 ***/
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@Slf4j
public class RevolverConfigUpdater implements Runnable {

    private RevolverConfig revolverConfig;
    private OptimizerConfig optimizerConfig;
    private OptimizerMetricsCache optimizerMetricsCache;

    @Override
    public void run() {

        Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap = Maps.newHashMap();
        Map<OptimizerCacheKey, OptimizerMetrics> metricsCache = optimizerMetricsCache.getCache();
        if(metricsCache.isEmpty()) {
            return;
        }

        Map<String, Number> aggregatedAppLevelMetricsValues = Maps.newHashMap();

        metricsCache.forEach((key, optimizerMetrics) -> {
            if(optimizerAggregatedMetricsMap.get(key.getName()) == null)
                optimizerAggregatedMetricsMap.put(key.getName(), OptimizerAggregatedMetrics.builder()
                        .pool(key.getName())
                        .metricsAggValueMap(Maps.newHashMap())
                        .build());

            OptimizerAggregatedMetrics optimizerAggregatedMetrics = optimizerAggregatedMetricsMap.get(key.getName());
            Map<String, Number> aggregatedMetricsValues = optimizerAggregatedMetrics.getMetricsAggValueMap();

            optimizerMetrics.getMetrics()
                    .forEach((metric, value) -> {

                        aggregateAppLevelMetrics(aggregatedAppLevelMetricsValues, metric, value);
                        aggregateApiLevelMetrics(optimizerMetrics, aggregatedMetricsValues, metric, value);
                    });

        });
        updateRevolverConfig(optimizerAggregatedMetricsMap);
        updateLatencyThreshold(aggregatedAppLevelMetricsValues);

    }

    private void updateLatencyThreshold(Map<String, Number> aggregatedAppLevelMetricsValues) {

        if(optimizerConfig.getTimeConfig() == null || aggregatedAppLevelMetricsValues.get(optimizerConfig.getTimeConfig()
                                                                                                  .getAppLatencyMetric()) == null) {
            return;
        }
        int latencyThresholdValue = aggregatedAppLevelMetricsValues.get(optimizerConfig.getTimeConfig()
                                                                                .getAppLatencyMetric())
                .intValue();
        optimizerConfig.getTimeConfig()
                .setAppLatencyThresholdValue(latencyThresholdValue);
    }

    private void aggregateAppLevelMetrics(Map<String, Number> aggregatedAppLevelMetricsValues, String metric, Number value) {

        if(aggregatedAppLevelMetricsValues.get(metric) == null) {
            aggregatedAppLevelMetricsValues.put(metric, value);
        } else {
            aggregatedAppLevelMetricsValues.put(metric, (aggregatedAppLevelMetricsValues.get(metric)
                                                                 .intValue() + value.intValue()) >> 1);
        }
        if(OptimizerUtils.LATENCY_PERCENTILE_99.equals(metric)){
            log.error("Aggregated 99%ile for app : " + aggregatedAppLevelMetricsValues.get(metric));
        }
    }

    private void aggregateApiLevelMetrics(OptimizerMetrics optimizerMetrics, Map<String, Number> aggregatedMetricsValues, String metric,
                                          Number value) {
        OptimizerMetrics.AggregationAlgo aggregationAlgo = optimizerMetrics.getAggregationAlgo();
        switch (aggregationAlgo) {
            case MAX:
                if(aggregatedMetricsValues.get(metric) == null || aggregatedMetricsValues.get(metric)
                                                                          .intValue() < value.intValue()) {
                    aggregatedMetricsValues.put(metric, value);
                }
                break;
            case AVG:
                if(aggregatedMetricsValues.get(metric) == null) {
                    aggregatedMetricsValues.put(metric, value);
                } else {
                    aggregatedMetricsValues.put(metric, (aggregatedMetricsValues.get(metric)
                                                                 .intValue() + value.intValue()) >> 1);
                }
        }
    }

    private void updateRevolverConfig(Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap) {
        AtomicBoolean configUpdated = new AtomicBoolean();
        revolverConfig.getServices()
                .forEach(revolverServiceConfig -> {
                    if(revolverServiceConfig.getThreadPoolGroupConfig() != null) {
                        revolverServiceConfig.getThreadPoolGroupConfig()
                                .getThreadPools()
                                .forEach(threadPoolConfig -> {
                                    updatedPoolSettings(threadPoolConfig, optimizerAggregatedMetricsMap, configUpdated);
                                });
                    }
                    if(revolverServiceConfig instanceof RevolverHttpServiceConfig) {
                        ((RevolverHttpServiceConfig)revolverServiceConfig).getApis()
                                .forEach(api -> {
                                    updatedApiSettings(revolverServiceConfig, api, optimizerAggregatedMetricsMap, configUpdated);
                                });
                    }
                });


        if(configUpdated.get()) {
            RevolverBundle.loadServiceConfiguration(revolverConfig);
        }
    }

    private void updatedPoolSettings(ThreadPoolConfig threadPoolConfig,
                                     Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap, AtomicBoolean configUpdated) {

        OptimizerAggregatedMetrics optimizerAggregatedMetrics = optimizerAggregatedMetricsMap.get(threadPoolConfig.getThreadPoolName());

        if(optimizerAggregatedMetrics == null) {
            return;
        }
        updateConcurrencySetting(threadPoolConfig, optimizerAggregatedMetrics, configUpdated, threadPoolConfig.getThreadPoolName());

    }

    private void updatedApiSettings(RevolverServiceConfig revolverServiceConfig, RevolverHttpApiConfig api,
                                    Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap, AtomicBoolean configUpdated) {

        String key = revolverServiceConfig.getService() + "." + api.getApi();
        OptimizerAggregatedMetrics optimizerAggregatedMetrics = optimizerAggregatedMetricsMap.get(key);

        if(optimizerAggregatedMetrics == null) {
            return;
        }
        updateConcurrencySetting(api.getRuntime()
                                         .getThreadPool(), optimizerAggregatedMetrics, configUpdated, api.getApi());
        updateTimeoutSettings(api.getRuntime()
                                      .getThreadPool(), optimizerAggregatedMetrics, configUpdated, api);

        updateLatencySettings(api, optimizerAggregatedMetrics);
    }

    private void updateConcurrencySetting(ThreadPoolConfig threadPoolConfig, OptimizerAggregatedMetrics optimizerAggregatedMetrics,
                                          AtomicBoolean configUpdated, String poolName) {
        if(optimizerAggregatedMetrics.getMetricsAggValueMap()
                   .get(OptimizerUtils.ROLLING_MAX_ACTIVE_THREADS) == null) {
            return;
        }
        OptimizerConcurrencyConfig concurrencyConfig = optimizerConfig.getConcurrencyConfig();
        int maxRollingActiveThreads = optimizerAggregatedMetrics.getMetricsAggValueMap()
                .get(OptimizerUtils.ROLLING_MAX_ACTIVE_THREADS)
                .intValue();
        int concurrency = threadPoolConfig.getConcurrency();

        if(maxRollingActiveThreads == 0) {
            threadPoolConfig.setConcurrency(1);
            log.error("Setting concurrency for : " + poolName + " from : " + concurrency + " to : " + threadPoolConfig.getConcurrency() +
                      ", maxRollingActiveThreads : " + maxRollingActiveThreads);
            return;
        }

        if(maxRollingActiveThreads > concurrency * concurrencyConfig.getMaxThreshold() ||
           maxRollingActiveThreads < concurrency * concurrencyConfig.getMinThreshold()) {

            int updatedConcurrency = (int)Math.ceil(maxRollingActiveThreads * concurrencyConfig.getBandwidth());
            threadPoolConfig.setConcurrency(updatedConcurrency);
            configUpdated.set(true);
            log.error("Setting concurrency for : " + poolName + " from : " + concurrency + " to : " + updatedConcurrency +
                      ", maxRollingActiveThreads : " + maxRollingActiveThreads);
        }

    }

    private void updateTimeoutSettings(ThreadPoolConfig threadPool, OptimizerAggregatedMetrics optimizerAggregatedMetrics,
                                       AtomicBoolean configUpdated, RevolverHttpApiConfig api) {

        OptimizerTimeConfig timeoutConfig = optimizerConfig.getTimeConfig();
        if(timeoutConfig == null || optimizerAggregatedMetrics.getMetricsAggValueMap()
                                            .get(timeoutConfig.getTimeoutMetric()) == null) {
            return;
        }
        int meanTimeoutValue = optimizerAggregatedMetrics.getMetricsAggValueMap()
                .get(timeoutConfig.getTimeoutMetric())
                .intValue();

        if(meanTimeoutValue <= 0) {
            return;
        }

        int currentTimeout = threadPool.getTimeout();
        int newTimeout = currentTimeout;

        Set<RevolverHttpApiConfig.RequestMethod> methods = api.getMethods();
        double timeoutBuffer;

        if(methods.isEmpty() || !(methods.contains(RevolverHttpApiConfig.RequestMethod.GET))) {
            timeoutBuffer = timeoutConfig.getAllMethodTimeoutBuffer();
        } else {
            timeoutBuffer = timeoutConfig.getGetMethodTimeoutBuffer();
        }

        if(currentTimeout < meanTimeoutValue) {
            newTimeout = (int)(meanTimeoutValue * timeoutBuffer);
            configUpdated.set(true);
        } else if(currentTimeout > (meanTimeoutValue * timeoutBuffer)) {
            newTimeout = (int)(meanTimeoutValue * timeoutBuffer);
            configUpdated.set(true);
        }
        log.error("Setting timeout for : " + api.getApi() + " from : " + threadPool.getTimeout() + " to : " + newTimeout + ", " +
                  "meanTimeoutValue : " + meanTimeoutValue + ", with timeout buffer : " + timeoutBuffer);
        threadPool.setTimeout(newTimeout);

    }

    private void updateLatencySettings(RevolverHttpApiConfig api, OptimizerAggregatedMetrics optimizerAggregatedMetrics) {
        String latencyMetric = optimizerConfig.getTimeConfig()
                .getApiLatencyMetric();
        int apiLatency = optimizerAggregatedMetrics.getMetricsAggValueMap()
                                  .get(latencyMetric) == null ? 0 : optimizerAggregatedMetrics.getMetricsAggValueMap()
                                  .get(latencyMetric)
                                  .intValue();

        if(apiLatency <= 0) {
            return;
        }
        log.error("apiLatency : " + apiLatency + " for api : " + api);
        if(api.getApiLatencyConfig() == null) {
            api.setApiLatencyConfig(ApiLatencyConfig.builder()
                                            .build());
        }
        api.getApiLatencyConfig()
                .setLatency(apiLatency);

    }
}
