package io.dropwizard.revolver.optimizer;

import com.google.common.cache.*;
import lombok.NonNull;
import org.msgpack.jackson.dataformat.Tuple;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/***
 Created by nitish.goyal on 29/03/19
 ***/
public class OptimizerMetricsCache {

    private static LinkedHashMap<Tuple<Long, String>, OptimizerMetrics> poolTimeBasedMetricsMap = new LinkedHashMap<>();

    private static LoadingCache<Tuple<Long, String>, OptimizerMetrics> cache = CacheBuilder.newBuilder()
            .concurrencyLevel(4)
            .expireAfterWrite(15, TimeUnit.MINUTES)
            .removalListener(new RemovalListener<Tuple<Long, String>, OptimizerMetrics>() {
                @Override
                public void onRemoval(RemovalNotification<Tuple<Long, String>, OptimizerMetrics> notification) {
                    poolTimeBasedMetricsMap.remove(notification.getKey());
                }
            })
            .build(new CacheLoader<Tuple<Long, String>, OptimizerMetrics>() {
                @Override
                public OptimizerMetrics load(@NonNull Tuple<Long, String> key) throws Exception {
                    return poolTimeBasedMetricsMap.get(key);
                }
            });

    public static OptimizerMetrics get(Tuple<Long, String> key) {
        try {
            return cache.get(key);
        } catch (CacheLoader.InvalidCacheLoadException e) {
            return null;
        } catch (ExecutionException e) {
            throw new RuntimeException("Error while getting value from the cache for the key : " + key);
        }
    }

    public static void put(Tuple<Long, String> key, OptimizerMetrics value) {
        cache.put(key, value);
    }

    public static Map<Tuple<Long, String>, OptimizerMetrics> getCache() {
        return Collections.unmodifiableMap(cache.asMap());
    }


}
