package io.dropwizard.revolver.optimizer;

import com.google.common.collect.Maps;
import io.dropwizard.revolver.RevolverBundle;
import io.dropwizard.revolver.core.config.RevolverConfig;
import io.dropwizard.revolver.core.config.hystrix.ThreadPoolConfig;
import io.dropwizard.revolver.optimizer.config.OptimizerConcurrencyConfig;
import io.dropwizard.revolver.optimizer.config.OptimizerConfig;
import io.dropwizard.revolver.optimizer.utils.OptimizerUtils;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.msgpack.jackson.dataformat.Tuple;

import java.util.Map;

/***
 Created by nitish.goyal on 29/03/19
 ***/
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class OptimizerConfigUpdater implements Runnable {

    private RevolverConfig revolverConfig;
    private OptimizerConfig optimizerConfig;

    @Override
    public void run() {

        Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap = Maps.newHashMap();
        Map<Tuple<Long, String>, OptimizerMetrics> metricsCache = OptimizerMetricsCache.getCache();
        if(metricsCache.isEmpty()) {
            return;
        }
        metricsCache.forEach((tuple, optimizerMetrics) -> {
            if(optimizerAggregatedMetricsMap.get(tuple.second()) == null)
                optimizerAggregatedMetricsMap.put(tuple.second(), OptimizerAggregatedMetrics.builder()
                        .pool(tuple.second())
                        .build());

            OptimizerAggregatedMetrics optimizerAggregatedMetrics = optimizerAggregatedMetricsMap.get(tuple.second());
            Map<String, Integer> aggregatedMetricsValues = optimizerAggregatedMetrics.getMetricsMaxValueMap();

            optimizerMetrics.getMetrics()
                    .forEach((metric, value) -> {
                        if(aggregatedMetricsValues.get(metric) == null || aggregatedMetricsValues.get(metric) < value) {
                            aggregatedMetricsValues.put(metric, value);
                        }
                    });

        });

        updateRevolverConfig(optimizerAggregatedMetricsMap);

    }

    private void updateRevolverConfig(Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap) {
        revolverConfig.getServices()
                .forEach(revolverServiceConfig -> {
                    if(revolverServiceConfig.getThreadPoolGroupConfig() == null) {
                        return;
                    }
                    revolverServiceConfig.getThreadPoolGroupConfig()
                            .getThreadPools()
                            .forEach(threadPoolConfig -> {
                                updatedPoolSettings(threadPoolConfig, optimizerAggregatedMetricsMap);
                            });
                });

        RevolverBundle.loadServiceConfiguration(revolverConfig);
    }

    private void updatedPoolSettings(ThreadPoolConfig threadPoolConfig,
                                     Map<String, OptimizerAggregatedMetrics> optimizerAggregatedMetricsMap) {

        OptimizerAggregatedMetrics optimizerAggregatedMetrics = optimizerAggregatedMetricsMap.get(threadPoolConfig.getThreadPoolName());

        if(optimizerAggregatedMetrics == null) {
            return;
        }

        updateConcurrencySetting(threadPoolConfig, optimizerAggregatedMetrics);

    }

    private void updateConcurrencySetting(ThreadPoolConfig threadPoolConfig, OptimizerAggregatedMetrics optimizerAggregatedMetrics) {
        if(optimizerAggregatedMetrics.getMetricsMaxValueMap()
                   .get(OptimizerUtils.ROLLING_MAX_ACTIVE_THREADS) == null) {
            return;
        }
        OptimizerConcurrencyConfig concurrencyConfig = optimizerConfig.getConcurrencyConfig();
        int maxRollingActiveThreads = optimizerAggregatedMetrics.getMetricsMaxValueMap()
                .get(OptimizerUtils.ROLLING_MAX_ACTIVE_THREADS);

        int concurrency = threadPoolConfig.getConcurrency();

        if(maxRollingActiveThreads > concurrency * concurrencyConfig.getMaxThreshold()) {
            concurrency = (int)(concurrency * concurrencyConfig.getIncreaseBy());
        } else if(maxRollingActiveThreads < concurrency * concurrencyConfig.getMinThreshold()) {
            concurrency = (int)(concurrency * concurrencyConfig.getDecreaseBy());
        }
        threadPoolConfig.setConcurrency(concurrency);
    }
}
