/*
 * Copyright 2016 Phaneesh Nagaraja <phaneesh.n@gmail.com>.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package io.dropwizard.revolver.discovery;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.flipkart.ranger.healthcheck.HealthcheckStatus;
import com.flipkart.ranger.model.ServiceNode;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import io.dropwizard.discovery.client.io.dropwizard.ranger.ServiceDiscoveryClient;
import io.dropwizard.discovery.common.ShardInfo;
import io.dropwizard.revolver.discovery.model.Endpoint;
import io.dropwizard.revolver.discovery.model.RangerEndpointSpec;
import io.dropwizard.revolver.discovery.model.SimpleEndpointSpec;
import lombok.Builder;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryNTimes;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @author phaneesh
 */
@Slf4j
public class RevolverServiceResolver {

    private ObjectMapper objectMapper;
    private final boolean discoverEnabled;
    private final CuratorFramework curatorFramework;
    private final ServiceResolverConfig resolverConfig;

    @Getter
    private Map<String, ServiceDiscoveryClient> serviceFinders = Maps.newConcurrentMap();
    private final ExecutorService executorService = Executors.newCachedThreadPool();

    @Builder
    public RevolverServiceResolver(final ServiceResolverConfig resolverConfig, final ObjectMapper objectMapper) {
        this.resolverConfig = resolverConfig;
        this.objectMapper = objectMapper;
        if (resolverConfig != null) {
            if (!Strings.isNullOrEmpty(resolverConfig.getZkConnectionString())) {
                this.curatorFramework = CuratorFrameworkFactory.builder().connectString(resolverConfig.getZkConnectionString())
                        .namespace(resolverConfig.getNamespace()).retryPolicy(new RetryNTimes(1000, 500)).build();
                this.curatorFramework.start();
                this.discoverEnabled = true;
            } else {
                this.discoverEnabled = false;
                this.curatorFramework = null;
            }
        } else {
            discoverEnabled = false;
            curatorFramework = null;
        }
    }

    @Builder(builderMethodName = "usingCurator")
    public RevolverServiceResolver(final ServiceResolverConfig resolverConfig, final ObjectMapper objectMapper, final CuratorFramework curatorFramework) {
        this.resolverConfig = resolverConfig;
        this.objectMapper = objectMapper;
        this.curatorFramework = curatorFramework;
        this.discoverEnabled = true;
    }

    public Endpoint resolve(final EndpointSpec endpointSpecification) {
        return new SpecResolver(this.discoverEnabled, this.serviceFinders).resolve(endpointSpecification);
    }


    public void register(final EndpointSpec endpointSpecification) {
        endpointSpecification.accept(new SpecVisitor() {

            @Override
            public void visit(final SimpleEndpointSpec simpleEndpointSpecification) {
                log.info("Initialized simple service: " + simpleEndpointSpecification.getHost());
            }

            @Override
            @SuppressWarnings("unchecked")
            public void visit(final RangerEndpointSpec rangerEndpointSpecification) {
                try {
                    if(!serviceFinders.containsKey(rangerEndpointSpecification.getService())) { //Avoid duplicate registration
                        ServiceDiscoveryClient discoveryClient = ServiceDiscoveryClient.fromCurator()
                                .curator(curatorFramework)
                                .environment(rangerEndpointSpecification.getEnvironment())
                                .namespace(resolverConfig.getNamespace())
                                .objectMapper(objectMapper)
                                .serviceName(rangerEndpointSpecification.getService())
                                .build();
                        serviceFinders.put(rangerEndpointSpecification.getService(), discoveryClient);
                        executorService.submit(() -> {
                                    try {
                                        log.info("Service finder starting for: " + rangerEndpointSpecification.getService());
                                        discoveryClient.start();
                                        log.info("Initialized ZK service: " + rangerEndpointSpecification.getService());
                                    } catch (Exception e) {
                                        log.error("Error registering service finder started for: " + rangerEndpointSpecification.getService(), e);
                                    }
                                    return null;
                                }
                        );
                    }
                } catch (Exception e) {
                    log.error("Error registering hander for service: " + rangerEndpointSpecification.getService(), e);
                }
            }
        });
    }

    private static class SpecResolver implements SpecVisitor {
        private Endpoint endpoint;
        private final boolean discoverEnabled;
        private final Map<String, ServiceDiscoveryClient> serviceDiscoveryClients;

        private SpecResolver(final boolean discoverEnabled, final Map<String, ServiceDiscoveryClient> serviceDiscoveryClients) {
            this.discoverEnabled = discoverEnabled;
            this.serviceDiscoveryClients = serviceDiscoveryClients;
        }

        @Override
        public void visit(final SimpleEndpointSpec simpleEndpointSpecification) {
            this.endpoint = Endpoint.builder().host(simpleEndpointSpecification.getHost()).port(simpleEndpointSpecification.getPort()).build();
        }

        @Override
        public void visit(final RangerEndpointSpec rangerEndpointSpecification) {
            if (!this.discoverEnabled) {
                throw new IllegalAccessError("Zookeeper is not initialized in config. Discovery based lookups will not be possible.");
            }
            final Optional<ServiceNode<ShardInfo>> node = serviceDiscoveryClients.get(rangerEndpointSpecification.getService()).getNode();
            //Get only the nodes that are healthy
            node.ifPresent( n -> {
                if(n.getHealthcheckStatus() == HealthcheckStatus.healthy) {
                    this.endpoint = Endpoint.builder().host(n.getHost()).port(n.getPort()).build();
                }
            });
        }

        Endpoint resolve(final EndpointSpec specification) {
            specification.accept(this);
            return this.endpoint;
        }
    }
}
