/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.primer.auth;

import com.codahale.metrics.annotation.Metered;
import com.github.toastshaman.dropwizard.auth.jwt.JsonWebTokenParser;
import com.github.toastshaman.dropwizard.auth.jwt.exceptions.TokenExpiredException;
import com.github.toastshaman.dropwizard.auth.jwt.hmac.HmacSHA512Verifier;
import com.github.toastshaman.dropwizard.auth.jwt.model.JsonWebToken;
import com.google.common.base.Strings;
import feign.FeignException;
import io.dropwizard.primer.auth.PrimerAuthorizationRegistry;
import io.dropwizard.primer.cache.TokenCacheManager;
import io.dropwizard.primer.core.PrimerError;
import io.dropwizard.primer.exception.PrimerException;
import io.dropwizard.primer.model.PrimerBundleConfiguration;
import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import javax.annotation.Priority;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.Provider;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.ReadableDuration;
import org.joda.time.ReadableInstant;
import org.joda.time.ReadableInterval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Provider
@Priority(value=1000)
public class PrimerAuthenticatorRequestFilter
implements ContainerRequestFilter {
    private static final Logger log = LoggerFactory.getLogger(PrimerAuthenticatorRequestFilter.class);
    private JsonWebTokenParser tokenParser;
    private HmacSHA512Verifier verifier;
    private PrimerBundleConfiguration configuration;
    private final Duration acceptableClockSkew;
    private static final String AUTHORIZED_FOR_ID = "X-AUTHORIZED-FOR-ID";
    private static final String AUTHORIZED_FOR_SUBJECT = "X-AUTHORIZED-FOR-SUBJECT";
    private static final String AUTHORIZED_FOR_NAME = "X-AUTHORIZED-FOR-NAME";

    public PrimerAuthenticatorRequestFilter(JsonWebTokenParser tokenParser, HmacSHA512Verifier verifier, PrimerBundleConfiguration configuration) {
        this.tokenParser = tokenParser;
        this.verifier = verifier;
        this.configuration = configuration;
        this.acceptableClockSkew = new Duration((long)configuration.getClockSkew());
    }

    @Metered(name="primer")
    public void filter(ContainerRequestContext requestContext) throws IOException {
        if (!this.configuration.isEnabled()) {
            return;
        }
        if (PrimerAuthorizationRegistry.isWhilisted(requestContext.getUriInfo().getPath())) {
            return;
        }
        Optional<String> token = this.getToken(requestContext);
        if (!token.isPresent()) {
            requestContext.abortWith(Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)PrimerError.builder().errorCode("PR000").message("Bad request").build()).build());
        } else {
            try {
                if (TokenCacheManager.checkBlackList(token.get())) {
                    requestContext.abortWith(Response.status((Response.Status)Response.Status.FORBIDDEN).entity((Object)PrimerError.builder().errorCode("PR001").message("Forbidden").build()).build());
                    return;
                }
                if (TokenCacheManager.checkCache(token.get())) {
                    JsonWebToken cachedToken = this.tokenParser.parse(token.get());
                    this.stampHeaders(requestContext, cachedToken);
                    return;
                }
            }
            catch (ExecutionException e) {
                log.warn("Error getting token from cache: {}", (Object)e.getMessage());
            }
            try {
                JsonWebToken webToken = this.verifyToken(token.get());
                this.checkExpiry(webToken);
                boolean isAuthorized = this.authorize(requestContext, webToken, token.get());
                if (!isAuthorized) {
                    requestContext.abortWith(Response.status((Response.Status)Response.Status.UNAUTHORIZED).entity((Object)PrimerError.builder().errorCode("PR002").message("Unauthorized").build()).build());
                }
                this.stampHeaders(requestContext, webToken);
            }
            catch (TokenExpiredException e) {
                log.error("Token Expiry Error", (Throwable)e);
                requestContext.abortWith(Response.status((Response.Status)Response.Status.PRECONDITION_FAILED).entity((Object)PrimerError.builder().errorCode("PR003").message("Expired").build()).build());
            }
            catch (FeignException e) {
                log.error("Feign error", (Throwable)e);
                TokenCacheManager.blackList(token.get());
                requestContext.abortWith(Response.status((int)Response.Status.UNAUTHORIZED.getStatusCode()).entity((Object)PrimerError.builder().errorCode("PR002").message("Unauthorized").build()).build());
            }
            catch (PrimerException e) {
                log.error("Primer error", (Throwable)e);
                TokenCacheManager.blackList(token.get());
                requestContext.abortWith(Response.status((int)Response.Status.UNAUTHORIZED.getStatusCode()).entity((Object)PrimerError.builder().errorCode("PR002").message("Unauthorized").build()).build());
            }
            catch (Exception e) {
                log.error("Primer error", (Throwable)e);
                requestContext.abortWith(Response.status((Response.Status)Response.Status.FORBIDDEN).entity((Object)PrimerError.builder().errorCode("PR001").message("Forbidden").build()).build());
            }
        }
    }

    private boolean authorize(ContainerRequestContext requestContext, JsonWebToken webToken, String token) throws PrimerException {
        return PrimerAuthorizationRegistry.authorize(requestContext.getUriInfo().getPath(), (String)webToken.claim().getParameter("role"), requestContext.getMethod(), token, webToken);
    }

    private Optional<String> getToken(ContainerRequestContext requestContext) {
        String header = (String)requestContext.getHeaders().getFirst((Object)"Authorization");
        log.debug("Authorization Header: {}", (Object)header);
        if (header != null) {
            String rawToken = header.replaceAll(this.configuration.getPrefix(), "").trim();
            if (Strings.isNullOrEmpty((String)rawToken)) {
                return Optional.empty();
            }
            return Optional.of(rawToken);
        }
        return Optional.empty();
    }

    private JsonWebToken verifyToken(String rawToken) {
        JsonWebToken token = this.tokenParser.parse(rawToken);
        this.verifier.verifySignature(token);
        return token;
    }

    private void checkExpiry(JsonWebToken token) {
        if (token.claim() != null) {
            Instant now = new Instant();
            Instant issuedAt = (Instant)com.google.common.base.Optional.fromNullable((Object)this.toInstant(token.claim().issuedAt())).or((Object)now);
            Instant expiration = (Instant)com.google.common.base.Optional.fromNullable((Object)this.toInstant(token.claim().expiration())).or((Object)new Instant(Long.MAX_VALUE));
            Instant notBefore = (Instant)com.google.common.base.Optional.fromNullable((Object)this.toInstant(token.claim().notBefore())).or((Object)now);
            if (issuedAt.isAfter((ReadableInstant)expiration) || notBefore.isAfterNow() || !this.inInterval(issuedAt, expiration, now)) {
                throw new TokenExpiredException();
            }
        }
    }

    private boolean inInterval(Instant start, Instant end, Instant now) {
        Interval interval = new Interval((ReadableInstant)start, (ReadableInstant)end);
        Interval currentTimeWithSkew = new Interval((ReadableInstant)now.minus((ReadableDuration)this.acceptableClockSkew), (ReadableInstant)now.plus((ReadableDuration)this.acceptableClockSkew));
        return interval.overlaps((ReadableInterval)currentTimeWithSkew);
    }

    private Instant toInstant(Long input) {
        if (input == null) {
            return null;
        }
        return new Instant(input * 1000L);
    }

    private void stampHeaders(ContainerRequestContext requestContext, JsonWebToken webToken) {
        String tokenType;
        switch (tokenType = (String)webToken.claim().getParameter("type")) {
            case "dynamic": {
                requestContext.getHeaders().putSingle((Object)AUTHORIZED_FOR_ID, (Object)((String)webToken.claim().getParameter("user_id")));
                requestContext.getHeaders().putSingle((Object)AUTHORIZED_FOR_SUBJECT, (Object)webToken.claim().subject());
                requestContext.getHeaders().putSingle((Object)AUTHORIZED_FOR_NAME, (Object)((String)webToken.claim().getParameter("name")));
                break;
            }
            case "static": {
                requestContext.getHeaders().putSingle((Object)AUTHORIZED_FOR_SUBJECT, (Object)webToken.claim().subject());
                break;
            }
            default: {
                log.warn("No auth header stamped for type: {}", (Object)tokenType);
            }
        }
    }

    public static PrimerAuthenticatorRequestFilterBuilder builder() {
        return new PrimerAuthenticatorRequestFilterBuilder();
    }

    public static class PrimerAuthenticatorRequestFilterBuilder {
        private JsonWebTokenParser tokenParser;
        private HmacSHA512Verifier verifier;
        private PrimerBundleConfiguration configuration;

        PrimerAuthenticatorRequestFilterBuilder() {
        }

        public PrimerAuthenticatorRequestFilterBuilder tokenParser(JsonWebTokenParser tokenParser) {
            this.tokenParser = tokenParser;
            return this;
        }

        public PrimerAuthenticatorRequestFilterBuilder verifier(HmacSHA512Verifier verifier) {
            this.verifier = verifier;
            return this;
        }

        public PrimerAuthenticatorRequestFilterBuilder configuration(PrimerBundleConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public PrimerAuthenticatorRequestFilter build() {
            return new PrimerAuthenticatorRequestFilter(this.tokenParser, this.verifier, this.configuration);
        }

        public String toString() {
            return "PrimerAuthenticatorRequestFilter.PrimerAuthenticatorRequestFilterBuilder(tokenParser=" + this.tokenParser + ", verifier=" + this.verifier + ", configuration=" + this.configuration + ")";
        }
    }
}

