(ns io.dominic.wedge.integrant-aero
  "Wedge-compatible :system for integrant/aero.  Example system_config.edn:
    * {:system io.dominic.wedge.integrant-aero}
    * {:system io.dominic.wedge.integrant-aero
       :key :integrant/system}

  Uses aero to read `config.edn`, and gets the system from it.

  Requires `aero` and `integrant` to be on the classpath, and for dev also
  requires `integrant/repl`.

  Runs `ig/prep` and `ig/load-namespaces` as part of the start process.

  Adds #ig/ref and #ig/refset to Aero, which correspond to ig/ref and ig/refset
  respectively.

  See [[system-config]] for supported options."
  (:require
    [aero.core :as aero]
    [integrant.core :as ig]
    [io.dominic.wedge.impl.aero :as impl.aero]))

(defmethod aero/reader 'ig/ref [_ _ value]
  (ig/ref value))

(defmethod aero/reader 'ig/refset [_ _ value]
  (ig/refset value))

(defn system-config
  "Opts is a map with valid keys:
     :key - the key in config which contains the integrant system."
  [{:keys [key]
    :or {key :ig/system}
    :as opts}]
  (let [system-config (get (impl.aero/config opts) key)]
    (locking clojure.lang.RT/REQUIRE_LOCK
      (ig/load-namespaces system-config))
    (ig/prep system-config)))

(defn start
  [system-conf]
  ;; TODO: :keys support, take opts
  (ig/init system-conf))

(defn stop
  [system]
  (ig/halt! system))

(defn load-dev
  []
  (load "/io/dominic/wedge/impl/integrant_aero_dev"))
