(ns io.dominic.wedge.system
  "Functions to operate on Wedge's system infrastructure."
  (:require
    [clojure.java.io :as io]
    [clojure.edn :as edn]))

(defn read-system-config
  "Read system-config.edn, or fallback onto a default system config."
  []
  (if-let [r (io/resource "io/dominic/wedge/system-config.edn")]
    (edn/read (java.io.PushbackReader. (io/reader r)))
    {:system 'io.dominic.wedge.clip-aero
     :dev {:profile :dev}}))

(defn- call-system
  [nm & args]
  (let [config (read-system-config)
        sym (symbol (name (:system config))
                    (name nm))]
    (if-let [f (try (requiring-resolve sym)
                    (catch java.io.FileNotFoundException _
                      (throw (ex-info (str "Missing " sym)
                                      {::nm nm
                                       ::namespace (name (:system config))
                                       ::config config
                                       ::type ::missing-ns}))))]
      (apply f args)
      (throw (ex-info (str "Missing " sym)
                      {::nm nm
                       ::config config
                       ::type ::missing-var})))))

(defn get-config
  "Return a config, used to influence the start of the system."
  [opts]
  (call-system 'get-config opts))

(defn start
  "Take config, and start a system (possibly) using that config."
  [config opts]
  (call-system 'start config opts))

(defn stop
  "Take started system from [[start]] and stop it."
  [system]
  (call-system 'stop system))

(defn load-dev
  "Load dev functions into `*ns*`"
  []
  (call-system 'load-dev))

(defn load-system
  "Load namespaces needed by system"
  [config opts]
  (call-system 'load-system config opts))
