(ns io.cvcf.gentw
  (:require
   [babashka.fs :as fs]
   [babashka.process :refer [process destroy-tree]]
   [clojure.java.io :as io]
   [clojure.string :as str]))

(declare generate-css)
(defn build
  [& {:keys [in out] :or {in "input.css" out "output.css"}}]
  @(generate-css
    :in        in
    :out       out
    :watch?    false
    :minify?   false
    :optimize? false))

(defn release
  [& {:keys [in out] :or {in "input.css" out "output.css"}}]
  @(generate-css
    :in        in
    :out       out
    :watch?    false
    :minify?   true
    :optimize? true))

(defn watch
  [& {:keys [in out] :or {in "input.css" out "output.css"}}]
  (generate-css
   :in        in
   :out       out
   :watch?    true
   :minify?   false
   :optimize? false))

(defn build-command
  [& {:keys [infile outfile watch? minify? optimize?]}]
  (str/join
   " "
   (concat ["yarn" "tailwindcss"]
           ["--input" infile]
           ["--output" outfile]
           (when watch?    ["--watch"])
           (when minify?   ["--minify"])
           (when optimize? ["--optimize"]))))

(defn extract-sources
  [infile]
  (assert (fs/exists? infile))
  (map (fn [[_ match]] match)
       (re-seq #"@source \"(.*)\";" (slurp infile))))

(defn destination-path [source infile]
  (str
   (fs/normalize
    (if (fs/absolute? source)
      (io/file source)
      (io/file (str (fs/parent infile)) source)))))

(defn copy-source
  [src dst]
  (fs/create-dirs (fs/parent dst))
  (with-open [i (io/input-stream  src)
              o (io/output-stream dst)]
    (io/copy i o)))

(defn copy-sources
  [infile]
  (doseq [source (extract-sources infile)]
    (when-let [src (io/resource source)]
      (let [dst (destination-path source infile)]
        (io/make-parents dst)
        (if (fs/directory? src)
          (fs/copy-tree src dst)
          (copy-source src dst))
        dst))))

(defn generate-css
  "Generate CSS by calling the Tailwind CLI."
  [& {:keys [in out watch? minify? optimize?]
      :or   {in "input.css" out "output.css"}}]
  (let [infile  (.getAbsolutePath (io/file in))
        outfile (.getAbsolutePath (io/file out))
        output  (if watch? :inherit :string)]
    (process
     {:out output :err output :shutdown destroy-tree}
     (build-command
      :infile    infile
      :outfile   outfile
      :watch?    watch?
      :minify?   minify?
      :optimize? optimize?))))
