(ns io.cvcf.gentw
  (:require
   [babashka.fs :as fs]
   [babashka.process :refer [process destroy-tree]]
   [clojure.java.io :as io]
   [clojure.string :as str]))

(declare generate-css)
(defn build
  [& {:keys [in out] :or {in "input.css" out "output.css"}}]
  (-> (generate-css
       :in        in
       :out       out
       :watch?    false
       :minify?   false
       :optimize? false)
      deref
      :out
      println))

(defn release
  [& {:keys [in out] :or {in "input.css" out "output.css"}}]
  (-> (generate-css
       :in        in
       :out       out
       :watch?    false
       :minify?   true
       :optimize? true)
      deref
      :out
      println))

(defn watch
  [& {:keys [in out] :or {in "input.css" out "output.css"}}]
  (generate-css
   :in        in
   :out       out
   :watch?    true
   :minify?   false
   :optimize? false))

(defn build-command
  [& {:keys [infile outfile watch? minify? optimize?]}]
  (str/join
   " "
   (concat ["yarn" "tailwindcss"]
           ["--input" infile]
           ["--output" outfile]
           (when watch?    ["--watch"])
           (when minify?   ["--minify"])
           (when optimize? ["--optimize"]))))

(defn extract-sources
  [source]
  (map (fn [[_ match]] match)
       (re-seq #"@source \"(.*)\";" source)))

(defn copy-sources
  [infile]
  (doseq [source (extract-sources (slurp infile))]
    (when-let [src (io/resource source)]
      (let [dst (io/file source)]
        (io/make-parents dst)
        (if (fs/directory? src)
          (fs/copy-tree src dst)
          (with-open [i (io/input-stream  src)
                      o (io/output-stream dst)]
            (io/copy i o)))
        dst))))

(defn generate-css
  "Generate CSS by calling the Tailwind CLI."
  [& {:keys [in out watch? minify? optimize?]
      :or   {in "input.css" out "output.css"}}]
  (let [infile  (.getAbsolutePath (io/file in))
        outfile (.getAbsolutePath (io/file out))
        output  (if watch? :inherit :string)]
    (copy-sources infile)
    (process
     {:out output :err output :shutdown destroy-tree}
     (build-command
      :infile    infile
      :outfile   outfile
      :watch?    watch?
      :minify?   minify?
      :optimize? optimize?))))
