(ns atombind.scopes
  (:require [atombind.constants :as c]
            [atombind.logging :as l]))

(def ^:dynamic ^:private *scopes* (atom {}))

(def ^:dynamic ^:private *scope-id-ctr* (atom 0))

(def ^:dynamic ^:private *next-id* (fn []
                                     (str (swap! *scope-id-ctr* inc))))

(defprotocol IScopeable
  (-scopeable? [t] "Does the given item have the potential to be an IScopedItem?"))

(defprotocol IRegisteredScope
  (-register-scope! [t] "Registers the scope in the global scope register.")
  (-updated? [t] "Has the scope been updated since the last known state?")
  (-save-state [t] "Updates the last known state of the scope.")
  (-last-state [t] "The last known state of the scope."))

(defprotocol IScopedItem
  "A scoped item is one which already has an IRegisteredScope associated with it."
  (-init-item! [t provided-scope] "Initializes an item to become a scoped item. May mutate the item.")
  (-scoped? [t] "Has the item been scoped to a IRegisteredScope?")
  (^atombind.scopes.IRegisteredScope -get-scope [t] "Retrieves a scope from an item.")
  (^atombind.scopes.IRegisteredScope -get-scope! [t] "Retrieves a scope from an item. Initializes if required."))

(extend-type cljs.core/Atom
  IRegisteredScope
  (-register-scope! [t]
    (when-not (get-in @t [:$atombind :scope-id])
      (let [id (*next-id*)]
        (swap! t assoc :$atombind {:scope-id id})
        (swap! *scopes* assoc id {:atom-ref t
                                  :last-state @t}))))
  (-updated? [t]
    (let [state (-last-state t)]
      (not (identical? @t state))))
  (-save-state [t]
    (let [id (get-in @t [:$atombind :scope-id])]
      (swap! *scopes* assoc-in [id :last-state] @t)))
  (-last-state [t]
    (let [id (get-in @t [:$atombind :scope-id])]
      (get-in @*scopes* [id :last-state]))))

(extend-type js/HTMLElement
  IScopeable
  (-scopeable? [el]
    (let [attr (or (.getAttribute el c/bind-scope)
                   (.getAttribute el c/scope-id))]
      (not (nil? attr))))

  IScopedItem
  (-scoped? [el]
    (not (nil? (.getAttribute el c/scope-id))))
  (-init-item! [el provided-scope]
    (let [scope (or provided-scope
                    (let [scope-name (.getAttribute el c/bind-scope)]
                      (try
                        (js/eval scope-name)
                        (catch js/Error e
                          (l/err (str "Error: could not resolve scope: " scope-name))
                          nil))))]
      (when scope
        (-register-scope! scope)
        (let [scope-id (get-in @scope [:$atombind :scope-id])]
          (.setAttribute el c/scope-id scope-id)))))
  (-get-scope [el]
    (let [scope-id (.getAttribute el c/scope-id)]
      (get-in @*scopes* [scope-id :atom-ref])))
  (-get-scope! [el]
    (if (-scoped? el)
      (-get-scope el)
      (do
        (-init-item! el nil)
        (-get-scope el)))))
