(ns atombind.scope-test
  (:require [atombind.scopes :as s]
            [atombind.constants :as c]
            [goog.dom :as d]
            [atombind.testutil :refer [set-canvas]])
  (:require-macros [atombind.cljs-macros :refer [deftest]]))

(enable-console-print!)

(deftest test-registering-scopes
  "Registering a scope and checking for updates"
  (binding [s/*scopes* (atom {})]
    (let [scope (atom {})]
      (s/-register-scope! scope)
      (assert (not (s/-updated? scope)))
      (swap! scope assoc :foo "bar")
      (assert (s/-updated? scope))
      (s/-save-state scope)
      (assert (not (s/-updated? scope))))))

(deftest test-register-scope-more-than-once
  "Multiple registering of same scope should do nothing"
  (binding [s/*scopes* (atom {})
            s/*scope-id-ctr* (atom 0)]
    (let [scope (atom {})]
      (s/-register-scope! scope)
      (s/-register-scope! scope)
      (s/-register-scope! scope)
      (assert (= 1 (count @s/*scopes*)))
      (assert (= 1 @s/*scope-id-ctr*)))))

(deftest test-scopeability
  "Scopeability of DOM elements"
  (set-canvas
   "<div>
<div id='not-scoped'></div>
<div id='scoped-elem' bind-scope='fake-scope'></div>
<div id='another-scoped-item'></div>
</div>")
  (let [el (d/getElement "scoped-elem")
        other-el (d/getElement "not-scoped")
        scoped-2 (d/getElement "another-scoped-item")
        _ (.setAttribute scoped-2 c/scope-id "some-scope-id")]
    (assert (s/-scopeable? el))
    (assert (s/-scopeable? scoped-2))
    (assert (not (s/-scopeable? other-el)))))

(def ^:export the-scope (atom {}))

(deftest test-initializing-dom-elements-with-scope
  "Initialize DOM elements with scope"
  (set-canvas "<div id='scoped' bind-scope='atombind.scope_test.the_scope'></div>")
  (binding [s/*scopes* (atom {})]
    (let [el (d/getElement "scoped")]
      (assert (not (s/-scoped? el)))
      (s/-init-item! el nil)
      (assert (s/-scoped? el))
      (assert (not (empty? @s/*scopes*))))))

(deftest test-retrieving-scopes
  "Retrieving scope from an initialized element"
  (set-canvas "<div id='scoped' bind-scope='atombind.scope_test.the_scope'></div>")
  (binding [s/*scopes* (atom {})]
    (reset! the-scope {})
    (let [el (d/getElement "scoped")
          _ (s/-init-item! el nil)
          retrieved (s/-get-scope el)]
      (assert (identical? retrieved the-scope)))))

(deftest test-retrieving-scope-uninitialized
  "Retrieving scope from an UNinitialized element"
  (set-canvas "<div id='scoped' bind-scope='atombind.scope_test.the_scope'></div>")
  (binding [s/*scopes* (atom {})]
    (reset! the-scope {})
    (let [el (d/getElement "scoped")
          retrieved (s/-get-scope! el)]
      (assert (identical? retrieved the-scope)))))

(deftest test-initing-item-with-provided-atom
  "Initing a scoped item with provided IRegisteredScope"
  (set-canvas "<div id='foo'></div>")
  (binding [s/*scopes* (atom {})]
    (let [scope (atom {})
          div (d/getElement "foo")]
      (s/-register-scope! scope)
      (s/-init-item! div scope)
      (assert (s/-scoped? div)))))

(defn ^:export run-tests
  []
  (println "** SCOPE TESTS **")
  (run-all-tests))
