(ns slim-build-tools.core
  (:require
    [clojure.java.io :as io]
    [clojure.string :as str]
    [clojure.tools.build.api :as b])
  (:import
    (java.util.zip ZipFile)))

(defn compile-java [{:keys [src-dirs] :as opts}]
  (let [basis (b/create-basis opts)]
    (b/javac {:src-dirs  (or src-dirs (:paths basis))
              :class-dir "classes"
              :basis     basis})))

(defn- contains-class? [absolute-path]
  (with-open [zip-file (ZipFile. absolute-path)]
    (->> (iterator-seq (.entries zip-file))
         (remove #(.isDirectory %))
         (map #(.getName %))
         (filter #(str/ends-with? % ".class"))
         first)))

(defn- copy-dependencies [target-dir {:keys [libs] :as _basis}]
  (let [paths (mapcat (comp :paths val) libs)]
    (doseq [src paths]
      (let [target (str target-dir "/lib/" (.getName (io/file src)))]
        (b/copy-file {:src src :target target})))))

(defn- split-basis [{:keys [libs] :as basis}]
  (let [grouped (group-by
                  (fn [[_sym {:keys [paths]} :as _lib]]
                    (boolean (contains-class? (first paths))))
                  libs)]
    [(assoc basis :libs (into {} (get grouped true)))
     (assoc basis :libs (into {} (get grouped false)))]))

(defn jar
  ([] (jar {:main-ns "my-project.core"
            :jar     "my-project.jar"}))
  ([{:keys [main-ns target-dir src-dirs compile-opts jar]
     :or   {target-dir   "target"
            compile-opts {:direct-linking true}}
     :as   opts}]
   (b/delete {:path target-dir})
   (let [basis (b/create-basis opts)
         [precompiled-basis uncompiled-basis] (split-basis basis)
         src-dirs (or src-dirs (:paths basis))
         classes-dir (str target-dir "/classes")
         uber-jar-path (str target-dir "/" jar)]
     (println "Copying source...")
     (b/copy-dir {:src-dirs   src-dirs
                  :target-dir classes-dir})
     (println "Copying precompiled dependency jars...")
     (copy-dependencies target-dir precompiled-basis)
     (println "Compiling" main-ns "...")
     (b/compile-clj {:basis        uncompiled-basis
                     :class-dir    classes-dir
                     :src-dirs     src-dirs
                     :compile-opts compile-opts
                     :ns-compile   [main-ns]})
     (println "Building" uber-jar-path)
     (b/uber {:basis     uncompiled-basis
              :class-dir classes-dir
              :main      main-ns
              :manifest  {"Class-Path" (->> (.listFiles (io/file "target/lib"))
                                            (mapv #(str "lib/" (.getName %)))
                                            (str/join " "))}
              :uber-file uber-jar-path}))))

(comment
  (-> (b/create-basis)
      split-basis
      first))
